/* SmoothWall helper program - restartsquid
 *
 * This program is distributed under the terms of the GNU General Public
 * Licence.  See the file COPYING for details.
 *
 * (c) Lawrence Manning, 2001
 * Restarting squid with transparent proxying.
 *
 * 05/02/2004 - Roy Walker <rwalker@miracomnetwork.com>
 * Exclude red network from transparent proxy to allow browsing to alias IPs
 * Read in VPN settings and exclude each VPN network from transparent proxy
 *
 * $Id: restartsquid.c 3430 2009-08-13 18:12:05Z owes $
 *
 */


#include <fcntl.h>
#include <getopt.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "common.h"
#include "setuid.h"


void usage(char *prg, int exit_code)
{
    printf("Usage: %s [OPTION]\n\n", prg);
    printf("Options:\n");
    printf("  -f, --flush           flush proxy cache\n");
    printf("  -r, --repair          repair proxy cache\n");
    printf("  -t, --test            test first, do not start if not running\n");
    printf("  -v, --verbose         be verbose\n");
    printf("      --help            display this help and exit\n");
    exit(exit_code);
}


/* read the vpn config file and adds a rule for every net-to-net definition
    that skip the transparent rules REDIRECT
*/
void setdirectvpn(int setdirectvpn_green, int setdirectvpn_blue)
{
    int count;
    char *result;
    char *name;
    char *type;
    char *running;
    char *vpn_network_mask;
    char *vpn_netaddress;
    char *vpn_netmask;
    FILE *file = NULL;
    char *conn_enabled;
    char buffer[STRING_SIZE];
    char s[STRING_SIZE];

    if (!setdirectvpn_green && !setdirectvpn_blue)
        return;                 /* nothing to do */

    if (!(file = fopen("/var/ipcop/vpn/config", "r"))) {
        fprintf(stderr, "Couldn't open vpn config file");
        return;                 /* error! exit or return? */
    }
    /* WARNING: empty lines or comment not handled */
    while (fgets(s, STRING_SIZE, file) != NULL) {
        if (s[strlen(s) - 1] == '\n')
            s[strlen(s) - 1] = '\0';
        running = strdup(s);
        result = strsep(&running, ",");
        count = 0;
        name = NULL;
        type = NULL;
        vpn_network_mask = NULL;
        conn_enabled = NULL;
        while (result) {
            if (count == 1)
                conn_enabled = result;
            if (count == 2)
                name = result;
            if (count == 4)
                type = result;
            if (count == 12)
                vpn_network_mask = result;
            count++;
            result = strsep(&running, ",");
        }

        if (strspn(name, LETTERS_NUMBERS) != strlen(name)) {
            fprintf(stderr, "Bad connection name: %s\n", name);
            continue;
        }

        if (!(strcmp(type, "net") == 0)) {
            continue;
        }

        /* Darren Critchley - new check to see if connection is enabled */
        if (!(strcmp(conn_enabled, "on") == 0)) {
            continue;
        }

        result = strsep(&vpn_network_mask, "/");
        count = 0;
        vpn_netaddress = NULL;
        vpn_netmask = NULL;
        while (result) {
            if (count == 0)
                vpn_netaddress = result;
            if (count == 1)
                vpn_netmask = result;
            count++;
            result = strsep(&vpn_network_mask, "/");
        }

        if (!VALID_IP(vpn_netaddress)) {
            fprintf(stderr, "Bad network for vpn connection %s: %s\n", name, vpn_netaddress);
            continue;
        }

        if ((!VALID_IP(vpn_netmask)) && (!VALID_SHORT_MASK(vpn_netmask))) {
            fprintf(stderr, "Bad mask for vpn connection %s: %s\n", name, vpn_netmask);
            continue;
        }

        memset(buffer, 0, STRING_SIZE);
        if (setdirectvpn_green) {
            if (snprintf(buffer, STRING_SIZE - 1,
                         "/sbin/iptables -t nat -A SQUID -i %s -p tcp -d %s/%s --dport 80 -j RETURN",
                         ipcop_ethernet.device[GREEN][1], vpn_netaddress, vpn_netmask) >= STRING_SIZE) {
                fprintf(stderr, "Command too long\n");
                fclose(file);
                exit(1);
            }
            safe_system(buffer);
        }
        if (setdirectvpn_blue) {
            if (snprintf(buffer,
                         STRING_SIZE - 1,
                         "/sbin/iptables -t nat -A SQUID -i %s -p tcp -d %s/%s --dport 80 -j RETURN",
                         ipcop_ethernet.device[BLUE][1], vpn_netaddress, vpn_netmask) >= STRING_SIZE) {
                fprintf(stderr, "Command too long\n");
                fclose(file);
                exit(1);
            }
            safe_system(buffer);
        }
    }
    fclose(file);
}


int main(int argc, char **argv)
{
    int flag_test = 0;
    int flag_flush = 0;
    int flag_repair = 0;
    int enabled_green = 0;
    int transparent_green = 0;
    int enabled_blue = 0;
    int transparent_blue = 0;
    struct stat st;
    NODEKV *squid_kv = NULL;
    char buffer[STRING_SIZE];
    char proxy_port[STRING_SIZE];
/* TODO: fix for both IPsec and OpenVPN */
//    NODEKV *vpn_kv = NULL;
//    char enableredvpn[STRING_SIZE] = "";
//    char enablebluevpn[STRING_SIZE] = "";

    static struct option long_options[] =
    {
        { "flush", no_argument, 0, 'f' },
        { "repair", no_argument, 0, 'r' },
        { "test", no_argument, 0, 't' },
        { "verbose", no_argument, 0, 'v' },
        { "help", no_argument, 0, 'h' },
        { 0, 0, 0, 0}
    };
    int c;
    int option_index = 0;

    if (!(initsetuid()))
        exit(1);

    while ((c = getopt_long(argc, argv, "frtv", long_options, &option_index)) != -1) {
        switch (c) {
        case 't':              /* test first */
            flag_test = 1;
            break;
        case 'f':              /* flush cache */
            flag_flush = 1;
            break;
        case 'r':              /* repair cache */
            flag_repair = 1;
            break;
        case 'v':              /* verbose */
            flag_verbose++;
            break;
        case 'h':
            usage(argv[0], 0);
        default:
            fprintf(stderr, "unknown option\n");
            usage(argv[0], 1);
        }
    }


    /* Retrieve the Squid pid file */
    if ((access("/var/run/squid.pid", F_OK) == -1) && flag_test) {
        verbose_printf(1, "Squid not running, no need to start\n");
        exit(0);                /* Not running, no need to start with -t */
    }

    if (access("/var/run/squid.pid", F_OK) != -1) {
        /* Kill running squid */
        verbose_printf(1, "Flush squid iptables chain ... \n");
        safe_system("/sbin/iptables -t nat -F SQUID");
        verbose_printf(1, "Shutdown squid ... \n");
        safe_system("/usr/sbin/squid -k shutdown >/dev/null 2>/dev/null");
        sleep(5);
        verbose_printf(1, "Really shutdown squid ... \n");
        safe_system("/bin/killall -9 squid >/dev/null 2>/dev/null");
    }
    if (access("/var/run/squid.pid", F_OK) != -1) {
        verbose_printf(2, "Remove leftover PID file ... \n");
        unlink("/var/run/squid.pid");
    }

    /* See if we need to flush/repair the cache */
    if (flag_flush) {
        struct passwd *pw;
        if ((pw = getpwnam("squid"))) {
            endpwent();         /* probably paranoia, but just in case.. */
            verbose_printf(1, "Flushing proxy cache ... \n");
            unpriv_system("/bin/rm -rf /var/log/cache/*", pw->pw_uid, pw->pw_gid);
        }
        else {
            fprintf(stderr, "User squid not found, cache not flushed\n");
            endpwent();
        }
    }

    int saferestart = 0;
    if (flag_repair) {
        struct passwd *pw;
        if ((pw = getpwnam("squid"))) {
            endpwent();         /* probably paranoia, but just in case.. */
            verbose_printf(1, "Repairing proxy cache ... \n");
            if (stat("/var/log/cache/swap.state", &st) == 0) {
                unpriv_system("/bin/rm -f /var/log/cache/swap.state", pw->pw_uid, pw->pw_gid);
            }
            saferestart = 1;
        }
        else {
            fprintf(stderr, "User squid not found, cache not repaired\n");
            endpwent();
        }
    }

    verbose_printf(1, "Reading Proxy settings ... \n");
    if (read_kv_from_file(&squid_kv, "/var/ipcop/proxy/settings") != SUCCESS) {
        fprintf(stderr, "Cannot read proxy settings\n");
        exit(1);
    }

    /* See if proxy is enabled and / or transparent */
    if (test_kv(squid_kv, "ENABLED_GREEN_1", "on") == SUCCESS) {
        enabled_green = 1;
    }
    if (test_kv(squid_kv, "TRANSPARENT_GREEN_1", "on") == SUCCESS) {
        transparent_green = 1;
    }
    if (test_kv(squid_kv, "ENABLED_BLUE_1", "on") == SUCCESS) {
        enabled_blue = 1;
    }
    if (test_kv(squid_kv, "TRANSPARENT_BLUE_1", "on") == SUCCESS) {
        transparent_blue = 1;
    }

    /* Retrieve the proxy port */
    if (transparent_green || transparent_blue) {
        if (find_kv_default(squid_kv, "PROXY_PORT", proxy_port) != SUCCESS) {
            strcpy(proxy_port, "8080");
        }
        else {
            if (strspn(proxy_port, NUMBERS) != strlen(proxy_port)) {
                fprintf(stderr, "Invalid proxy port: %s, defaulting to 8080\n", proxy_port);
                strcpy(proxy_port, "8080");
            }
        }
    }
    free_kv(&squid_kv);

    if (!enabled_green && !enabled_blue) {
        verbose_printf(1, "Proxy not enabled ... exit ... \n");
        return 0;
    }

    /* Fetch ethernet/settings, exit on error */
    read_ethernet_settings(1);


    verbose_printf(1, "Reading VPN settings ... \n");
/* TODO: fix for both IPsec and OpenVPN */
/*
	if (read_kv_from_file(&vpn_kv, "/var/ipcop/vpn/settings") != SUCCESS) {
		fprintf(stderr, "Cannot read vpn settings\n");
		exit(1);
	}
	find_kv_default(vpn_kv, "ENABLED", enableredvpn);
	find_kv_default(vpn_kv, "ENABLED_BLUE_1", enablebluevpn);
	free_kv(&vpn_kv);
*/

    if (enabled_green || enabled_blue) {
        /* rebuild firewall rules, proxy port may be different now */
        verbose_printf(1, "Rebuild firewall rules ... \n");
        safe_system("/usr/local/bin/setfwrules --ipcop");

        verbose_printf(1, "Starting squid ... \n");
        safe_system("/usr/sbin/squid -D -z");
        if (saferestart)
            safe_system("/usr/sbin/squid -DS");
        else
            safe_system("/usr/sbin/squid -D");
    }

    /* static (green/blue) interfaces must exist if transparence is requested */
    if (transparent_green && enabled_green && !ipcop_ethernet.count[GREEN]) {
        fprintf(stderr, "No GREEN device, not running transparent\n");
        exit(1);
    }

    if (transparent_blue && enabled_blue && !ipcop_ethernet.count[BLUE]) {
        fprintf(stderr, "No BLUE device, not running transparent\n");
        exit(1);
    }

    /* disable transparence for known vpn networks */
/* TODO: fix for both IPsec and OpenVPN */
/*
	setdirectvpn (enabled_green && transparent_green && !strcmp(enableredvpn, "on"),
			enabled_blue && transparent_blue && !strcmp(enablebluevpn, "on") );
*/

    /* choose RED destination: 'localip' or 'red_netaddress/red_netmask' */
    char destination[STRING_SIZE] = "";
    if (strcmp(ipcop_ethernet.red_type[1], "STATIC") == 0) {
        snprintf(destination, STRING_SIZE, "%s/%s", ipcop_ethernet.address[RED][1], ipcop_ethernet.netmask[RED][1]);
    }
    else {
        if (ipcop_ethernet.red_address[1][0] && VALID_IP(ipcop_ethernet.red_address[1])) {
            snprintf(destination, STRING_SIZE, "%s", ipcop_ethernet.red_address[1]);
        }
    }

    /* RED may be down */
    if (!strlen(destination)) {
        fprintf(stderr, "Cannot determine RED network.\n");
    }
    else {
        verbose_printf(2, "Dest IP is set to: %s\n", destination);
    }

    /* install the transparency rules */
    /* green transparent ? */
    if (transparent_green && enabled_green) {
        /* direct http GREEN-->RED network */
        verbose_printf(1, "Setting transparent iptables rule for GREEN ... \n");
        if (snprintf(buffer, STRING_SIZE - 1,
                     "/sbin/iptables -t nat -A SQUID -i %s -p tcp -d %s --dport 80 -j RETURN",
                     ipcop_ethernet.device[GREEN][1], destination) >= STRING_SIZE) {
            fprintf(stderr, "Command too long\n");
            exit(1);
        }
        if (strlen(destination))
            safe_system(buffer);        /* only id known RED */

        /* install the redirect for other port http destinations from green */
        if (snprintf(buffer, STRING_SIZE - 1,
                     "/sbin/iptables -t nat -A SQUID -i %s -p tcp --dport 80 -j REDIRECT --to-port %s",
                     ipcop_ethernet.device[GREEN][1], proxy_port) >= STRING_SIZE) {
            fprintf(stderr, "Command too long\n");
            exit(1);
        }
        safe_system(buffer);
    }
    /* blue transparent ? */
    if (transparent_blue && enabled_blue) {
        /* direct http BLUE-->RED network */
        verbose_printf(1, "Setting transparent iptables rule for BLUE ... \n");
        if (snprintf(buffer, STRING_SIZE - 1,
                     "/sbin/iptables -t nat -A SQUID -i %s -p tcp -d %s --dport 80 -j RETURN",
                     ipcop_ethernet.device[BLUE][1], destination) >= STRING_SIZE) {
            fprintf(stderr, "Command too long\n");
            exit(1);
        }
        if (strlen(destination))
            safe_system(buffer);        /* only id known RED */

        /* install the redirect for other port http destinations from blue */
        if (snprintf(buffer, STRING_SIZE - 1,
                     "/sbin/iptables -t nat -A SQUID -i %s -p tcp --dport 80 -j REDIRECT --to-port %s",
                     ipcop_ethernet.device[BLUE][1], proxy_port) >= STRING_SIZE) {
            fprintf(stderr, "Command too long\n");
            exit(1);
        }
        safe_system(buffer);
    }
    return 0;
}
