#!/usr/bin/perl
#
# This file is part of the IPCop Firewall.
#
# IPCop is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# IPCop is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with IPCop; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
#
# (c) 2003 Alan Hourihane <alanh@fairlite.demon.co.uk>
# (c) 2005 Eric Oberlander, Robert Kerr - Inline editing & DHCP leases
#
# $Id: wireless.cgi 3323 2009-07-25 14:15:28Z owes $
#

# Add entry in menu
# MENUENTRY firewall 020 "blue access" "blue access" haveBlue
#
# Make sure translation exists $Lang::tr{'blue access'}

use strict;
use Time::Local;

# enable only the following on debugging purpose
use warnings; no warnings 'once';# 'redefine', 'uninitialized';
use CGI::Carp 'fatalsToBrowser';

require '/usr/lib/ipcop/general-functions.pl';
require '/usr/lib/ipcop/lang.pl';
require '/usr/lib/ipcop/header.pl';

my %cgiparams    = ();
my %checked      = ();
my $errormessage = '';
my $filename     = '/var/ipcop/firewall/wireless';
our %dhcpsettings = ();
our %netsettings  = ();

my $counter;

$cgiparams{'ENABLED'}    = 'on';
$cgiparams{'ACTION'}     = '';
$cgiparams{'VALID'}      = '';
$cgiparams{'SOURCE_IP'}  = '';
$cgiparams{'SOURCE_MAC'} = '';
$cgiparams{'REMARK'}     = '';

&General::getcgihash(\%cgiparams);

&General::readhash('/var/ipcop/dhcp/settings',     \%dhcpsettings);
&General::readhash('/var/ipcop/ethernet/settings', \%netsettings);

&Header::showhttpheaders();

open(FILE, $filename) or die 'Unable to open config file.';
my @current = <FILE>;
close(FILE);

if ($cgiparams{'ACTION'} eq 'add') {

    # Verify the input data

    # An IP address or a MAC address, or both, are required.
    if ($cgiparams{'SOURCE_IP'} eq '' && $cgiparams{'SOURCE_MAC'} eq '') {
        $errormessage = $Lang::tr{'invalid ip'} . " " . $Lang::tr{'or'} . " " . $Lang::tr{'invalid mac'};
        goto ADDERROR;
    }

    $cgiparams{'SOURCE_MAC'} =~ tr/-/:/;

    my $key = 0;
    foreach my $line (@current) {
        $key++;
        my @temp = split(/\,/, $line);

        if ($temp[1] ne '' && $cgiparams{'SOURCE_IP'} eq $temp[1] && $cgiparams{'EDITING'} ne $key) {
            $errormessage = $Lang::tr{'duplicate ip'};
            goto ADDERROR;
        }
        if ($temp[2] ne '' && lc($cgiparams{'SOURCE_MAC'}) eq lc($temp[2]) && $cgiparams{'EDITING'} ne $key) {
            $errormessage = $Lang::tr{'duplicate mac'};
            goto ADDERROR;
        }
    }

    if ($cgiparams{'SOURCE_IP'} eq '') {
        $cgiparams{'SOURCE_IP'} = 'NONE';
    }
    else {
        unless (&General::validip($cgiparams{'SOURCE_IP'})) {
            $errormessage = $Lang::tr{'invalid ip'};
            goto ADDERROR;
        }
        else {
            my $insubnet = 0;

            # IP must be in a blue subnet
            for ($counter = 1; $counter <= $netsettings{'BLUE_COUNT'}; $counter++) {
                if (&General::IpInSubnet($cgiparams{'SOURCE_IP'},
                        $netsettings{"BLUE_${counter}_NETADDRESS"},
                        $netsettings{"BLUE_${counter}_NETMASK"})) {
                    $insubnet++;
                }
            }
            if ($insubnet == 0) {
                $errormessage = $Lang::tr{'invalid ip'};
                goto ADDERROR;
            }
        }
    }

    if ($cgiparams{'SOURCE_MAC'} eq '') {
        $cgiparams{'SOURCE_MAC'} = 'NONE';
    }
    else {
        unless (&General::validmac($cgiparams{'SOURCE_MAC'})) {
            $errormessage = $Lang::tr{'invalid mac'};
        }
    }

ADDERROR:
    if ($errormessage) {
        $cgiparams{'SOURCE_MAC'} = '' if $cgiparams{'SOURCE_MAC'} eq 'NONE';
        $cgiparams{'SOURCE_IP'}  = '' if $cgiparams{'SOURCE_IP'}  eq 'NONE';
    }
    else {
        $cgiparams{'REMARK'} = &Header::cleanhtml($cgiparams{'REMARK'});
        if ($cgiparams{'EDITING'} eq 'no') {
            open(FILE, ">>$filename") or die 'Unable to open config file.';
            flock FILE, 2;
            print FILE
                "$key,$cgiparams{'SOURCE_IP'},$cgiparams{'SOURCE_MAC'},$cgiparams{'ENABLED'},$cgiparams{'REMARK'}\n";
        }
        else {
            open(FILE, ">$filename") or die 'Unable to open config file.';
            flock FILE, 2;
            my $id = 0;
            foreach my $line (@current) {
                $id++;
                if ($cgiparams{'EDITING'} eq $id) {
                    print FILE
"$id,$cgiparams{'SOURCE_IP'},$cgiparams{'SOURCE_MAC'},$cgiparams{'ENABLED'},$cgiparams{'REMARK'}\n";
                }
                else { print FILE "$line"; }
            }
        }
        close(FILE);
        # Restore defaults for next entry (if any)
        undef %cgiparams;
        $cgiparams{'ENABLED'} = 'on';
        &General::log($Lang::tr{'wireless config added'});
        `/usr/local/bin/setfwrules --wireless < /dev/null > /dev/null 2>&1 &`;

        # sort wireless device list by IP address
        system "/usr/bin/sort -n -t '.' -k 2,2 -k 3,3 -k 4,4 $filename -o $filename";
    }
ADDEXIT:
}

if ($cgiparams{'ACTION'} eq 'edit') {
    my $id = 0;
    foreach my $line (@current) {
        $id++;
        if ($cgiparams{'ID'} eq $id) {
            chomp($line);
            my @temp = split(/\,/, $line);
            $cgiparams{'SOURCE_IP'}  = $temp[1];
            $cgiparams{'SOURCE_MAC'} = $temp[2];
            $cgiparams{'ENABLED'}    = $temp[3];
            $cgiparams{'REMARK'}     = $temp[4];
            $cgiparams{'SOURCE_IP'}  = '' if $cgiparams{'SOURCE_IP'} eq 'NONE';
            $cgiparams{'SOURCE_MAC'} = '' if $cgiparams{'SOURCE_MAC'} eq 'NONE';
        }
    }
    &General::log($Lang::tr{'wireless config changed'});
    `/usr/local/bin/setfwrules --wireless < /dev/null > /dev/null 2>&1 &`;
}

if ($cgiparams{'ACTION'} eq 'remove' || $cgiparams{'ACTION'} eq 'toggle') {
    my $id = 0;
    open(FILE, ">$filename") or die 'Unable to open config file.';
    flock FILE, 2;
    foreach my $line (@current) {
        $id++;
        unless ($cgiparams{'ID'} eq $id) { print FILE "$line"; }
        elsif ($cgiparams{'ACTION'} eq 'toggle') {
            chomp($line);
            my @temp = split(/\,/, $line);
            print FILE "$temp[0],$temp[1],$temp[2],$cgiparams{'ENABLE'},$temp[4]\n";
        }
    }
    close(FILE);
    &General::log($Lang::tr{'wireless config changed'});
    `/usr/local/bin/setfwrules --wireless < /dev/null > /dev/null 2>&1 &`;
}

$checked{'ENABLED'}{'off'}                 = '';
$checked{'ENABLED'}{'on'}                  = '';
$checked{'ENABLED'}{$cgiparams{'ENABLED'}} = "checked='checked'";

&Header::openpage($Lang::tr{'wireless configuration'}, 1, '');

&Header::openbigbox('100%', 'left', '');

if ($errormessage) {
    &Header::openbox('100%', 'left', "$Lang::tr{'error messages'}:", 'error');
    print "<class name='base'>$errormessage\n";
    print "&nbsp;</class>\n";
    &Header::closebox();
}

print "<form method='post' action='$ENV{'SCRIPT_NAME'}'>\n";

my $buttontext = $Lang::tr{'add'};
if ($cgiparams{'ACTION'} eq 'edit') {
    &Header::openbox('100%', 'left', "$Lang::tr{'edit device'}");
    $buttontext = $Lang::tr{'update'};
}
else {
    &Header::openbox('100%', 'left', "$Lang::tr{'add device'}");
}

print <<END
<table width='100%'>
<tr>
<td width='25%' class='base'>$Lang::tr{'ip address'}:&nbsp;<img src='/blob.gif' alt='*' /></td>
<td width='25%'><input type='text' name='SOURCE_IP' value='$cgiparams{'SOURCE_IP'}' size='20' /></td>
<td width='25%' align='right' class='base'>$Lang::tr{'mac address'}:&nbsp;<img src='/blob.gif' alt='*' />&nbsp;</td>
<td width='25%'><input type='text' name='SOURCE_MAC' value='$cgiparams{'SOURCE_MAC'}' size='20' /></td>
</tr>
<tr>
<td width='25%' class='base'>$Lang::tr{'remark'}:&nbsp;<img src='/blob.gif' alt='*' /></td>
<td colspan='3'><input type='text' name='REMARK' value='$cgiparams{'REMARK'}' size='40' /></td>
</tr>
<tr>
<td width='25%' class='base'>$Lang::tr{'enabled'}:&nbsp;</td>
<td colspan='3'><input type='checkbox' name='ENABLED' $checked{'ENABLED'}{'on'} /></td>
</tr>
</table>
<hr />
<table width='100%'>
<tr>
    <td class='comment1button'><strong>$Lang::tr{'note'}:</strong>&nbsp;$Lang::tr{'blue access use hint'}</td>
    <td colspan='2'>&nbsp;</td>
</tr>
<tr>
    <td class='comment1button'>
      <img src='/blob.gif' alt='*' />
      $Lang::tr{'this field may be blank'}</td>
    <td class='button1button'>
      <input type='hidden' name='ACTION' value='add' />
      <input type='submit' name='SUBMIT' value='$buttontext' />
    </td>
    <td class='onlinehelp'>
    <a href='${General::adminmanualurl}/firewall-blue-access.html' target='_blank'>
    <img src='/images/web-support.png' alt='$Lang::tr{'online help en'}' title='$Lang::tr{'online help en'}' /></a></td>
</tr>
</table>
END
    ;

if ($cgiparams{'ACTION'} eq 'edit') {
    print "<input type='hidden' name='EDITING' value='$cgiparams{'ID'}' />\n";
}
else {
    print "<input type='hidden' name='EDITING' value='no' />\n";
}

&Header::closebox();

print "</form>\n";

&Header::openbox('100%', 'left', "$Lang::tr{'devices on blue'}");
print <<END
<div align='center'>
END
    ;
open(FILE, "$filename");
@current = <FILE>;
close(FILE);

print <<END
<table width='100%'>
<tr>
<td align='center' width='20%'><b>$Lang::tr{'ip address'}</b> $Header::sortup</td>
<td align='center' width='20%'><b>$Lang::tr{'mac address'}</b></td>
<td align='center' width='55%'><b>$Lang::tr{'remark'}</b></td>
<td align='center' colspan='3'><b>$Lang::tr{'action'}</b></td>
</tr>
END
    ;

my $id = 0;

foreach my $line (@current) {
    my $tid = ($id % 2) + 1;
    $id++;
    chomp($line);
    my $gif        = "";
    my $gdesc      = "";
    my $toggle     = "";
    my @temp       = split(/\,/, $line);
    my $wirelessid = $temp[0];
    my $sourceip   = $temp[1];
    my $sourcemac  = $temp[2];

    if   ($temp[3] eq 'on') { $gif = 'on.gif';  $toggle = 'off'; $gdesc = $Lang::tr{'click to disable'}; }
    else                    { $gif = 'off.gif'; $toggle = 'on';  $gdesc = $Lang::tr{'click to enable'}; }

    my $remark = $temp[4];

    if ($cgiparams{'ACTION'} eq 'edit' && $cgiparams{'ID'} eq $id) {
        print "<tr class='selectcolour'>\n";
    }
    else {
        print "<tr class='table${tid}colour'>\n";
    }
    print "<td align='center'>$sourceip</td>\n";
    print "<td align='center'>$sourcemac</td>\n";
    print "<td align='center'>$remark</td>\n";
    print <<END
<td align='center'>
	<form method='post' name='frma$id' action='$ENV{'SCRIPT_NAME'}'>
	<input type='image' name='$Lang::tr{'toggle enable disable'}' src='/images/$gif' alt='$gdesc' title='$gdesc' />
	<input type='hidden' name='ACTION' value='toggle' />
	<input type='hidden' name='ID' value='$id' />
	<input type='hidden' name='ENABLE' value='$toggle' />
	</form>
</td>

<td align='center'>
	<form method='post' name='frmb$id' action='$ENV{'SCRIPT_NAME'}'>
	<input type='hidden' name='ACTION' value='edit' />
	<input type='image' name='$Lang::tr{'edit'}' src='/images/edit.gif' alt='$Lang::tr{'edit'}' title='$Lang::tr{'edit'}' />
	<input type='hidden' name='ID' value='$id' />
	</form>
</td>

<td align='center'>
	<form method='post' name='frmc$id' action='$ENV{'SCRIPT_NAME'}'>
	<input type='hidden' name='ACTION' value='remove' />
	<input type='image' name='$Lang::tr{'remove'}' src='/images/delete.gif' alt='$Lang::tr{'remove'}' title='$Lang::tr{'remove'}' />
	<input type='hidden' name='ID' value='$id' />
	</form>
</td>
END
        ;
    print "</tr>\n";
}
print "</table>\n";

print "</div>\n";

&Header::closebox();

my $haveBlueDHCP = 0;
for ($counter = 1; $counter <= $netsettings{'BLUE_COUNT'}; $counter++) {
    if ($dhcpsettings{"ENABLED_BLUE_${counter}"} eq 'on') {
        $haveBlueDHCP++;
    }
}

if ($haveBlueDHCP) {
    &printblueleases;
}

&Header::closebigbox();

&Header::closepage();

sub printblueleases {
    our %entries = ();

    sub blueleasesort {

        # Sort by IP address
        my $qs = 'IPADDR';
        my @a  = split(/\./, $entries{$a}->{$qs});
        my @b  = split(/\./, $entries{$b}->{$qs});
               ($a[0] <=> $b[0])
            || ($a[1] <=> $b[1])
            || ($a[2] <=> $b[2])
            || ($a[3] <=> $b[3]);
    }

    &Header::openbox('100%', 'left', "$Lang::tr{'current dhcp leases on blue'}");
    print <<END
<table width='100%'>
<tr>
<td width='25%' align='center'><b>$Lang::tr{'ip address'}</b> $Header::sortup</td>
<td width='25%' align='center'><b>$Lang::tr{'mac address'}</b></td>
<td width='20%' align='center'><b>$Lang::tr{'hostname'}</b></td>
<td width='30%' align='center'><b>$Lang::tr{'lease expires'} (local time d/m/y)</b></td>
</tr>
END
        ;

    my ($ip, $endtime, $ether, $hostname, @record, $record);
    open(LEASES, "/var/run/dnsmasq/dnsmasq.leases") or die "Can't open dhcpd.leases";
    while (my $line = <LEASES>) {
        next if ($line =~ /^\s*#/);
        chomp($line);
        my @temp = split(' ', $line);

        $endtime  = $temp[0];
        $ip       = $temp[2];
        $ether    = $temp[1];
        $hostname = $temp[3];

        # Select records in Blue subnet
        # TODO: prepare for multiple blues
        if (&General::IpInSubnet($ip, $netsettings{"BLUE_1_NETADDRESS"}, $netsettings{"BLUE_1_NETMASK"})) {
            @record = ('IPADDR', $ip, 'ENDTIME', $endtime, 'ETHER', $ether, 'HOSTNAME', $hostname);
            $record = {};    # create a reference to empty hash
            %{$record} = @record;    # populate that hash with @record
            $entries{$record->{'IPADDR'}} = $record;    # add this to a hash of hashes
        }
    }
    close(LEASES);

    my $id = 0;
    foreach my $key (sort blueleasesort keys %entries) {

        my $hostname = &Header::cleanhtml($entries{$key}->{HOSTNAME}, "y");
        my $tid = ($id % 2) + 1;

        print <<END
<tr class='table${tid}colour'>
<td align='center'>$entries{$key}->{IPADDR}</td>
<td align='center'>$entries{$key}->{ETHER}</td>
<td align='center'>&nbsp;$hostname </td>
<td align='center'>
END
            ;

        if ($entries{$key}->{ENDTIME} eq 'never') {
            print "$Lang::tr{'no time limit'}";
        }
        else {
            my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $dst);
            ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $dst) = localtime($entries{$key}->{ENDTIME});
            my $enddate = sprintf("%02d/%02d/%d %02d:%02d:%02d", $mday, $mon + 1, $year + 1900, $hour, $min, $sec);

            if ($entries{$key}->{ENDTIME} < time()) {
                print "<strike>$enddate</strike>";
            }
            else {
                print "$enddate";
            }
        }

        if ($hostname eq '') {
            $hostname = $Lang::tr{'device'};
        }

        # check if MAC address is already in list of devices
        my $macinlist = 0;
        foreach my $line (@current) {
            my @temp = split(/\,/, $line);

            if (lc($temp[2]) eq lc($entries{$key}->{ETHER})) {

                # set flag to disable add device button
                $macinlist = 1;
                last;
            }
        }

        if ($macinlist < 1) {
            print <<END
</td>
<td align='center'>
	<form method='post' name='frmd$id' action='$ENV{'SCRIPT_NAME'}'>
	<input type='hidden' name='ACTION' value='add' />
	<input type='hidden' name='SOURCE_IP' value='' />
	<input type='hidden' name='SOURCE_MAC' value='$entries{$key}->{ETHER}' />
	<input type='hidden' name='REMARK' value='$hostname $Lang::tr{'added from dhcp lease list'}' />
	<input type='hidden' name='ENABLED' value='on' />
	<input type='hidden' name='EDITING' value='no' />
	<input type='image' name='$Lang::tr{'add device'}' src='/images/addblue.gif' alt='$Lang::tr{'add device'}' title='$Lang::tr{'add device'}' />
	</form>
</td></tr>
END
                ;
        }
        else {
            print <<END
</td>
<td>
	<img src='/images/addfaint.gif' alt='' width='20' />
</td></tr>
END
                ;
        }
        $id++;
    }

    print "</table>";
    &Header::closebox();
}

