/* This file is part of the IPCop Firewall.
 *
 * IPCop is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * IPCop is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with IPCop; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * Copyright (C) 2003-07-12 Robert Kerr <rkerr@go.to>
 *
 * $Id: restartsyslogd.c 1774 2008-09-01 17:42:36Z owes $
 *
 */


#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "common.h"
#include "setuid.h"


#define ERR_ANY 1
#define ERR_SETTINGS 2          /* error in settings file */
#define ERR_ETC 3               /* error with /etc permissions */
#define ERR_CONFIG 4            /* error updated sshd_config */
#define ERR_SYSLOG 5            /* error restarting syslogd */


void usage(char *prg, int exit_code)
{
    printf("Usage: %s [OPTION]\n\n", prg);
    printf("Options:\n");
    printf("  -v, --verbose         be verbose\n");
    printf("      --help            display this help and exit\n");
    exit(exit_code);
}


int main(int argc, char **argv)
{
    char buffer[STRING_SIZE], hostname[STRING_SIZE];
    int config_fd, rc, fd, pid;
    struct stat st;
    NODEKV *log_kv = NULL;

    static struct option long_options[] =
    {
        { "boot", no_argument, 0, 'b' },
        { "verbose", no_argument, 0, 'v' },
        { "help", no_argument, 0, 'h' },
        { 0, 0, 0, 0}
    };
    int c;
    int option_index = 0;

    memset(buffer, 0, STRING_SIZE);
    memset(hostname, 0, STRING_SIZE);

    if (!(initsetuid()))
        exit(1);

    while ((c = getopt_long(argc, argv, "v", long_options, &option_index)) != -1) {
        switch (c) {
        case 'v':              /* verbose */
            flag_verbose++;
            break;
        case 'h':
            usage(argv[0], 0);
        default:
            fprintf(stderr, "unknown option\n");
            usage(argv[0], 1);
        }
    }

    verbose_printf(1, "Reading log settings ... \n");
    if (read_kv_from_file(&log_kv, "/var/ipcop/logging/settings") != SUCCESS) {
        fprintf(stderr, "Cannot read syslog settings\n");
        exit(ERR_SETTINGS);
    }

    if (find_kv_default(log_kv, "ENABLE_REMOTELOG", buffer) != SUCCESS) {
        fprintf(stderr, "Cannot read ENABLE_REMOTELOG\n");
        exit(ERR_SETTINGS);
    }

    if (find_kv_default(log_kv, "REMOTELOG_ADDR", hostname) != SUCCESS) {
        fprintf(stderr, "Cannot read REMOTELOG_ADDR\n");
        exit(ERR_SETTINGS);
    }

    if (strspn(hostname, VALID_FQDN) != strlen(hostname)) {
        fprintf(stderr, "Bad REMOTELOG_ADDR: %s\n", hostname);
        exit(ERR_SETTINGS);
    }

    free_kv(&log_kv);


    /* If anyone other than root can write to /etc this would be totally
     * insecure - same if anyone other than root owns /etc, as they could
     * change the file mode to give themselves or anyone else write access. */

    verbose_printf(1, "Verify /etc ... \n");
    if (lstat("/etc", &st)) {
        perror("Unable to stat /etc");
        exit(ERR_ETC);
    }
    if (!S_ISDIR(st.st_mode)) {
        fprintf(stderr, "/etc is not a directory?!\n");
        exit(ERR_ETC);
    }
    if (st.st_uid != 0 || st.st_mode & S_IWOTH || ((st.st_gid != 0) && (st.st_mode & S_IWGRP))) {
        fprintf(stderr, "/etc is owned/writable by non-root users\n");
        exit(ERR_ETC);
    }

    verbose_printf(1, "Modifying syslogd configfile ... \n");
    /* O_CREAT with O_EXCL will make open() fail if the file already exists -
     * mostly to prevent 2 copies running at once */
    if ((config_fd = open("/etc/syslog.conf.new", O_WRONLY | O_CREAT | O_EXCL, 0644)) == -1) {
        perror("Unable to open new config file");
        exit(ERR_CONFIG);
    }

    if (!strcmp(buffer, "on"))
        snprintf(buffer, STRING_SIZE - 1,
                 "/bin/sed -e 's/^#\\?\\(\\*\\.\\*[[:blank:]]\\+@\\).\\+$/\\1%s/' /etc/syslog.conf >&%d", hostname,
                 config_fd);
    else
        snprintf(buffer, STRING_SIZE - 1,
                 "/bin/sed -e 's/^#\\?\\(\\*\\.\\*[[:blank:]]\\+@.\\+\\)$/#\\1/' /etc/syslog.conf >&%d", config_fd);

    /* if the return code isn't 0 failsafe */
    if ((rc = unpriv_system(buffer, 99, 99)) != 0) {
        fprintf(stderr, "sed returned bad exit code: %d\n", rc);
        close(config_fd);
        unlink("/etc/syslog.conf.new");
        exit(ERR_CONFIG);
    }
    close(config_fd);
    if (rename("/etc/syslog.conf.new", "/etc/syslog.conf") == -1) {
        perror("Unable to replace old config file");
        unlink("/etc/syslog.conf.new");
        exit(ERR_CONFIG);
    }


    /* Get syslogd to read the new config file */
    if ((fd = open("/var/run/syslogd.pid", O_RDONLY)) == -1) {
        verbose_printf(1, "Starting syslogd ... \n");
        if (errno == ENOENT) {
            /* pid file doesn't exists.. restart syslog */
            if ((rc = safe_system("/usr/sbin/syslogd -u syslogd")) == 0)
                return 0;
            else {
                fprintf(stderr, "Unable to restart syslogd - returned exit code %d\n", rc);
                exit(ERR_SYSLOG);
            }
        }
        else {
            /* Something odd is going on, failsafe */
            perror("Unable to open pid file");
            exit(ERR_SYSLOG);
        }
    }

    memset(buffer, 0, STRING_SIZE);
    if (read(fd, buffer, STRING_SIZE - 1) == -1) {
        close(fd);
        perror("Couldn't read from pid file");
        exit(ERR_SYSLOG);
    }
    close(fd);
    /* strtol does sanity checks that atoi doesn't do */
    errno = 0;
    pid = (int) strtol(buffer, (char **) NULL, 10);
    if (errno || pid <= 1) {
        fprintf(stderr, "Bad pid value\n");
        exit(ERR_SYSLOG);
    }
    verbose_printf(1, "Restarting syslogd ... \n");
    if (kill(pid, SIGHUP) == -1) {
        fprintf(stderr, "Unable to send SIGHUP\n");
        exit(ERR_SYSLOG);
    }

    return 0;
}
