/* This file is part of the IPCop Firewall.
 *
 * This program is distributed under the terms of the GNU General Public
 * Licence.  See the file COPYING for details.
 *
 * Copyright (C) 2004-05-31 Robert Kerr <rkerr@go.to>
 *
 * Loosely based on the smoothwall helper program by the same name,
 * portions are (c) Lawrence Manning, 2001
 *
 * $Id: installpackage.c 2088 2008-11-23 14:58:02Z owes $
 * 
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/file.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/statvfs.h>
#include <time.h>
#include "setuid.h"

#define ERR_ANY 1
#define ERR_TMPDIR 2
#define ERR_SIG 3
#define ERR_TAR 4
#define ERR_INFO 5
#define ERR_PACKLIST 6
#define ERR_INSTALLED 7
#define ERR_SETUP 9
#define ERR_MISSING_PREVIOUS 10
#define ERR_DISK 11

/* Supplement of available space on disk to be safe to untar on setup.
   Using one by one extraction method, size of biggest file is needed.
   Using normal method, we have to consider another process creating a file. */
#define MINIMUMSPACE 1500 * 1000

#define GPGFILE "/var/patches/patch.tgz.gpg"

/* The lines in the package information file and the patches/installed list
 * are often longer than STRING_SIZE so we use a larger buffer */
#define BUFFER_SIZE 4096

char *info = NULL;
FILE *infofile = NULL;
char command[STRING_SIZE];
/* Set to /var/patches if not enought available space on /var/log */
char tmpdir[] = "/var/log/pat_instal_XXXXXX";
/* Need a second tmp dir if /var/log is used but twice patch size is not available */
char tmp2dir[] = "/var/patches/pat_instal2_XXXXXX";
struct stat stbuf;
int param;
#define READ	1
#define INSTALL	2
#define DELETE	3
#define STRIP	4

void exithandler(void)
{
	if ( param==INSTALL || param==DELETE) {
		/* remove GPGFILE if abort due to low available space */
		if (!lstat( GPGFILE,&stbuf) ) unlink( GPGFILE);
		/* remove information and signature files if --read was used before */
		unlink("/var/patches/information");
		unlink("/var/patches/signature");
	}

	if(info) free(info);
	if(infofile)
	{
		flock(fileno(infofile), LOCK_UN);
		fclose(infofile);
	}
	/* Cleanup tmpdir */
	chdir("/var/patches"); /* get out of it before erasing */
	snprintf(command, STRING_SIZE - 1, "/bin/rm -rf %s", tmpdir);
	if(safe_system(command))
		perror("Couldn't remove temp dir");
	snprintf(command, STRING_SIZE - 1, "/bin/rm -rf %s", tmp2dir);
	if(safe_system(command))
		perror("Couldn't remove tmp2 dir");
}
static void usage()
{
	fprintf(stderr,"Usage:\n");
	fprintf(stderr,"installpackage --read\n");
	fprintf(stderr,"\tDecrypt " GPGFILE " and make information file available\n");
	fprintf(stderr,"\t" GPGFILE " remain on disk\n");
	fprintf(stderr,"installpackage --install\n");
	fprintf(stderr,"\tInstall content of " GPGFILE "\n");
	fprintf(stderr,"\t" GPGFILE " is removed after operation\n");
	fprintf(stderr,"installpackage --delete\n");
	fprintf(stderr,"\tDelete " GPGFILE " and information\n");
	fprintf(stderr,"installpackage --strip\n");
	fprintf(stderr,"\tRemove unselected kernel type (smp or monoprocessor)\n");
}

int main(int argc, char *argv[])
{
	char buffer[BUFFER_SIZE];
	int ret;
	int varlogused=0;
	struct statvfs statvfsbuf;
	double spaceavailable, rootspacerequired;

	if (!(initsetuid()))
		exit(1);

	/* need one parameter */
	if (argc!=2) {
		usage();
		exit(1);
	} else if (strcmp(argv[1], "--read")==0) {
		param = READ;
	} else if (strcmp(argv[1], "--install")==0) {
		param = INSTALL;
	} else if (strcmp(argv[1], "--delete")==0) {
		param = DELETE;
	} else if (strcmp(argv[1], "--strip")==0) {
		param = STRIP;
	} else if (strcmp(argv[1], "--help")==0) {
		usage();
		exit(0);
	} else {
		usage();
		exit(1);
	}

	if (param==STRIP) {
		exit(safe_system("/usr/local/bin/strip.pl")>>8);
	}

	atexit(exithandler);

	if (param==DELETE) {
		exit(0);
	}

	/* read size of the patch file */
	if (lstat( GPGFILE,&stbuf)) {
		fprintf(stderr, "Unable to stat " GPGFILE "\n");
		exit(ERR_ANY);
	}
	fprintf(stdout,"Update size is %ld kB\n", stbuf.st_size / 1024 );

	/* Check space available on disk to decrypt the gpg file */
	if (statvfs("/var/log", &statvfsbuf)) {
		fprintf(stderr,
			"Couldn't test available space on /var/log partition.\n");
		exit(ERR_ANY);
	}
	/* Need more than the patch size to write the tgz, setup and information
	  Test if it is possible to write decrypted file on /var/log partition
	  to spare patch size on rootfs */
	spaceavailable = statvfsbuf.f_frsize * statvfsbuf.f_bavail;
	if (stbuf.st_size + MINIMUMSPACE > spaceavailable) {
		/* Will not be able to extract on /var/log */
		strcpy (tmpdir,"/var/patches/install_XXXXXX");
		varlogused=0;
	} else {
		varlogused=1;
		if ( 2 * stbuf.st_size + MINIMUMSPACE < spaceavailable) {
			/* be able to create patch-1.tgz and patch.tar.gz */
			varlogused=2;
		}
	}
	fprintf(stdout,"Available space on /var/log is %4.0lf kB\n", spaceavailable / 1024 );
	/* Test rootfs partition */
	if (statvfs("/var/patches", &statvfsbuf)) {
		fprintf(stderr, "Couldn't test available space on rootfs partition.\n");
		exit(ERR_ANY);
	}
	spaceavailable = statvfsbuf.f_frsize * statvfsbuf.f_bavail;
	fprintf(stdout,"Available space on rootfs is %4.0lf kB\n",
			spaceavailable / 1024 );
	/* Process is :
	   - gpg file is on /var/patches at start
	   - file is decrypted to patch-1.tgz.gz inside a tmp dir ( on var/log or /var/patches )
	   - file is untarred inside tmp dir to patch.tar.gz, information and setup
	   - space required on rootfs to untar patch.tar.gz is between
	     patch.tar.gz size to patch.tar size.
	     Depending of available space, we may need to move files from
	     /var/log to rootfs once gpg file is decrypted (and removed) */
	if (varlogused) {
		/* var/patches now contain patch.tgz.gpg
		  When /var/log is used, place used by gpg file on
		  rootfs will be used during the final tar zxf */
		rootspacerequired=stbuf.st_size+MINIMUMSPACE;
	} else {
		/* Need more on rootfs when all is done there */
		rootspacerequired= 2 * stbuf.st_size + MINIMUMSPACE;
	}
	fprintf(stdout,"Required space on rootfs for normal tar zxf is %4.0lf kB\n",
			rootspacerequired / 1024 );
	if ((MINIMUMSPACE > spaceavailable) && (varlogused==0)) {
		/* no way to reset buffer size on 2.4 kernel, so advice to reboot */
		fprintf(stderr,
			"You need to free space on rootfs partition for update.\n"
			"Available space is %4.0lf kB,"
			"required space on rootfs is %4.0lf kB\n"
			"As disk buffers are smaller on boot, rebooting may help\n",
			spaceavailable / 1024 ,
			rootspacerequired / 1024);
		exit(ERR_DISK);
	}

	if(!mkdtemp(tmpdir)) {
		perror("Unable to create secure temp dir");
		exit(ERR_TMPDIR);
	}

	/* Verify and extract package */
	snprintf(command, STRING_SIZE-1,
		"/usr/bin/gpg --batch --logger-fd 1 --homedir /root/.gnupg "
		"-o %s/patch-1.tgz --decrypt " GPGFILE " >%s/signature",
		tmpdir, tmpdir);
	ret = safe_system(command) >> 8;
	/* prepare signature for display, take only the 2 first lines */
	snprintf(command, STRING_SIZE-1,
		"/bin/grep '^gpg' -m 2 %s/signature >/var/patches/signature",
		tmpdir);
	safe_system(command);
	if(ret==1) {
		/* 1=> gpg-key error */
		fprintf(stderr, "Invalid package: signature check failed\n");
		exit(ERR_SIG);
	}
	if(ret==2) {
		/* 2=> gpg pub key not found */
		fprintf(stderr, "Public signature not found (who signed package?) !\n");
		exit(ERR_SIG);
	}
	if (param==INSTALL) {
		/* gpg signed package is no more needed, free that space */
		unlink( GPGFILE );
		sync();
	}
	if (param==READ) {
		snprintf(command, STRING_SIZE-1,
			"/bin/tar xzf %s/patch-1.tgz -C /var/patches information", tmpdir);
		if (safe_system(command)) {
			fprintf(stderr, "information extraction fail\n");
			exit(ERR_INFO);
		}
		/* should be safe to keep information to be read from GUI with
		   umask 0644 and owned root */
		exit(0);
	}

	if (varlogused!=1) {
		strcpy (tmp2dir,tmpdir);
	} else {
		/* Need to shift the decrypted file to rootfs
		   as only patch size is available on /var/log */
		if(!mkdtemp(tmp2dir)) {
			perror("Unable to create secure tmp2 dir");
			exit(ERR_TMPDIR);
		}
		snprintf(command, STRING_SIZE-1,
			"/bin/mv %s/patch-1.tgz %s", tmpdir, tmp2dir);
		if(safe_system(command)) {
			fprintf(stderr, "mv patch-1.tgz fail\n");
			exit(ERR_TAR);
		}
		sync();
	}

	/* unzip the package */
	chdir (tmpdir);
	snprintf(command, STRING_SIZE-1,
		"/bin/tar xzf %s/patch-1.tgz", tmp2dir);
	if(safe_system(command)) {
		fprintf(stderr, "Invalid package: untar failed\n");
		exit(ERR_TAR);
	}

	/* patch-1 is no more needed, free that space */
	snprintf(command, STRING_SIZE-1, "%s/patch-1.tgz", tmp2dir);
	unlink( command );
	sync();

	/* And read 'information' to check validity */
	snprintf(buffer, STRING_SIZE-1, "%s/information", tmpdir);
	if(!(infofile = fopen(buffer,"r")))
	{
		if(errno == ENOENT)
			fprintf(stderr, "Invalid package: contains no information file\n");
		else
			perror("Unable to open package information file");
		exit(ERR_INFO);
	}
	if(!fgets(buffer, BUFFER_SIZE, infofile))
	{
		perror("Couldn't read package information");
		exit(ERR_INFO);
	}
	fclose(infofile);
	if(buffer[strlen(buffer)-1] == '\n')
		buffer[strlen(buffer)-1] = '\0';
	if(!strchr(buffer,'|'))
	{
		fprintf(stderr, "Invalid package: malformed information string.\n");
		exit(ERR_INFO);
	}
	info = strdup(buffer);

	/* check if able to append on 'installed' file */
	if(!(infofile = fopen("/var/ipcop/patches/installed","a+")))
	{
		perror("Unable to open installed package list");
		exit(ERR_PACKLIST);
	}
	/* get exclusive lock to prevent a mess if 2 copies run at once, and set
	 * close-on-exec flag so the FD doesn't leak to the setup script */
	flock(fileno(infofile), LOCK_EX);
	fcntl(fileno(infofile), F_SETFD, FD_CLOEXEC);

	/* install package */
	if (rootspacerequired > spaceavailable) {
		/* We will extract files one by one, very slow but work */
		fprintf(stdout, "Low space condition, slow extracting method used\n" );
		snprintf(command, STRING_SIZE - 1, "%s/setup lowspace", tmpdir);
	} else {
		snprintf(command, STRING_SIZE - 1, "%s/setup", tmpdir);
	}
	ret = safe_system(command)>>8;
	if(ret)
	{
		fprintf(stderr, "setup script returned exit code %d\n", ret);
		exit(ERR_SETUP);
	}

	/* write to package db */
	if(strncmp(info, "000|", 4))
	{
		time_t curtime = time(NULL);
		strftime(buffer, STRING_SIZE, "%Y-%m-%d", gmtime(&curtime));
		fprintf(infofile, "%s|%s\n", info, buffer);
		flock(fileno(infofile), LOCK_UN);
		fclose(infofile);
	} else { /* Full system upgrade to new version */
		flock(fileno(infofile), LOCK_UN);
		fclose(infofile);
		unlink("/var/ipcop/patches/available");
		unlink("/var/ipcop/patches/installed");
	}
	free(info);
	exit(0);
}
