/*
 * accountingctrl.c: Simple program to setup iptables rules for traffic accounting.
 *
 * This file is part of the IPCop Firewall.
 *
 * IPCop is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * IPCop is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with IPCop; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * $Id: accountingctrl.c 2220 2008-12-22 20:18:09Z dotzball $
 *
 */


#include <getopt.h>
#include <string.h>
#include "common.h"
#include "setuid.h"


void usage(char *prg, int exit_code)
{
    printf("Usage: %s [OPTION]\n\n", prg);
    printf("Options:\n");
    printf("  -v, --verbose         be verbose\n");
    printf("      --help            display this help and exit\n");
    exit(exit_code);
}


int main(int argc, char **argv)
{
    int enabled = 0;
    int i, j;
    char command[STRING_SIZE_LARGE];
    char options[STRING_SIZE];
    char dev_id[STRING_SIZE];
    char *iface;
    NODEKV *kv = NULL;


    static struct option long_options[] = {
        {"verbose", no_argument, 0, 'v'},
        {"help", no_argument, 0, 'h'},
        {0, 0, 0, 0}
    };
    int c;
    int option_index = 0;

    if (!(initsetuid()))
        exit(1);

    while ((c = getopt_long(argc, argv, "sv", long_options, &option_index)) != -1) {
        switch (c) {
        case 'v':              /* verbose */
            flag_verbose++;
            break;
        case 'h':
            usage(argv[0], 0);
        default:
            fprintf(stderr, "unknown option\n");
            usage(argv[0], 1);
        }
    }

    verbose_printf(1, "Reading traffic accounting settings ... \n");
    if (read_kv_from_file(&kv, "/var/ipcop/traffic/settings") != SUCCESS) {
        fprintf(stderr, "Cannot read traffic accounting settings\n");
        return 1;
    }
    if (test_kv(kv, "ENABLED", "on") == SUCCESS) {
        verbose_printf(2, "Traffic accounting enabled\n");
        enabled = 1;
    }
    else {
        verbose_printf(2, "Traffic accounting not enabled\n");
    }
    free_kv(&kv);

    verbose_printf(1, "Flush traffic accounting iptables rules... \n");
    safe_system("/sbin/iptables -F ACCOUNT_INPUT\n");
    safe_system("/sbin/iptables -F ACCOUNT_FORWARD_IN\n");
    safe_system("/sbin/iptables -F ACCOUNT_FORWARD_OUT\n");
    safe_system("/sbin/iptables -F ACCOUNT_OUTPUT");


    if (enabled) {
        //~ OPTIONS='-j ULOG --ulog-nlgroup 1 --ulog-qthreshold 50 --ulog-cprange 48';
        snprintf(options, STRING_SIZE, "-j ULOG --ulog-nlgroup 1 --ulog-qthreshold 50 --ulog-cprange 48");

        /* Fetch ethernet/settings, exit on error */
        read_ethernet_settings(1);

        verbose_printf(1, "Create traffic accounting iptables rules... \n");

        /* for all colours */
        for (i = 0; i < NONE; i++) {

            for (j = 1; j <= ipcop_ethernet.count[i]; j++) {
                if (i == RED) {
                    if (strlen(ipcop_ethernet.red_device[j]) == (strlen(""))) {
                        continue;
                    }

                    iface = ipcop_ethernet.red_device[j];
                }
                else {
                    iface = ipcop_ethernet.device[i][j];
                }

                snprintf(dev_id, STRING_SIZE, "%s_%d", ipcop_colours_text[i], j);
                verbose_printf(2, "  For %s:\n", dev_id);

                snprintf(command, STRING_SIZE_LARGE, "/sbin/iptables -A ACCOUNT_INPUT -i %s %s --ulog-prefix %s_INPUT",
                         iface, options, dev_id);
                verbose_printf(2, "    %s\n", command);
                safe_system(command);

                snprintf(command, STRING_SIZE_LARGE,
                         "/sbin/iptables -A ACCOUNT_FORWARD_IN -i %s %s --ulog-prefix %s_FORWARD_IN", iface, options,
                         dev_id);
                verbose_printf(2, "    %s\n", command);
                safe_system(command);

                snprintf(command, STRING_SIZE_LARGE,
                         "/sbin/iptables -A ACCOUNT_FORWARD_OUT -o %s %s --ulog-prefix %s_FORWARD_OUT", iface, options,
                         dev_id);
                verbose_printf(2, "    %s\n", command);
                safe_system(command);

                snprintf(command, STRING_SIZE_LARGE,
                         "/sbin/iptables -A ACCOUNT_OUTPUT -o %s %s --ulog-prefix %s_OUTPUT", iface, options, dev_id);
                verbose_printf(2, "    %s\n", command);
                safe_system(command);

            }
        }
    }

    return 0;
}
