/*
 * This file is part of the IPCop Firewall.
 *
 * IPCop is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * IPCop is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with IPCop; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * $Id: openvpnctrl.c 2737 2009-04-20 15:19:41Z owes $
 *
 */


#include <getopt.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include "common.h"
#include "setuid.h"


static int flag_start = 0;
static int flag_stop = 0;
static int flag_restart = 0;
static int enabled[CFG_COLOURS_COUNT][MAX_NETWORK_COLOUR + 1];
static int enabled_count;


void usage(char *prg, int exit_code)
{
    printf("Usage: %s [OPTION]\n\n", prg);
    printf("Options:\n");
    printf("  --start               .\n"); 
    printf("  --stop                .\n"); 
    printf("  --restart             .\n"); 
    printf("  -v, --verbose         be verbose\n");
    printf("      --help            display this help and exit\n");
    exit(exit_code);
}


int main(int argc, char *argv[])
{
    int i, j;
    char buffer[STRING_SIZE];
    NODEKV *ovpn_kv = NULL;

    static struct option long_options[] =
    {
        { "start",   no_argument, &flag_start, 1 },
        { "stop",    no_argument, &flag_stop, 1 },
        { "restart", no_argument, &flag_restart, 1 },
        { "verbose", no_argument, 0, 'v' },
        { "help",    no_argument, 0, 'h' },
        { 0, 0, 0, 0}
    };
    int c;
    int option_index = 0;

    while ((c = getopt_long(argc, argv, "v", long_options, &option_index)) != -1) {
        switch (c) {
        case 0:
            break;
        case 'v':              /* verbose */
            flag_verbose++;
            break;
        case 'h':
            usage(argv[0], 0);
        default:
            fprintf(stderr, "unknown option\n");
            usage(argv[0], 1);
        }
    }
    
    if (!flag_start && !flag_stop && !flag_restart) {
        /* need at least one of start, stop, restart */
        usage(argv[0], 1);
    }

    if ( !(initsetuid()) )
        exit(1);

    /* Terminate running OpenVPN server */
    if (access("/var/run/openvpn.pid", 0) != -1) {
        verbose_printf(2, "Stopping OpenVPN server ... \n");
        if (mysignalpidfile("/var/run/openvpn.pid", SIGTERM) != SUCCESS ) {
            exit(0);
        }
        safe_system("/bin/rm -f /var/run/openvpn.pid");

        if (flag_start || flag_restart) {
            sleep(1);
        }
    }

    /* Fetch ethernet/settings, exit on error */
    read_ethernet_settings(1);

    /* Fetch openvpn/settings */
    verbose_printf(1, "Reading OpenVPN settings ... \n");
    if (read_kv_from_file(&ovpn_kv, "/var/ipcop/ovpn/settings") != SUCCESS) {
        fprintf(stderr, "Cannot read OpenVPN settings\n");
        exit(1);
    }

    enabled_count = 0;
    memset(enabled, 0, sizeof(enabled));
    for (i = 0; i < CFG_COLOURS_COUNT; i++) {
        /* filter GREEN, no OpenVPN daemon there */
        if (i == GREEN) {
            continue;
        }

        for (j = 1; j <= MAX_NETWORK_COLOUR; j++) {
            snprintf(buffer, STRING_SIZE, "ENABLED_%s_%d", ipcop_colours_text[i], j);

            if (test_kv(ovpn_kv, buffer, "on") == SUCCESS) {
                /* this card is enabled in openvpn/settings */
                if (j > ipcop_ethernet.count[i]) {
                    /* card is missing in ethernet/settings */
                    fprintf(stderr, "%s_%d enabled but no device defined\n", ipcop_colours_text[i], j);
                    exit(1);
                }

                enabled[i][j] = 1;
                enabled_count++;
            }
        }
    }
    verbose_printf(2, "  %d enabled interface(s)\n", enabled_count);

    if (enabled_count == 0) {
        verbose_printf(1, "OpenVPN not enabled ... \n");
    }

    if (enabled_count && (flag_start || flag_restart)) {
        safe_system("/sbin/modprobe tun");
        verbose_printf(1, "Starting OpenVPN server ... \n");
        safe_system("/usr/sbin/openvpn --config /var/ipcop/ovpn/server.conf");
    }

    /* rebuild rules, maybe server is now disabled, or some other change */
    verbose_printf(1, "Rebuild firewall rules ... \n");
    safe_system("/usr/local/bin/setfwrules --ipcop");

    return(0);
}
