/*
 *
 * File originally from the Smoothwall project
 * (c) 2001 Smoothwall Team
 *
 * $Id: ipsecctrl.c 2748 2009-04-27 20:29:24Z owes $
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
#include "setuid.h"
#include "common.h"


/*
    This module is responsible for start stop of the vpn system.

    1) it allows AH & ESP to get in from interface where a vpn is mounted
	The NAT traversal is used on the udp 4500 port.

    2) it starts the ipsec daemon
	The RED interface is a problem because it can be up or down a startup.
	Then, the state change and it must not affect other VPN mounted on
	other interface.
	Unfortunatly, openswan 1 cannot do that correctly. It cannot use an
	interface without restarting everything.

    IPCop should control vpn this way:

    rc.netaddrsesup.up
    	    call ipsecctrl once to start vpns on all interface
    	    RED based vpn won't start because "auto=ignore" instead off "auto=start"

    rc.updatered
	    call ipsectrl to turn on or off vpn based on RED

    but now it is only:

    rc.updatered
	    call ipsectrl S at every event on RED.
	    Consequence: BLUE vpn is not started until RED goes up.


*/


void usage()
{
    fprintf(stderr, "Usage:\n");
    fprintf(stderr, "\tipsecctrl S [connectionkey]\n");
    fprintf(stderr, "\tipsecctrl D [connectionkey]\n");
    fprintf(stderr, "\tipsecctrl R\n");
    fprintf(stderr, "\t\tS : Start/Restart Connection\n");
    fprintf(stderr, "\t\tD : Stop Connection\n");
    fprintf(stderr, "\t\tR : Reload Certificates and Secrets\n");
}


void add_alias_interfaces(char *configtype, char *redtype, char *redif, int offset)     //reserve room for ipsec0=red, ipsec1=green, ipsec2=orange,ipsec3=blue
{
    FILE *file = NULL;
    char s[STRING_SIZE];
    int alias = 0;

    /* Check for RED_COUNT!=0 i.e. RED ethernet present. If not,
     * exit gracefully.  This is not an error... */
    if (strcmp(configtype, "0") == 0)
        return;

    /* Now check the RED_TYPE - aliases only work with STATIC. */
    if (!(strcmp(redtype, "STATIC") == 0))
        return;

    /* Now set up the new aliases from the config file */
    if (!(file = fopen("/var/ipcop/ethernet/aliases", "r"))) {
        fprintf(stderr, "Unable to open aliases configuration file\n");
        return;
    }
    while (fgets(s, STRING_SIZE, file) != NULL && (offset + alias) < 16) {
        if (s[strlen(s) - 1] == '\n')
            s[strlen(s) - 1] = '\0';
        int count = 0;
        char *aliasip = NULL;
        char *enabled = NULL;
        char *comment = NULL;
        char *sptr = strtok(s, ",");
        while (sptr) {
            if (count == 0)
                aliasip = sptr;
            if (count == 1)
                enabled = sptr;
            else
                comment = sptr;
            count++;
            sptr = strtok(NULL, ",");
        }

        if (!(aliasip && enabled))
            continue;

        if (!VALID_IP(aliasip)) {
            fprintf(stderr, "Bad alias : %s\n", aliasip);
            return;
        }

        if (strcmp(enabled, "on") == 0) {
            memset(s, 0, STRING_SIZE);
            snprintf(s, STRING_SIZE - 1, "/usr/sbin/ipsec tncfg --attach --virtual ipsec%d --physical %s:%d >/dev/null",
                     offset + alias, redif, alias);
            safe_system(s);
            alias++;
        }
    }
}

/*
 return values from the vpn config file or false if not 'on'
*/
int decode_line(char *s, char **key, char **name, char **type, char **interface)
{
    int count = 0;
    *key = NULL;
    *name = NULL;
    *type = NULL;
    *interface = NULL;

    if (s[strlen(s) - 1] == '\n')
        s[strlen(s) - 1] = '\0';

    char *result = strsep(&s, ",");
    while (result) {
        if (count == 0)
            *key = result;
        if ((count == 1) && strcmp(result, "on") != 0)
            return 0;           // a disabled line
        if (count == 2)
            *name = result;
        if (count == 4)
            *type = result;
        if (count == 27)
            *interface = result;
        count++;
        result = strsep(&s, ",");
    }

    // check other syntax
    if (!*name)
        return 0;
    if (count != 30) {
        // name was found, so we know that keynumber and name (may) make some sense
        fprintf(stderr, "Bad configline, key %s, name %s\n", *key, *name);
        return 0;
    }

    if (strspn(*name, LETTERS_NUMBERS) != strlen(*name)) {
        fprintf(stderr, "Bad connection name: %s\n", *name);
        return 0;
    }

    if (!(strcmp(*type, "host") == 0 || strcmp(*type, "net") == 0)) {
        fprintf(stderr, "Bad connection type: %s\n", *type);
        return 0;
    }

    if (!(strcmp(*interface, "RED") == 0 || strcmp(*interface, "GREEN") == 0 ||
          strcmp(*interface, "ORANGE") == 0 || strcmp(*interface, "BLUE") == 0)) {
        fprintf(stderr, "Bad interface name: %s\n", *interface);
        return 0;
    }
    //it's a valid & active line
    return 1;
}

/*
    issue ipsec commmands to turn on connection 'name'
*/
void turn_connection_on(char *name, char *type)
{
    char command[STRING_SIZE];

    safe_system("/usr/sbin/ipsec auto --rereadsecrets >/dev/null");
    memset(command, 0, STRING_SIZE);
    snprintf(command, STRING_SIZE - 1, "/usr/sbin/ipsec auto --replace %s >/dev/null", name);
    safe_system(command);
    if (strcmp(type, "net") == 0) {
        memset(command, 0, STRING_SIZE);
        snprintf(command, STRING_SIZE - 1, "/usr/sbin/ipsec auto --asynchronous --up %s >/dev/null", name);
        safe_system(command);
    }
}

/*
    issue ipsec commmands to turn off connection 'name'
*/
void turn_connection_off(char *name)
{
    char command[STRING_SIZE];

    memset(command, 0, STRING_SIZE);
    snprintf(command, STRING_SIZE - 1, "/usr/sbin/ipsec auto --down %s >/dev/null", name);
    safe_system(command);
    memset(command, 0, STRING_SIZE);
    snprintf(command, STRING_SIZE - 1, "/usr/sbin/ipsec auto --delete %s >/dev/null", name);
    safe_system(command);
    safe_system("/usr/sbin/ipsec auto --rereadsecrets >/dev/null");
}


int main(int argc, char *argv[])
{

    char configtype[STRING_SIZE] = "";
    char redtype[STRING_SIZE] = "";
    NODEKV *ipsec_kv = NULL;

    if (argc < 2) {
        usage();
        exit(1);
    }
    if (!(initsetuid()))
        exit(1);

    /* FIXME: workaround for pclose() issue - still no real idea why
     * this is happening */
    signal(SIGCHLD, SIG_DFL);

    /* handle operations that doesn't need start the ipsec system */
    if (argc == 2) {
        if (strcmp(argv[1], "D") == 0) {
            safe_system("/usr/local/bin/vpn-watch --stop");
            /* Only shutdown pluto if it really is running */
            int fd;
            /* Get pluto pid */
            if ((fd = open("/var/run/pluto.pid", O_RDONLY)) != -1) {
                safe_system("/etc/rc.d/ipsec stop 2> /dev/null >/dev/null");
                close(fd);
            }
            exit(0);
        }

        if (strcmp(argv[1], "R") == 0) {
            safe_system("/usr/sbin/ipsec auto --rereadall");
            exit(0);
        }

        if (strcmp(argv[1], "status") == 0) {
			safe_system("/usr/sbin/ipsec auto --status");
			exit(0);
        }
    }

    /* stop the watch script as soon as possible */
    safe_system("/usr/local/bin/vpn-watch --stop");

    /* read vpn config */
    if (read_kv_from_file(&ipsec_kv, "/var/ipcop/vpn/settings") != SUCCESS) {
        fprintf(stderr, "Cannot read vpn settings\n");
        exit(1);
    }

    /* check is the vpn system is enabled */
    {
        char s[STRING_SIZE] = "";
        find_kv_default(ipsec_kv, "ENABLED", s);
        free_kv(&ipsec_kv);
        if (strcmp(s, "on") != 0)
            exit(0);
    }

    /* read interface settings */
    if (read_kv_from_file(&eth_kv, "/var/ipcop/ethernet/settings") != SUCCESS) {
        fprintf(stderr, "Cannot read ethernet settings\n");
        exit(1);
    }
    if (find_kv_default(eth_kv, "RED_COUNT", configtype) != SUCCESS) {
        fprintf(stderr, "Cannot read RED_COUNT\n");
        exit(1);
    }
    find_kv_default(eth_kv, "RED_1_TYPE", redtype);


    /* Loop through the config file to find physical interface that will accept IPSEC */
    int enable_red = 0;         // states 0: not used
    int enable_green = 0;       //        1: error condition
    int enable_orange = 0;      //        2: good
    int enable_blue = 0;
    char if_red[STRING_SIZE] = "";
    char if_green[STRING_SIZE] = "";
    char if_orange[STRING_SIZE] = "";
    char if_blue[STRING_SIZE] = "";
    char *s;
    size_t s_size = STRING_SIZE;
    FILE *file = NULL;

    s = malloc(s_size);
    if (s == NULL) {
        fprintf(stderr, "Couldn't allocate memory");
        exit(1);
    }

    if (!(file = fopen("/var/ipcop/vpn/config", "r"))) {
        fprintf(stderr, "Couldn't open vpn settings file");
        exit(1);
    }
    while (getline(&s, &s_size, file) != -1) {
        char *key;
        char *name;
        char *type;
        char *interface;
        if (!decode_line(s, &key, &name, &type, &interface))
            continue;
        /* search interface */
        if (!enable_red && strcmp(interface, "RED") == 0) {
            // when RED is up, find interface name in special file
            FILE *ifacefile = NULL;
            if ((ifacefile = fopen("/var/ipcop/red/iface", "r"))) {
                if (fgets(if_red, STRING_SIZE, ifacefile)) {
                    if (if_red[strlen(if_red) - 1] == '\n')
                        if_red[strlen(if_red) - 1] = '\0';
                }
                fclose(ifacefile);

                if (VALID_DEVICE(if_red))
                    enable_red += 2;    // present and running
            }
        }

        if (!enable_green && strcmp(interface, "GREEN") == 0) {
            enable_green = 1;

            if ((find_kv_default(eth_kv, "GREEN_1_DEV", if_green) == SUCCESS) && VALID_DEVICE(if_green))
                enable_green++;
            else
                fprintf(stderr, "IPSec enabled on green but green interface is invalid or not found\n");
        }

        if (!enable_orange && strcmp(interface, "ORANGE") == 0) {
            enable_orange = 1;

            if ((find_kv_default(eth_kv, "ORANGE_1_DEV", if_orange) == SUCCESS) && VALID_DEVICE(if_orange))
                enable_orange++;
            else
                fprintf(stderr, "IPSec enabled on orange but orange interface is invalid or not found\n");
        }

        if (!enable_blue && strcmp(interface, "BLUE") == 0) {
            enable_blue++;

            if ((find_kv_default(eth_kv, "BLUE_1_DEV", if_blue) == SUCCESS) && VALID_DEVICE(if_blue))
                enable_blue++;
            else
                fprintf(stderr, "IPSec enabled on blue but blue interface is invalid or not found\n");

        }
    }
    fclose(file);
    free_kv(&eth_kv);

    // do nothing if something is in error condition
    if ((enable_red == 1) || (enable_green == 1) || (enable_orange == 1) || (enable_blue == 1))
        exit(1);

    // reset firewall rules
    safe_system("/usr/local/bin/setfwrules --ipcop");

    // exit if nothing to do
    if ((enable_red + enable_green + enable_orange + enable_blue) == 0)
        exit(0);

    // start the system
    if ((argc == 2) && strcmp(argv[1], "S") == 0) {
        safe_system("/sbin/modprobe ipsec");
        safe_system("/usr/sbin/ipsec tncfg --clear >/dev/null");
        safe_system("/etc/rc.d/ipsec restart >/dev/null");
        add_alias_interfaces(configtype, redtype, if_red,
                             (enable_red + enable_green + enable_orange + enable_blue) >> 1);
        safe_system("/usr/local/bin/vpn-watch --start");
        exit(0);
    }

    // it is a selective start or stop
    // second param is only a number 'key'
    if ((argc == 2) || strspn(argv[2], NUMBERS) != strlen(argv[2])) {
        fprintf(stderr, "Bad arg\n");
        usage();
        exit(1);
    }

    // search the vpn pointed by 'key'
    if (!(file = fopen("/var/ipcop/vpn/config", "r"))) {
        fprintf(stderr, "Couldn't open vpn settings file");
        exit(1);
    }
    while (getline(&s, &s_size, file) != -1) {
        char *key;
        char *name;
        char *type;
        char *interface;
        if (!decode_line(s, &key, &name, &type, &interface))
            continue;

        // start/stop a vpn if belonging to specified interface
        if (strcmp(argv[1], interface) == 0) {
            if (strcmp(argv[2], "0") == 0)
                turn_connection_off(name);
            else
                turn_connection_on(name, type);
            continue;
        }
        // is it the 'key' requested ?
        if (strcmp(argv[2], key) != 0)
            continue;
        // Start or Delete this Connection
        if (strcmp(argv[1], "S") == 0)
            turn_connection_on(name, type);
        else if (strcmp(argv[1], "D") == 0)
            turn_connection_off(name);
        else {
            fprintf(stderr, "Bad command\n");
            exit(1);
        }
    }
    fclose(file);
    safe_system("/usr/local/bin/vpn-watch --start");
    return 0;
}
