# SmoothWall CGIs
#
# This code is distributed under the terms of the GPL
#
# (c) The SmoothWall Team
# Copyright (C) 2002 Alex Hudson - getcgihash() rewrite
# Copyright (C) 2002 Bob Grant <bob@cache.ucr.edu> - validmac()
# Copyright (c) 2002/04/13 Steve Bootes - add alias section, helper functions
# Copyright (c) 2002/08/23 Mark Wormgoor <mark@wormgoor.com> validfqdn()
# Copyright (c) 2003/09/11 Darren Critchley <darrenc@telus.net> srtarray()
# Copyright (c) 2004-2009 The IPCop Team
#
# $Id: general-functions.pl 2691 2009-04-11 09:36:41Z owes $
#

package General;

use strict;
use Socket;
use CGI();
use IO::Socket;
use Net::DNS;
use Net::SSLeay;

$| = 1;    # line buffering

$General::version    = 'VERSION';
$General::swroot     = '/var/ipcop';
$General::noipprefix = 'noipg-';
@General::weekDays = ('sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday');
# Make sure the days are seen by translation stuff
# $Lang::tr{'sunday'} $Lang::tr{'monday'} $Lang::tr{'tuesday'} $Lang::tr{'wednesday'}
# $Lang::tr{'thursday'} $Lang::tr{'friday'} $Lang::tr{'saturday'}

#
# log ("message") use default 'ipcop' tag
# log ("tag","message") use your tag
#
sub log
{
    my $tag = 'ipcop';
    $tag = shift if (@_ > 1);
    my $logmessage = $_[0];
    $logmessage =~ /([\w\W]*)/;
    $logmessage = $1;
    system('/usr/bin/logger', '-t', $tag, $logmessage);
}

sub getcgihash
{
    my ($hash, $params) = @_;
    my $cgi = CGI->new();
    return if ($ENV{'REQUEST_METHOD'} ne 'POST');
    if (!$params->{'wantfile'}) {
        $CGI::DISABLE_UPLOADS = 1;
        $CGI::POST_MAX        = 512 * 1024;
    }
    else {
        $CGI::POST_MAX = 10 * 1024 * 1024;
    }

    $cgi->referer() =~ m/^https?\:\/\/([^\/]+)/;
    my $referer = $1;
    $cgi->url() =~ m/^https?\:\/\/([^\/]+)/;
    my $servername = $1;
    if ($referer ne $servername) {
        &General::log('ipcop', "Invalid referer: doesn't match servername!");
        return;
    }

    ### Modified for getting multi-vars, split by |
    my %temp = $cgi->Vars();
    foreach my $key (keys %temp) {
        $hash->{$key} = $temp{$key};
        $hash->{$key} =~ s/\0/|/g;
        $hash->{$key} =~ s/^\s*(.*?)\s*$/$1/;
    }

    if (($params->{'wantfile'}) && ($params->{'filevar'})) {
        $hash->{$params->{'filevar'}} = $cgi->upload($params->{'filevar'});
    }
    return;
}

sub readhash
{
    my $filename = $_[0];
    my $hash     = $_[1];
    my ($var, $val);

    # Some ipcop code expects that readhash 'complete' the hash if new entries
    # are presents. Not clear it !!!
    #%$hash = ();

    open(FILE, $filename) or die "Unable to read file $filename";

    while (<FILE>) {
        chop;
        ($var, $val) = split /=/, $_, 2;
        if ($var) {
            $val =~ s/^\'//g;
            $val =~ s/\'$//g;

            # Untaint variables read from hash
            $var =~ /([A-Za-z0-9_-]*)/;
            $var = $1;
            $val =~ /([\w\W]*)/;
            $val = $1;
            $hash->{$var} = $val;
        }
    }
    close FILE;
}

sub writehash
{
    my $filename = $_[0];
    my $hash     = $_[1];
    my ($var, $val);

    # write cgi vars to the file.
    open(FILE, ">${filename}") or die "Unable to write file $filename";
    flock FILE, 2;
    foreach $var (keys %$hash) {
        $val = $hash->{$var};

        # Darren Critchley Jan 17, 2003 added the following because when submitting with a graphic, the x and y
        # location of the mouse are submitted as well, this was being written to the settings file causing
        # some serious grief! This skips the variable.x and variable.y
        if (!($var =~ /\.(x|y)$/)) {
            if ($val =~ / /) {
                $val = "\'$val\'";
            }
            if (!($var =~ /^ACTION/)) {
                print FILE "${var}=${val}\n";
            }
        }
    }
    close FILE;
}

sub age
{
    my ($dev, $ino, $mode, $nlink, $uid, $gid, $rdev, $size, $atime, $mtime, $ctime, $blksize, $blocks) = stat $_[0];
    my $now = time;

    my $totalsecs  = $now - $mtime;
    my $days       = int($totalsecs / 86400);
    my $totalhours = int($totalsecs / 3600);
    my $hours      = $totalhours % 24;
    my $totalmins  = int($totalsecs / 60);
    my $mins       = $totalmins % 60;
    my $secs       = $totalsecs % 60;

    return "${days}d ${hours}h ${mins}m ${secs}s";
}

sub validip
{
    my $ip = $_[0];

    if (!($ip =~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)$/)) {
        return 0;
    }
    else {
        my @octets = ($1, $2, $3, $4);
        foreach $_ (@octets) {
            if (/^0./) {
                return 0;
            }
            if ($_ < 0 || $_ > 255) {
                return 0;
            }
        }
        return 1;
    }
}

sub validmask
{
    my $mask = $_[0];

    # secord part an ip?
    if (&validip($mask)) {
        return 1;
    }

    # second part a number?
    if (/^0/) {
        return 0;
    }
    if (!($mask =~ /^\d+$/)) {
        return 0;
    }
    if ($mask >= 0 && $mask <= 32) {
        return 1;
    }
    return 0;
}

sub validipormask
{
    my $ipormask = $_[0];

    # see if it is a IP only.
    if (&validip($ipormask)) {
        return 1;
    }

    # split it into number and mask.
    if (!($ipormask =~ /^(.*?)\/(.*?)$/)) {
        return 0;
    }
    my $ip   = $1;
    my $mask = $2;

    # first part not a ip?
    if (!(&validip($ip))) {
        return 0;
    }
    return &validmask($mask);
}

sub validipandmask
{
    my $ipandmask = $_[0];

    # split it into number and mask.
    if (!($ipandmask =~ /^(.*?)\/(.*?)$/)) {
        return 0;
    }
    my $ip   = $1;
    my $mask = $2;

    # first part not a ip?
    if (!(&validip($ip))) {
        return 0;
    }
    return &validmask($mask);
}

sub validport
{
    $_ = $_[0];

    if (!/^\d+$/) {
        return 0;
    }
    if (/^0./) {
        return 0;
    }
    if ($_ >= 1 && $_ <= 65535) {
        return 1;
    }
    return 0;
}

sub validmac
{
    my $checkmac = $_[0];
    my $ot       = '[0-9a-f]{2}';    # 2 Hex digits (one octet)
    if ($checkmac !~ /^$ot:$ot:$ot:$ot:$ot:$ot$/i) {
        return 0;
    }
    return 1;
}

sub validhostname
{

    # Checks a hostname against less strict rules than RFC1035
    my $hostname = $_[0];

    # Each part should be at least two characters in length
    # but no more than 63 characters
    if (length($hostname) < 1 || length($hostname) > 63) {
        return 0;
    }

    # Only valid characters are a-z, A-Z, 0-9 and -
    if ($hostname !~ /^[a-zA-Z0-9-]*$/) {
        return 0;
    }

    # First character can only be a letter or a digit
    if (substr($hostname, 0, 1) !~ /^[a-zA-Z0-9]*$/) {
        return 0;
    }

    # Last character can only be a letter or a digit
    if (substr($hostname, -1, 1) !~ /^[a-zA-Z0-9]*$/) {
        return 0;
    }
    return 1;
}

sub validdomainname
{
    my $part;

    # Checks a domain name against less strict rules than RFC1035
    my $domainname = $_[0];
    my @parts = split(/\./, $domainname);    # Split hostname at the '.'

    foreach $part (@parts) {

        # Each part should be at least two characters in length
        # but no more than 63 characters
        if (length($part) < 2 || length($part) > 63) {
            return 0;
        }

        # Only valid characters are a-z, A-Z, 0-9 and -
        if ($part !~ /^[a-zA-Z0-9-]*$/) {
            return 0;
        }

        # First character can only be a letter or a digit
        if (substr($part, 0, 1) !~ /^[a-zA-Z0-9]*$/) {
            return 0;
        }

        # Last character can only be a letter or a digit
        if (substr($part, -1, 1) !~ /^[a-zA-Z0-9]*$/) {
            return 0;
        }
    }
    return 1;
}

sub validfqdn
{
    my $part;

    # Checks a fully qualified domain name against less strict rules than RFC1035
    # url like 0.us.pool.ntp.org are used
    my $fqdn = $_[0];
    my @parts = split(/\./, $fqdn);    # Split hostname at the '.'
    if (scalar(@parts) < 2) {    # At least two parts should
        return 0;
    }    # exist in a FQDN
         # (i.e. hostname.domain)
    foreach $part (@parts) {

        # Each part should be at least one character in length
        # but no more than 63 characters
        if (length($part) < 1 || length($part) > 63) {
            return 0;
        }

        # Only valid characters are a-z, A-Z, 0-9 and -
        if ($part !~ /^[a-zA-Z0-9-]*$/) {
            return 0;
        }

        # First character can only be a letter or a digit
        if (substr($part, 0, 1) !~ /^[a-zA-Z0-9]*$/) {
            return 0;
        }

        # Last character can only be a letter or a digit
        if (substr($part, -1, 1) !~ /^[a-zA-Z0-9]*$/) {
            return 0;
        }
    }
    return 1;
}

sub validiporfqdn    # check ip or fdqn
{
    my $test = shift;
    return validip($test) || validfqdn($test);
}

sub validportrange    # used to check a port range
{
    my $port = $_[0];    # port values
    $port =~ tr/-/:/;    # replace all - with colons just in case someone used -
    my $srcdst = $_[1];  # is it a source or destination port

    if (!($port =~ /^(\d+)\:(\d+)$/)) {

        if (!(&validport($port))) {
            if ($srcdst eq 'src') {
                return $Lang::tr{'source port numbers'};
            }
            else {
                return $Lang::tr{'destination port numbers'};
            }
        }
    }
    else {
        my @ports = ($1, $2);
        if ($1 >= $2) {
            if ($srcdst eq 'src') {
                return $Lang::tr{'bad source range'};
            }
            else {
                return $Lang::tr{'bad destination range'};
            }
        }
        foreach $_ (@ports) {
            if (!(&validport($_))) {
                if ($srcdst eq 'src') {
                    return $Lang::tr{'source port numbers'};
                }
                else {
                    return $Lang::tr{'destination port numbers'};
                }
            }
        }
        return;
    }
}

#
# verify host by using a DNS resolve
#
sub validdnshost {
    my $hostname = $_[0];
    unless ($hostname) { return "No hostname"};
    my $res = new Net::DNS::Resolver;
    my $query = $res->search("$hostname");
    if ($query) {
        foreach my $rr ($query->answer) {
            ## Potential bug - we are only looking at A records:
            return 0 if $rr->type eq "A";
        }
    } else {
        return $res->errorstring;
    }
}

# Test if IP is within a subnet
# Call: IpInSubnet (Addr, Subnet, Subnet Mask)
#       Subnet can be an IP of the subnet: 10.0.0.0 or 10.0.0.1
#       Everything in dottted notation
# Return: TRUE/FALSE
sub IpInSubnet
{
    my $ip    = unpack('N', &Socket::inet_aton(shift));
    my $start = unpack('N', &Socket::inet_aton(shift));
    my $mask  = unpack('N', &Socket::inet_aton(shift));
    $start &= $mask;    # base of subnet...
    my $end = $start + ~$mask;
    return (($ip >= $start) && ($ip <= $end));
}

#
# Return the following IP (IP+1) in dotted notation.
# Call: NextIP ('1.1.1.1');
# Return: '1.1.1.2'
#
sub NextIP
{
    return &Socket::inet_ntoa(pack("N", 1 + unpack('N', &Socket::inet_aton(shift))));
}

sub validemail
{
    my $mail = shift;
    return 0 if ($mail !~ /^[0-9a-zA-Z\.\-\_]+\@[0-9a-zA-Z\.\-]+$/);
    return 0 if ($mail =~ /^[^0-9a-zA-Z]|[^0-9a-zA-Z]$/);
    return 0 if ($mail !~ /([0-9a-zA-Z]{1})\@./);
    return 0 if ($mail !~ /.\@([0-9a-zA-Z]{1})/);
    return 0 if ($mail =~ /.\.\-.|.\-\..|.\.\..|.\-\-./g);
    return 0 if ($mail =~ /.\.\_.|.\-\_.|.\_\..|.\_\-.|.\_\_./g);
    return 0 if ($mail !~ /\.([a-zA-Z]{2,4})$/);
    return 1;
}

#
# Currently only vpnmain use this three procs (readhasharray, writehasharray, findhasharray)
# The 'key' used is numeric but is perfectly unneeded! This will to be removed so don't use
# this code. Vpnmain will be splitted in parts: x509/pki, connection ipsec, connection other,... .
#
sub readhasharray
{
    my ($filename, $hash) = @_;
    %$hash = ();

    open(FILE, $filename) or die "Unable to read file $filename";

    while (<FILE>) {
        my ($key, $rest, @temp);
        chomp;
        ($key, $rest) = split(/,/, $_, 2);
        if ($key =~ /^[0-9]+$/) {
            @temp = split(/,/, $rest);
            $hash->{$key} = \@temp;
        }
    }
    close FILE;
    return;
}

sub writehasharray
{
    my ($filename, $hash) = @_;
    my ($key, @temp, $i);

    open(FILE, ">$filename") or die "Unable to write to file $filename";

    foreach $key (keys %$hash) {
        if ($key =~ /^[0-9]+$/) {
            print FILE "$key";
            foreach $i (0 .. $#{$hash->{$key}}) {
                print FILE ",$hash->{$key}[$i]";
            }
            print FILE "\n";
        }
    }
    close FILE;
    return;
}

sub findhasharraykey
{
    foreach my $i (1 .. 1000000) {
        if (!exists $_[0]{$i}) {
            return $i;
        }
    }
}

sub srtarray

    # Darren Critchley - darrenc@telus.net - (c) 2003
    # &srtarray(SortOrder, AlphaNumeric, SortDirection, ArrayToBeSorted)
    # This subroutine will take the following parameters:
    #   ColumnNumber = the column which you want to sort on, starts at 1
    #   AlphaNumberic = a or n (lowercase) defines whether the sort should be alpha or numberic
    #   SortDirection = asc or dsc (lowercase) Ascending or Descending sort
    #   ArrayToBeSorted = the array that wants sorting
    #
    #   Returns an array that is sorted to your specs
    #
    #   If SortOrder is greater than the elements in array, then it defaults to the first element
    #
{
    my ($colno, $alpnum, $srtdir, @tobesorted) = @_;
    my @tmparray;
    my @srtedarray;
    my $line;
    my $newline;
    my $ctr;
    my $ttlitems = scalar @tobesorted;    # want to know the number of rows in the passed array
    if ($ttlitems < 1) {                  # if no items, don't waste our time lets leave
        return (@tobesorted);
    }
    my @tmp = split(/\,/, $tobesorted[0]);
    $ttlitems = scalar @tmp;              # this should be the number of elements in each row of the passed in array

    # Darren Critchley - validate parameters
    if ($colno > $ttlitems) { $colno = '1'; }
    $colno--;                             # remove one from colno to deal with arrays starting at 0
    if ($colno < 0) { $colno = '0'; }
    if   ($alpnum ne '') { $alpnum = lc($alpnum); }
    else                 { $alpnum = 'a'; }
    if   ($srtdir ne '') { $srtdir = lc($srtdir); }
    else                 { $srtdir = 'src'; }

    foreach $line (@tobesorted) {
        chomp($line);
        if ($line ne '') {
            my @temp = split(/\,/, $line);

            # Darren Critchley - juggle the fields so that the one we want to sort on is first
            my $tmpholder = $temp[0];
            $temp[0]      = $temp[$colno];
            $temp[$colno] = $tmpholder;
            $newline      = "";
            for ($ctr = 0; $ctr < $ttlitems; $ctr++) {
                $newline = $newline . $temp[$ctr] . ",";
            }
            chop($newline);
            push(@tmparray, $newline);
        }
    }
    if ($alpnum eq 'n') {
        @tmparray = sort { $a <=> $b } @tmparray;
    }
    else {
        @tmparray = (sort @tmparray);
    }
    foreach $line (@tmparray) {
        chomp($line);
        if ($line ne '') {
            my @temp = split(/\,/, $line);
            my $tmpholder = $temp[0];
            $temp[0]      = $temp[$colno];
            $temp[$colno] = $tmpholder;
            $newline      = "";
            for ($ctr = 0; $ctr < $ttlitems; $ctr++) {
                $newline = $newline . $temp[$ctr] . ",";
            }
            chop($newline);
            push(@srtedarray, $newline);
        }
    }

    if ($srtdir eq 'dsc') {
        @tmparray = reverse(@srtedarray);
        return (@tmparray);
    }
    else {
        return (@srtedarray);
    }
}

##
# Sort Hash Arrays
sub sortHashArray
{
    my ($col, $alpnum, $srtdir, $tobesortedRef) = @_;

    my @tobesortedKeys = keys %$tobesortedRef;
    my @tmparray;
    my @srtedarray;
    my $ttlitems = scalar @tobesortedKeys;    # want to know the number of rows in the passed array
    if ($ttlitems < 1) {                      # if no items, don't waste our time lets leave
        return (@tobesortedKeys);
    }

    # if column is not defined in Hash, return the sorted keys
    unless (defined($tobesortedRef->{$tobesortedKeys[0]}{$col})) {
        if ($srtdir eq 'dsc') {
            return reverse sort(@tobesortedKeys);
        }
        else {
            return sort(@tobesortedKeys);
        }
    }

    # validate parameters
    if   ($alpnum ne '') { $alpnum = lc($alpnum); }
    else                 { $alpnum = 'a'; }
    if   ($srtdir ne '') { $srtdir = lc($srtdir); }
    else                 { $srtdir = 'asc'; }

    if ($alpnum eq 'n') {

        # Use first numbers of the entry (e.g. for Portrange "1024:65535" it is sorted by "1024")
        @srtedarray = sort {
            (my $first = $tobesortedRef->{$a}{$col} =~ /^(\d+)/ ? $1 : 0)
                <=> (my $second = $tobesortedRef->{$b}{$col} =~ /^(\d+)/ ? $1 : 0)
        } @tobesortedKeys;
    }
    else {
        @srtedarray = sort { $tobesortedRef->{$a}{$col} cmp $tobesortedRef->{$b}{$col} } @tobesortedKeys;
    }

    if ($srtdir eq 'dsc') {
        @tmparray = reverse(@srtedarray);
        return (@tmparray);
    }
    else {
        return (@srtedarray);
    }
}

sub FetchPublicIp
{
    my %proxysettings;
    &General::readhash("${General::swroot}/proxy/settings", \%proxysettings);
    if ($_ = $proxysettings{'UPSTREAM_PROXY'}) {
        my ($peer, $peerport) = (
/^(?:[a-zA-Z ]+\:\/\/)?(?:[A-Za-z0-9\_\.\-]*?(?:\:[A-Za-z0-9\_\.\-]*?)?\@)?([a-zA-Z0-9\.\_\-]*?)(?:\:([0-9]{1,5}))?(?:\/.*?)?$/
        );
        Net::SSLeay::set_proxy($peer, $peerport, $proxysettings{'UPSTREAM_USER'}, $proxysettings{'UPSTREAM_PASSWORD'});
    }
    my ($out, $response) =
        Net::SSLeay::get_http('checkip.dyndns.org', 80, "/", Net::SSLeay::make_headers('User-Agent' => 'Ipcop'));
    if ($response =~ m%HTTP/1\.. 200 OK%) {
        $out =~ /Current IP Address: (\d+.\d+.\d+.\d+)/;
        return $1;
    }
    return '';
}

#
# Check if hostname.domain provided have IP provided
# use gethostbyname to verify that
# Params:
#	IP
#	hostname
#	domain
# Output
#	1 IP matches host.domain
#	0 not in sync
#
sub DyndnsServiceSync ($;$;$)
{

    my ($ip, $hostName, $domain) = @_;
    my @addresses;

    #fix me no ip GROUP, what is the name ?
    $hostName =~ s/$General::noipprefix//;
    if ($hostName) {    #may be empty
        $hostName  = "$hostName.$domain";
        @addresses = gethostbyname($hostName);
    }

    if ($addresses[0] eq '') {    # nothing returned ?
        $hostName  = $domain;                    # try resolving with domain only
        @addresses = gethostbyname($hostName);
    }

    if ($addresses[0] ne '') {                   # got something ?
                                                 #&General::log("name:$addresses[0], alias:$addresses[1]");
                                                 # Build clear text list of IP
        @addresses = map (&Socket::inet_ntoa($_), @addresses[ 4 .. $#addresses ]);
        if ($ip eq $addresses[0]) {
            return 1;
        }
    }
    return 0;
}

#
# This sub returns the red IP used to compare in DyndnsServiceSync
#
sub GetDyndnsRedIP
{
    my %settings;
    &General::readhash("${General::swroot}/ddns/settings", \%settings);

    open(IP, "${General::swroot}/red/local-ipaddress") or return 'unavailable';
    my $ip = <IP>;
    close(IP);
    chomp $ip;

    if (   &General::IpInSubnet($ip, '10.0.0.0', '255.0.0.0')
        || &General::IpInSubnet($ip, '172.16.0.0.', '255.240.0.0')
        || &General::IpInSubnet($ip, '192.168.0.0', '255.255.0.0'))
    {
        if ($settings{'BEHINDROUTER'} eq 'FETCH_IP') {
            my $RealIP = &General::FetchPublicIp;
            $ip = (&General::validip($RealIP) ? $RealIP : 'unavailable');
        }
    }
    return $ip;
}

sub isrunning($)
{
    my $cmd     = $_[0];
    my $status  = "<td>&nbsp;</td><th align='center' class='ipcop_stopped'>$Lang::tr{'stopped'}</th>";
    my $pid     = '';
    my $testcmd = '';
    my $exename;
    my $vmsize;

    $cmd =~ /(^[a-z]+)/;
    $exename = $1;

    if ($exename eq 'dhcpd') {
        # Special case for dnsmasq as DHCP server. dnsmasq is both DNS proxy and DHCP server, we want to
        # show both status. For DHCP we check if DHCP is enabled on at least 1 interface first.
        my @INTERFACEs = ('GREEN', 'BLUE');
        my %dhcpsettings = ();
        my $counter = 0;
        
        &General::readhash('/var/ipcop/dhcp/settings', \%dhcpsettings);
        foreach my $interface (@INTERFACEs) {
            for (my $i = 1; $i <= 1; $i++) {
                if ($dhcpsettings{"ENABLED_${interface}_${i}"} eq "on") {
                    $counter++;
                }
            }
        }

        return $status if ($counter == 0);

        $cmd = 'dnsmasq/dnsmasq';
        $exename = 'dnsmasq';
    }

    if (open(FILE, "/var/run/${cmd}.pid")) {
        $pid = <FILE>;
        chomp $pid;
        close FILE;
        if (open(FILE, "/proc/${pid}/status")) {
            while (<FILE>) {
                if (/^Name:\W+(.*)/)            { $testcmd = $1; }
                if (/^VmSize:\W+((\d+) \w{2})/) { $vmsize  = $1; }
            }
            close FILE;
            if ($testcmd =~ /$exename/) {
                $status = "<td align='center' >$vmsize</td>";
                $status .= "<th align='center' class='ipcop_running'>$Lang::tr{'running'}</th>";
            }
        }
    }
    return $status;
}

# Download some file
# If no URL parameter is passed, fetch update information from ipcop.org
sub download
{
    unless (-e "/var/ipcop/red/active") {
        return 0;
    }
    my $URL;
    if (@_ > 0) {
        $URL = $_;
    }
    else {
        $URL = "http://www.ipcop.org/patches/${General::version}";
    }

    my $downloader = LWP::UserAgent->new;
    $downloader->timeout(5);

    my %proxysettings = ();
    &General::readhash("/var/ipcop/proxy/settings", \%proxysettings);

    if ($_ = $proxysettings{'UPSTREAM_PROXY'}) {
        my ($peer, $peerport) = (
/^(?:[a-zA-Z ]+\:\/\/)?(?:[A-Za-z0-9\_\.\-]*?(?:\:[A-Za-z0-9\_\.\-]*?)?\@)?([a-zA-Z0-9\.\_\-]*?)(?:\:([0-9]{1,5}))?(?:\/.*?)?$/
        );
        if ($proxysettings{'UPSTREAM_USER'}) {
            $downloader->proxy("http",
                "http://$proxysettings{'UPSTREAM_USER'}:$proxysettings{'UPSTREAM_PASSWORD'}@" . "$peer:$peerport/");
        }
        else {
            $downloader->proxy("http", "http://$peer:$peerport/");
        }
    }
    return $downloader->get($URL, 'Cache-Control', 'no-cache');
}

# Translate ICMP code to text
# ref: http://www.iana.org/assignments/icmp-parameters
sub GetIcmpDescription ($)
{
    my $index            = shift;
    my @icmp_description = (
        'Echo Reply',    #0
        'Unassigned',
        'Unassigned',
        'Destination Unreachable',
        'Source Quench',
        'Redirect',
        'Alternate Host Address',
        'Unassigned',
        'Echo',
        'Router Advertisement',
        'Router Solicitation',    #10
        'Time Exceeded',
        'Parameter Problem',
        'Timestamp',
        'Timestamp Reply',
        'Information Request',
        'Information Reply',
        'Address Mask Request',
        'Address Mask Reply',
        'Reserved (for Security)',
        'Reserved (for Robustness Experiment)',    #20
        'Reserved',
        'Reserved',
        'Reserved',
        'Reserved',
        'Reserved',
        'Reserved',
        'Reserved',
        'Reserved',
        'Reserved',
        'Traceroute',                              #30
        'Datagram Conversion Error',
        'Mobile Host Redirect',
        'IPv6 Where-Are-You',
        'IPv6 I-Am-Here',
        'Mobile Registration Request',
        'Mobile Registration Reply',
        'Domain Name Request',
        'Domain Name Reply',
        'SKIP',
        'Photur',                                  #40
        'Experimental'
    );
    if   ($index > 41) { return 'unknown' }
    else               { return @icmp_description[$index] }
}

#
# Sorting of allocated leases
#
sub CheckSortOrder
{
    my %dhcpsettings = ();
    &General::readhash("${General::swroot}/dhcp/settings", \%dhcpsettings);

    if ($ENV{'QUERY_STRING'} =~ /^IPADDR|^ETHER|^HOSTNAME|^ENDTIME/) {
        my $newsort = $ENV{'QUERY_STRING'};
        my $act     = $dhcpsettings{'SORT_LEASELIST'};

        #Default sort if unspecified
        $act = 'IPADDRRev' if !defined($act);

        #Reverse actual ?
        if ($act =~ $newsort) {
            my $Rev = '';
            if ($act !~ 'Rev') { $Rev = 'Rev' }
            $newsort .= $Rev;
        }

        $dhcpsettings{'SORT_LEASELIST'} = $newsort;
        &General::writehash("${General::swroot}/dhcp/settings", \%dhcpsettings);
    }
}

#
# Arg: a list of ACTIONS button names to create at bottom of the page.
# These buttons are supposed to define ACTIONS to apply on each checked line.
#
sub PrintActualLeases
{
    our %dhcpsettings = ();
    our %entries      = ();

    sub leasesort
    {
        my $qs = '';
        if (rindex($dhcpsettings{'SORT_LEASELIST'}, 'Rev') != -1) {
            $qs = substr($dhcpsettings{'SORT_LEASELIST'}, 0, length($dhcpsettings{'SORT_LEASELIST'}) - 3);
            if ($qs eq 'IPADDR') {
                my @a = split(/\./, $entries{$a}->{$qs});
                my @b = split(/\./, $entries{$b}->{$qs});
                       ($b[0] <=> $a[0])
                    || ($b[1] <=> $a[1])
                    || ($b[2] <=> $a[2])
                    || ($b[3] <=> $a[3]);
            }
            else {
                $entries{$b}->{$qs} cmp $entries{$a}->{$qs};
            }
        }
        else    #not reverse
        {
            $qs = $dhcpsettings{'SORT_LEASELIST'};
            if ($qs eq 'IPADDR') {
                my @a = split(/\./, $entries{$a}->{$qs});
                my @b = split(/\./, $entries{$b}->{$qs});
                       ($a[0] <=> $b[0])
                    || ($a[1] <=> $b[1])
                    || ($a[2] <=> $b[2])
                    || ($a[3] <=> $b[3]);
            }
            else {
                $entries{$a}->{$qs} cmp $entries{$b}->{$qs};
            }
        }
    }

    my $buttonlist = '';
    map ($buttonlist .= "<input type='submit' name='ACTION_ALL' value='$_' />", split(',', shift || ''));

    my ($ip, $endtime, $ether, $hostname, @record, $record);
    open(LEASES, "/var/run/dnsmasq/dnsmasq.leases");
    while (my $line = <LEASES>) {
        next if ($line =~ /^\s*#/);
        chomp($line);
        my @temp = split(' ', $line);

        @record = ('IPADDR', $temp[2], 'ENDTIME', $temp[0], 'ETHER', $temp[1], 'HOSTNAME', $temp[3]);
        $record = {};    # create a reference to empty hash
        %{$record} = @record;    # populate that hash with @record
        $entries{$record->{'IPADDR'}} = $record;    # add this to a hash of hashes
    }
    close(LEASES);

    #Get sort method
    $dhcpsettings{'SORT_LEASELIST'} = 'IPADDR';     #default
    &General::readhash("${General::swroot}/dhcp/settings", \%dhcpsettings);    #or maybe saved !

    # Add visual indicators to column headings to show sort order - EO
    my ($a1, $a2, $a3, $a4) = '';

    if ($dhcpsettings{'SORT_LEASELIST'} eq 'ETHERRev') {
        $a1 = $Header::sortdn;
    }
    elsif ($dhcpsettings{'SORT_LEASELIST'} eq 'ETHER') {
        $a1 = $Header::sortup;
    }
    elsif ($dhcpsettings{'SORT_LEASELIST'} eq 'IPADDRRev') {
        $a2 = $Header::sortdn;
    }
    elsif ($dhcpsettings{'SORT_LEASELIST'} eq 'IPADDR') {
        $a2 = $Header::sortup;
    }
    elsif ($dhcpsettings{'SORT_LEASELIST'} eq 'HOSTNAMERev') {
        $a3 = $Header::sortdn;
    }
    elsif ($dhcpsettings{'SORT_LEASELIST'} eq 'HOSTNAME') {
        $a3 = $Header::sortup;
    }
    elsif ($dhcpsettings{'SORT_LEASELIST'} eq 'ENDTIMERev') {
        $a4 = $Header::sortdn;
    }
    elsif ($dhcpsettings{'SORT_LEASELIST'} eq 'ENDTIME') {
        $a4 = $Header::sortup;
    }

    &Header::openbox('100%', 'left', "$Lang::tr{'current dynamic leases'}:");
    print "<form method='post' action='$ENV{'SCRIPT_NAME'}'>" if ($buttonlist);
    print "<table width='100%'>";
    print "<tr>";
    print "<td></td>" if ($buttonlist);    # a new column for checkboxes
    print <<END
<td width='25%' align='center'><a href='$ENV{'SCRIPT_NAME'}?ETHER'><b>$Lang::tr{'mac address'}</b></a> $a1</td>
<td width='25%' align='center'><a href='$ENV{'SCRIPT_NAME'}?IPADDR'><b>$Lang::tr{'ip address'}</b></a> $a2</td>
<td width='20%' align='center'><a href='$ENV{'SCRIPT_NAME'}?HOSTNAME'><b>$Lang::tr{'hostname'}</b></a> $a3</td>
<td width='30%' align='center'><a href='$ENV{'SCRIPT_NAME'}?ENDTIME'><b>$Lang::tr{'lease expires'} (local time d/m/y)</b></a> $a4</td>
</tr>
END
        ;

    my $id = 0;
    foreach my $key (sort leasesort keys %entries) {

        my $hostname = &Header::cleanhtml($entries{$key}->{HOSTNAME}, "y");

        if ($id % 2) {
            print "<tr bgcolor='$Header::table1colour'>";
        }
        else {
            print "<tr bgcolor='$Header::table2colour'>";
        }

        print "<td><input type='checkbox' name='$entries{$key}->{IPADDR}!$entries{$key}->{ETHER}!$hostname' /></td>"
            if ($buttonlist);
        print <<END
<td align='center'>$entries{$key}->{ETHER}</td>
<td align='center'>$entries{$key}->{IPADDR}</td>
<td align='center'>&nbsp;$hostname </td>
<td align='center'>
END
            ;

        if ($entries{$key}->{ENDTIME} eq 'never') {
            print "$Lang::tr{'no time limit'}";
        }
        else {
            my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $dst);
            ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $dst) = localtime($entries{$key}->{ENDTIME});
            my $enddate = sprintf("%02d/%02d/%d %02d:%02d:%02d", $mday, $mon + 1, $year + 1900, $hour, $min, $sec);

            if ($entries{$key}->{ENDTIME} < time()) {
                print "<strike>$enddate</strike>";
            }
            else {
                print "$enddate";
            }
        }
        print "</td></tr>";
        $id++;
    }

    print "</table>";

    print "&nbsp;&nbsp;<img src='/images/vert-horiz.png' alt='' />$buttonlist</form>"
        if ($buttonlist);    # Issue a default command button
    &Header::closebox();
}

sub speedtouchversion
{
    my $speedtouch;
    if (-f "/proc/bus/usb/devices") {
        $speedtouch = `/bin/cat /proc/bus/usb/devices | /bin/grep 'Vendor=06b9 ProdID=4061' | /usr/bin/cut -d ' ' -f6`;
        if ($speedtouch eq '') {
            $speedtouch = $Lang::tr{'connect the modem'};
        }
    }
    else {
        $speedtouch = 'USB ' . $Lang::tr{'not running'};
    }
    return $speedtouch;
}

#
# Make a link from the selected profile to the "default" one.
# And update the secrets file.
#
sub SelectProfile
{
    my $profilenr = shift;
    our %modemsettings = ();
    our %pppsettings = ();

    die "No such profile: ${profilenr}" unless(-e "/var/ipcop/ppp/settings-${profilenr}");

    unlink('/var/ipcop/ppp/settings');
    link("/var/ipcop/ppp/settings-${profilenr}", '/var/ipcop/ppp/settings');
    system('/usr/bin/touch', '/var/ipcop/ppp/updatesettings');

    if ($pppsettings{'TYPE'} eq 'eagleusbadsl') {

        # eagle-usb.conf is in backup but link DSPcode.bin can't, so the link is created in rc.eagleusbadsl
        open(FILE, ">//var/ipcop/eagle-usb/eagle-usb.conf") or die "Unable to write eagle-usb.conf file";
        flock(FILE, 2);

        # decimal to hexa
        $modemsettings{'VPI'} = uc(sprintf('%X', $pppsettings{'VPI'}));
        $modemsettings{'VCI'} = uc(sprintf('%X', $pppsettings{'VCI'}));
        if ($pppsettings{'PROTOCOL'} eq 'RFC1483') {
            $modemsettings{'Encapsulation'} = 1 + $pppsettings{'ENCAP'};
        }
        elsif ($pppsettings{'PROTOCOL'} eq 'RFC2364') {
            $modemsettings{'Encapsulation'} = 6 - $pppsettings{'ENCAP'};
        }
        print FILE "<eaglectrl>\n";
        print FILE "VPI=$modemsettings{'VPI'}\n";
        print FILE "VCI=$modemsettings{'VCI'}\n";
        print FILE "Encapsulation=$modemsettings{'Encapsulation'}\n";
        print FILE "Linetype=0A\n";
        print FILE "RatePollFreq=00000009\n";
        print FILE "</eaglectrl>\n";
        close FILE;
    }

    # Read pppsettings to be able to write username and password to secrets file
    &General::readhash("/var/ipcop/ppp/settings-${profilenr}", \%pppsettings);

    # Write secrets file
    open(FILE, ">/var/ipcop/ppp/secrets") or die "Unable to write secrets file.";
    flock(FILE, 2);
    print FILE "'$pppsettings{'USERNAME'}' * '$pppsettings{'PASSWORD'}'\n";
    chmod 0600, "/var/ipcop/ppp/secrets";
    close FILE;
}


sub color_devices()
{
    my @itfs = ('ORANGE', 'BLUE', 'GREEN', 'RED');
    my $output = shift;
    $output = &Header::cleanhtml($output, "y");
    my %netsettings = ();
    &General::readhash('/var/ipcop/ethernet/settings', \%netsettings);

    foreach my $itf (@itfs) {
        my $ColorName = '';
        my $lc_itf    = lc($itf);
        $ColorName = "${lc_itf}";    #dereference variable name...
        my $icount = $netsettings{"${itf}_COUNT"};
        while ($icount > 0) {
            my $dev = $netsettings{"${itf}_${icount}_DEV"};
            $output =~ s/\b$dev/<b><span class='ipcop_iface_$ColorName'>$dev<\/span><\/b>/g;
            $icount--;
        }
    }

    if (-e '/proc/net/ipsec_eroute') {
        $output =~ s/ipsec(\d*)/<b><span class='ipcop_iface_ipsec'>ipsec$1<\/span><\/b>/g ;
    }

    if (open(REDIFACE, '/var/ipcop/red/iface')) {
        my $lc_itf = 'red';
        my $reddev = <REDIFACE>;
        close(REDIFACE);
        chomp $reddev;
        $output =~ s/\b$reddev/<b><span class='ipcop_iface_red'>${reddev}<\/span><\/b>/g;
    }

    return $output;
}

###
### escape all characters not digit-letter eg: frank&ipcop => franck\&ipcop
###
sub escape_shell ($) {
    my $ret = shift;
    $ret =~ s/(\W)/\\$1/g;
    return $ret;
}

1;
