/*
 * hardware.c: Probing, Scanning, everything to find out what's there
 *
 * This file is part of the IPCop Firewall.
 *
 * IPCop is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * IPCop is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with IPCop; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * (c) 2007-2009, the IPCop team
 *
 * $Id: hardware.c 2851 2009-05-13 19:02:56Z owes $
 * 
 */


#include <discover/discover-conf.h>

#include <dirent.h>
#include <newt.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "common.h"
#include "common_newt.h"
#include "arch_defs.h"


// List of all interesting hardware items
struct hardware_s *hardwares;
// How many did we find
unsigned int numhardwares;
unsigned int numharddisk;
unsigned int numcdrom;
unsigned int numnetwork;


static FILE *fhwdetect;
static int have_idedisk = 0;
static int have_scsidisk = 0;   /* Also for SATA controller */
static int have_idecd = 0;
static int install_setup = 0;   /* 0 when running setup (NIC detection), 1 for installer */


/* Retrieve disk capacity from proc or sys.
   Size is reported in blocks of 512 bytes, make a string with info GiB, MiB, KiB.
   Return size in MiB.
*/
static unsigned long getdrivesize(char *procname, char *strsize)
{
    FILE *f;
    uint64_t size;              /* using 32bit would limit to 1000 GB, which is not too far away in the future */
    unsigned long mbsize;

    mbsize = 0;
    strcpy(strsize, "? KiB");

    if ((f = fopen(procname, "r")) != NULL) {
        if (fgets(strsize, STRING_SIZE, f)) {
            size = strtoull(strsize, NULL, 10);
            size = size * 512 / 1024;   /* do not use the disk vendor way of specifying KBytes */
            mbsize = size / 1024;

            if (size >= 4000000) {
                /* Everything larger than 4000 MB */
                snprintf(strsize, STRING_SIZE, "%llu GiB", size / (1024*1024));
            }
            else if (size >= 4000) {
                /* Everything larger than 4000 KB */
                snprintf(strsize, STRING_SIZE, "%llu MiB", size / 1024);
            }
            else {
                /* Anything else, unlikely since we need something like 100+ MB anyway */
                snprintf(strsize, STRING_SIZE, "%llu KiB", size);
            }
        }
    }

    return mbsize;
}


/* Add something to our hardware list */
static void hardwareadd(supported_media_t type, char *module, char *device, char *vendor, char *description,
                        char *vendorid, char *modelid)
{
    char vendordesc[STRING_SIZE] = "";

    if ((device != NULL) && (numhardwares != 0)) {
        /* avoid duplicates */
        int i;

        for (i = 0; i < numhardwares; i++) {
            if (!strcmp(device, hardwares[i].device)) {
                return;
            }
        }
    }

    hardwares = realloc(hardwares, sizeof(struct hardware_s) * (numhardwares + 1));

    hardwares[numhardwares].type = type;

    if (module != NULL) {
        hardwares[numhardwares].module = strdup(module);
    }
    else {
        hardwares[numhardwares].module = strdup("");
    }

    if (device != NULL) {
        hardwares[numhardwares].device = strdup(device);
    }
    else {
        hardwares[numhardwares].device = strdup("");
    }

    // Now build full description from vendor name and description
    // if description is NULL, full description becomes "Unknown" + IDs
    if (description != NULL) {
        if (vendor != NULL) {
            strcpy(vendordesc, vendor);
            strcat(vendordesc, " ");
        }
        strcat(vendordesc, description);
    }
    else {
        /* no description, use IDs */
        snprintf(vendordesc, STRING_SIZE, "Unknown %s:%s",
                 vendorid == NULL ? "" : vendorid, modelid == NULL ? "" : modelid);
    }
    hardwares[numhardwares].description = strdup(vendordesc);

    fprintf(flog, "  HWadd %3d, %s, %s, %s\n", type, hardwares[numhardwares].module,
            hardwares[numhardwares].device, hardwares[numhardwares].description);
    if (install_setup) {
        fprintf(fhwdetect, "  HWadd %3d, %s, %s, %s\n", type, hardwares[numhardwares].module,
                hardwares[numhardwares].device, hardwares[numhardwares].description);
    }
    // increment tallies
    numhardwares++;
    switch (type) {
    case network:
        numnetwork++;
        break;
    case harddisk:
        numharddisk++;
        break;
    case cdrom:
        numcdrom++;
        break;
    default:
        break;
    }
}


/* 
    Filter function for scanning /sys/bus/ide/devices
    Used by scandir in scanprocdrives function.
    Return 0 to skip a device
*/
int ide_filter(const struct dirent *b)
{
    char string[STRING_SIZE];

    snprintf(string, STRING_SIZE, "/sys/bus/ide/devices/%s/media", b->d_name);

    if (access(string, 0) == 0) {
        return 1;
    }

    return 0;
}


/* Scan /sys/bus/ide and /sys/block for drives */
static void scanprocdrives(int modprobe)
{
    FILE *f = NULL;
    char procname[STRING_SIZE];
    char media[STRING_SIZE];
    char model[STRING_SIZE];
    supported_media_t type = none;
    char command[STRING_SIZE];
    char deviceletter;
    char strsize[STRING_SIZE];
    struct dirent **names;
    int numdevices = 0;
    int i;

    /* look for IDE harddisk and cdrom */
    numdevices = scandir("/sys/bus/ide/devices", &names, &ide_filter, alphasort);
    for (i = 0; i < numdevices; i++) {
        snprintf(procname, STRING_SIZE, "/sys/bus/ide/devices/%s/media", names[i]->d_name);
        if ((f = fopen(procname, "r")) == NULL) {
            continue;
        }

        /* media holds disk or cdrom */
        if (fgets(media, STRING_SIZE, f)) {
            stripnl(media);
            if (!strcmp(media, "disk")) {
                type = harddisk;
            }
            else if (!strcmp(media, "cdrom")) {
                type = cdrom;
                if (modprobe && !have_idecd) {
                    have_idecd = 1;
                    /* Since kernel 2.6.25 ide-cd is ide-cd_mod */
                    snprintf(command, STRING_SIZE, "/sbin/modprobe %s", "ide-cd_mod");
                    mysystem(command);
                    /* no need to put in hardware list */

                    /* give some time to settle */
                    sleep(1);
                }
            }
        }
        fclose(f);

        if (type != none) {
            char device[STRING_SIZE];
            char description[STRING_SIZE] = "Unknown";

            /* found something, get device name (hd?), model and size */
            snprintf(procname, STRING_SIZE, "/sys/bus/ide/devices/%s/drivename", names[i]->d_name);
            if ((f = fopen(procname, "r")) != NULL) {
                if (fgets(device, STRING_SIZE, f)) {
                    stripnl(device);
                    fclose(f);

                    snprintf(procname, STRING_SIZE, "/sys/bus/ide/devices/%s/model", names[i]->d_name);
                    if ((f = fopen(procname, "r")) != NULL) {
                        if (fgets(model, STRING_SIZE, f)) {
                            stripnl(model);
                            if (type == harddisk) {
                                snprintf(procname, STRING_SIZE, "/sys/block/%s/size", device);
                                getdrivesize(procname, strsize);

                                snprintf(description, STRING_SIZE, "%-30.30s (%s)", model, strsize);
                            }
                            else {
                                /* size is not interesting for CDROM */
                                strcpy(description, model);
                            }
                        }
                    }
                }
                fclose(f);
            }

            hardwareadd(type, NULL, device, NULL, description, NULL, NULL);
        }
    }


    /* Look for SCSI, SATA harddisk, USB attached devices */
    for (deviceletter = 'a'; deviceletter <= 'z'; deviceletter++) {
        snprintf(procname, STRING_SIZE, "/sys/block/sd%c/device/model", deviceletter);

        if ((f = fopen(procname, "r")) == NULL) {
            continue;
        }

        /* We need some mechanism to differentiate between installation from USB stick
           or installation on USB device
         */

        if (fgets(model, STRING_SIZE, f)) {
            char device[4];
            char description[STRING_SIZE] = "Unknown";

            stripnl(model);
            sprintf(device, "sd%c", deviceletter);

            snprintf(procname, STRING_SIZE, "/sys/block/sd%c/size", deviceletter);
            if (getdrivesize(procname, strsize) < 64) {
                /* Silently discard anything too small */
                fprintf(flog, "   discard sd%c %-30.30s (%s)\n", deviceletter, model, strsize);
                if (install_setup) {
                    fprintf(fhwdetect, "   discard sd%c %-30.30s (%s)\n", deviceletter, model, strsize);
                }
                continue;
            }
            snprintf(description, STRING_SIZE, "%-30.30s (%s)", model, strsize);

            hardwareadd(harddisk, NULL, device, NULL, description, NULL, NULL);
        }

        fclose(f);
    }

    /* Look for SCSI, SATA, USB cdrom */
    for (deviceletter = '0'; deviceletter <= '9'; deviceletter++) {
        snprintf(procname, STRING_SIZE, "/sys/block/sr%c/device/model", deviceletter);

        if ((f = fopen(procname, "r")) == NULL) {
            continue;
        }

        if (fgets(model, STRING_SIZE, f)) {
            char device[4];

            stripnl(model);
            sprintf(device, "sr%c", deviceletter);
            hardwareadd(cdrom, NULL, device, NULL, model, NULL, NULL);
        }

        fclose(f);
    }
}


/* fill in tables with data */
void scan_hardware(int flag_i_s, int nopcmcia, int nousb, int noscsi)
{
    discover_bus_map_t *busmap;
    discover_error_t *status;
    discover_device_t *devices;
    char command[STRING_SIZE];
    int i;
    newtComponent form;
    newtComponent text;
    newtComponent scale;
    char line[STRING_SIZE];
    char pcimap[STRING_SIZE];
    int numBusses;
    int firstscan;
    int addmodules;

    numhardwares = 0;
    numharddisk = 0;
    numcdrom = 0;
    numnetwork = 0;
    hardwareadd(none, NULL, NULL, NULL, NULL, NULL, NULL);

    status = discover_error_new();

    install_setup = flag_i_s;
    if (install_setup) {
        /* also write HW detection to file, for easier reference */
        fhwdetect = fopen("/tmp/hwdetect", "w");
    }

    numBusses = BUS_COUNT;
    /* disable stuff the user does not want or does not need */
    if (!install_setup) {
        busmap = discover_conf_get_bus_map_by_name("ata", status);
        busmap->scan_never = 1;
        numBusses--;
    }
    if (nopcmcia) {
        busmap = discover_conf_get_bus_map_by_name("pcmcia", status);
        busmap->scan_never = 1;
        numBusses--;
    }
    if (nousb) {
        busmap = discover_conf_get_bus_map_by_name("usb", status);
        busmap->scan_never = 1;
        numBusses--;
    }
    if (noscsi || !install_setup) {
        busmap = discover_conf_get_bus_map_by_name("scsi", status);
        busmap->scan_never = 1;
        numBusses--;
    }

    snprintf(line, STRING_SIZE, ipcop_gettext("TR_SCANNING_HARDWARE"), "");
    text = newtLabel(1, 1, line);
    scale = newtScale(1, 3, 70, numBusses * 10);
    newtCenteredWindow(72, 5, ipcop_gettext("TR_TITLE_HARDWARE"));
    form = newtForm(NULL, NULL, 0);
    newtFormAddComponents(form, text, scale, NULL);

    newtDrawForm(form);
    newtRefresh();

    /* start bus scan */
    busmap = discover_conf_get_full_bus_map(status);
    for (i = 0; busmap[i].name; i++) {
        snprintf(line, STRING_SIZE, ipcop_gettext("TR_SCANNING_HARDWARE"), busmap[i].name);
        strcat(line, "           ");
        newtLabelSetText(text, line);
        newtRefresh();

        if (busmap[i].scan_never) {
            fprintf(flog, "Noscan %s\n", busmap[i].name);
            if (install_setup) {
                fprintf(fhwdetect, "Noscan %s\n", busmap[i].name);
            }
            newtScaleSet(scale, (i + 1) * 10);
            newtRefresh();
            continue;
        }

        fprintf(flog, "Scan %s\n", busmap[i].name);
        if (install_setup) {
            fprintf(fhwdetect, "Scan %s\n", busmap[i].name);
        }
        devices = discover_get_devices(i, status);

        newtScaleSet(scale, i * 10 + 1);
        newtRefresh();

        /* walk list of devices */
        for (; devices; devices = discover_device_get_next(devices)) {
            char *busclass;
            char *module;
            char *description;
            supported_media_t type = none;

            if ((busclass = discover_device_get_busclass(devices)) == NULL)
                continue;

            module = discover_device_get_data(devices, "linux/module/name", utsbuf.release, status);
            description = discover_device_get_model_name(devices);

            /*  Busclass found, now go hunt for NIC, mass storage and PCMCIA/Cardbus.
                We'll leave out some special hardware.
                Info about PCI classes can be found here: http://pci-ids.ucw.cz/read/PD/
            */
            if (!strcmp(busclass, "0200") ||    /* Ethernet */
                !strcmp(busclass, "0201") ||    /* Token Ring */
//                !strcmp(busclass, "0202") ||    /* FDDI */
//                !strcmp(busclass, "0203") ||    /* ATM */
//                !strcmp(busclass, "0204") ||    /* ISDN */
//                !strcmp(busclass, "0205") ||    /* WorldFip */
//                !strcmp(busclass, "0206") ||    /* PICMG */
                !strcmp(busclass, "0280")       /* Other, some ISDN and WLAN */
                ) {
                type = network;
            }
            else if (!strcmp(busclass, "0680")) {
                /* onboard forcedeth seems to identify as 0680 instead of 0200 */
                if (module && !strcmp(module, "forcedeth")) {
                    type = network;
                }
            }
            else if (install_setup && ((!strcmp(busclass, "0100") && !noscsi) ||        /* SCSI controller */
                                       !strcmp(busclass, "0101") ||     /* IDE controller */
//                                       !strcmp(busclass, "0102") ||     /* Floppy controller */
//                                       !strcmp(busclass, "0103") ||     /* IPI controller (anyone using this ?) */
                                       !strcmp(busclass, "0104") ||     /* RAID controller */
                                       !strcmp(busclass, "0105") ||     /* ATA controller */
                                       !strcmp(busclass, "0106") ||     /* SATA controller */
//                                       !strcmp(busclass, "0107") ||     /* Serial Attached SCSI controller */
                                       !strcmp(busclass, "0180")        /* Other mass storage controller */
                     )) {
                type = specialmodule;
            }
            else if (install_setup && ((!strcmp(busclass, "0605") && !nopcmcia) ||      /* PCMCIA bridge */
                                       (!strcmp(busclass, "0607") && !nopcmcia) /* Cardbus bridge */
                     )) {
                type = specialmodule;
            }
            else if (install_setup && ((!strcmp(busclass, "0c03") && !nousb)    /* USB controller */
                     )) {
                type = specialmodule;
                /* in some cases the modulename is empty ? */
            }

            /* 
               If we found something interesting, add to our module list.
               We probably do not really need IDE, SCSI etc. things in here, 
               since we will build initrd from lsmod output.
             */
            if (type == none) {
                /* owes: keep logging for now */
                snprintf(line, STRING_SIZE, "  Skip %s %s:%s, %s, %s\n",
                        busclass, discover_device_get_vendor_id(devices), discover_device_get_model_id(devices), 
                        module == NULL ? "---" : module, description == NULL ? "---" : description);
                fprintf(flog, line);
                if (install_setup) {
                    fprintf(fhwdetect, line);
                }
            }
            else {
                if ((module == NULL) && !strcmp(busmap[i].name, "pci")) {
                    /* This is a totally impossible hack, maybe the future will bring something better or different.
                       Anyway, if discover found a PCI device without naming a module, go hunting in pcimap.        
                     */
                    FILE *p;

                    snprintf(command, STRING_SIZE,
                             "/bin/grep \"%s 0x0000%s\" /lib/modules/%s/modules.pcimap",
                             discover_device_get_vendor_id(devices), discover_device_get_model_id(devices), utsbuf.release);

                    if (install_setup) {
                        fprintf(fhwdetect, "*****\n  Search 0x%s, 0x%s, 0x%s in pcimap\n", 
                            busclass, discover_device_get_vendor_id(devices), discover_device_get_model_id(devices));
                    }

                    p = popen(command, "r");
                    if (fgets(pcimap, STRING_SIZE, p)) {
                        if (install_setup) {
                            fprintf(fhwdetect, "%s", pcimap);
                        }
                        if ((module = strchr(pcimap, ' ')) != NULL) {
                            *module = 0;
                            module = pcimap;
                        }
                    }
                    pclose(p);
                    if (install_setup) {
                        fprintf(fhwdetect, "  After search in pcimap: %s\n*****\n", module == NULL ? "still unknown" : module);
                    }
                }

                snprintf(line, STRING_SIZE, "  Add  %s %s:%s, %s, %s\n",
                        busclass, discover_device_get_vendor_id(devices), discover_device_get_model_id(devices), 
                        module == NULL ? "---" : module, description == NULL ? "---" : description);
                fprintf(flog, line);
                if (install_setup) {
                    fprintf(fhwdetect, line);
                }
                hardwareadd(type, module, NULL, discover_device_get_vendor_name(devices), description,
                            discover_device_get_vendor_id(devices), discover_device_get_model_id(devices));
            }

            /* Some special handling for special devices */
            if (install_setup && (type == specialmodule)) {
                if (module != NULL) {
                    snprintf(command, STRING_SIZE, "/sbin/modprobe %s", module);
                    mysystem(command);
                }

                /* SCSI, SATA, add some modules */
                if ((!strcmp(busclass, "0100") || !strcmp(busclass, "0106")) && !have_scsidisk) {
                    have_scsidisk = 1;
                    snprintf(command, STRING_SIZE, "/sbin/modprobe %s", "sd_mod");
                    mysystem(command);
                    hardwareadd(specialmodule, "sd_mod", NULL, NULL, NULL, NULL, NULL);

                    snprintf(command, STRING_SIZE, "/sbin/modprobe %s", "sr_mod");
                    mysystem(command);
                    /* no need to put in hardware list */
                }

                /* IDE, add some modules */
                if (!strcmp(busclass, "0101") && !have_idedisk) {
                    have_idedisk = 1;
                    // snprintf(command, STRING_SIZE, "/sbin/modprobe %s", "ide-generic");
                    // mysystem(command);
                    // hardwareadd(specialmodule, "ide-generic", NULL, NULL, NULL, NULL, NULL);
                    snprintf(command, STRING_SIZE, "/sbin/modprobe %s", "ide-disk");
                    mysystem(command);
                    hardwareadd(specialmodule, "ide-disk", NULL, NULL, NULL, NULL, NULL);
                }

                /* USB, probe */
                if (!strcmp(busclass, "0c03")) {
                    /* 
                       Dependancies will load sd_mod, but sd_mod will not be in initramfs .
                       Not sure if installing onto USB stick is a good idea though.
                     */
                    snprintf(command, STRING_SIZE, "/sbin/modprobe %s", "sd_mod");
                    mysystem(command);
                    hardwareadd(specialmodule, "sd_mod", NULL, NULL, NULL, NULL, NULL);
                    snprintf(command, STRING_SIZE, "/sbin/modprobe %s", "usb-storage");
                    mysystem(command);
                    hardwareadd(specialmodule, "usb-storage", NULL, NULL, NULL, NULL, NULL);

                    /* need sr_mod when installing from USB CDROM */
                    snprintf(command, STRING_SIZE, "/sbin/modprobe %s", "sr_mod");
                    mysystem(command);
                    /* no need to put in hardware list */
                }
            }
        }

        newtScaleSet(scale, (i + 1) * 10);
        newtRefresh();
        /* not really necessary, but with our nice screen the user probably won't mind waiting a bit */
        sleep(1);
    }

    discover_error_free(status);
    newtFormDestroy(form);
    newtPopWindow();


    /* testing (not for NIC detection), eventually we may need some function like this */
    addmodules = 1;
    while (install_setup && addmodules) {
        newtComponent button_done, button_add;
        newtComponent module, moduleentry;
        struct newtExitStruct exitstruct;
        const char *modulename;
        int numLines;

        text = newtTextboxReflowed(1, 1, "Load and add a specific module.", 68, 0, 0, 0);
        numLines = newtTextboxGetNumLines(text);
        newtCenteredWindow(72, numLines + 10, ipcop_gettext("TR_TITLE_HARDWARE"));
        form = newtForm(NULL, NULL, 0);
        newtFormAddComponent(form, text);

        module = newtTextbox(2, numLines + 2, 25, 1, 0);
        newtTextboxSetText(module, "Module (without .ko):");
        newtFormAddComponent(form, module);
        moduleentry = newtEntry(12, numLines + 3, "", 20, &modulename, 0);
        newtFormAddComponent(form, moduleentry);

        button_add = newtButton(6, numLines + 5, ipcop_gettext("TR_OK"));
        button_done = newtButton(26, numLines + 5, ipcop_gettext("TR_SKIP"));
        newtFormAddComponents(form, button_add, button_done, NULL);

        newtRefresh();
        newtDrawForm(form);
        newtFormRun(form, &exitstruct);

        if (exitstruct.u.co == button_add) {
            snprintf(command, STRING_SIZE, "/sbin/modprobe %s", modulename);
            if (!mysystem(command)) {
                hardwareadd(specialmodule, (char *) modulename, NULL, NULL, NULL, NULL, NULL);
            }
            else {
                /* owes: show errorbox here */
            }
        }

        newtFormDestroy(form);
        newtPopWindow();

        if (exitstruct.u.co == button_done) {
            addmodules = 0;
        }
    }


    firstscan = TRUE;
    while (install_setup && (numharddisk == 0)) {
        if (install_setup) {
            fflush(fhwdetect);
        }
        statuswindow(72, 5, ipcop_gettext("TR_TITLE_HARDWARE"), ipcop_gettext("TR_SCANNING_HARDWARE"), "drives");

        if (!firstscan) {
            /* since we've not yet waited since last modprobe, sleep now */
            sleep(2);
        }
        firstscan = FALSE;

        /* go hunting for drives */
        fprintf(flog, "Scan for drives\n");
        if (install_setup) {
            fprintf(fhwdetect, "Scan for drives\n");
        }
        scanprocdrives(1);

        newtPopWindow();

        if (numharddisk == 0) {
            newtComponent button_rescan, button_modprobe, button_cancel;
            newtComponent module, moduleentry;
            struct newtExitStruct exitstruct;
            const char *modulename;
            int numLines;

            text = newtTextboxReflowed(1, 1, ipcop_gettext("TR_NO_HARDDISK_MODPROBE"), 68, 0, 0, 0);
            numLines = newtTextboxGetNumLines(text);
            newtCenteredWindow(72, numLines + 9, ipcop_gettext("TR_TITLE_HARDWARE"));
            form = newtForm(NULL, NULL, 0);
            newtFormAddComponent(form, text);

            module = newtTextbox(2, numLines + 2, 10, 1, 0);
            newtTextboxSetText(module, "Module:");
            newtFormAddComponent(form, module);
            moduleentry = newtEntry(12, numLines + 2, "", 20, &modulename, 0);
            newtFormAddComponent(form, moduleentry);

            button_rescan = newtButton(6, numLines + 4, ipcop_gettext("TR_RESCAN"));
            button_modprobe = newtButton(26, numLines + 4, "Modprobe");
            button_cancel = newtButton(46, numLines + 4, ipcop_gettext("TR_CANCEL"));
            newtFormAddComponents(form, button_rescan, button_modprobe, button_cancel, NULL);

            newtRefresh();
            newtDrawForm(form);
            newtFormRun(form, &exitstruct);

            if (exitstruct.u.co == button_modprobe) {
                snprintf(command, STRING_SIZE, "/sbin/modprobe %s", modulename);
                if (!mysystem(command)) {
                    hardwareadd(specialmodule, (char *) modulename, NULL, NULL, NULL, NULL, NULL);
                }
                else {
                    /* owes: show errorbox here */
                }
            }

            newtFormDestroy(form);
            newtPopWindow();

            if (exitstruct.u.co == button_cancel) {
                return;
            }
        }
    }


    fprintf(flog, "Scan complete. Hardware %d, Harddisk %d, CDROM %d, Network %d\n", numhardwares, numharddisk,
            numcdrom, numnetwork);
    if (install_setup) {
        fprintf(fhwdetect, "Scan complete. Hardware %d, Harddisk %d, CDROM %d, Network %d\n", numhardwares, numharddisk,
                numcdrom, numnetwork);
        fclose(fhwdetect);
    }
}
