#!/bin/bash
# vim: set columns=120 noexpandtab:
#
#########################################################################################################
#													#
# This file is part of the IPCop Firewall.								#
#													#
# IPCop is free software; you can redistribute it and/or modify						#
# it under the terms of the GNU General Public License as published by					#
# the Free Software Foundation; either version 2 of the License, or					#
# (at your option) any later version.									#
#													#
# IPCop is distributed in the hope that it will be useful,						#
# but WITHOUT ANY WARRANTY; without even the implied warranty of					#
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the						#
# GNU General Public License for more details.								#
#													#
# You should have received a copy of the GNU General Public License					#
# along with IPCop; if not, write to the Free Software							#
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA				#
#													#
# Copyright (C) 2001 Mark Wormgoor <mark@wormgoor.com>.							#
#													#
# (c) 2001 Eric S. Johansson <esj@harvee.billerica.ma.us> Check for Bash				#
# (c) 2002 Thorsten Fischer <frosch@cs.tu-berlin.de> MD5Sum checking					#
#													#
#########################################################################################################
#													#
# $Id: make.sh 2876 2009-05-20 09:24:35Z owes $
#########################################################################################################




#########################################################################################################
#########################################################################################################
# BLOCK 1 -- Variables											#
#########################################################################################################
#########################################################################################################


#########################################################################################################
# Some important variables -- these are not customizable						#
#########################################################################################################

# The official project name
NAME="IPCop"

# Just a short name for IPCop
SNAME="ipcop"

# This is the IPCop internal version number.  May or may not be the version released eventually.
VERSION=1.9.5

# This is the last official IPCop version number.  Meaningless in SVN HEAD.
#PREVIOUSTAG=IPCOP_v1_4_4_FINAL

# Just an arbitrary name for the downloadable, prebuilt toolchain (if you want to save time compiling).
TOOLCHAINVERSION=1.9.5

# A collection of all the external software needed to build, install, and run ipcop.  This is for GPL compliance.
OTHERSRC=${SNAME}-${VERSION}-othersrc.tar.bz2

# The official IPCop slogan
SLOGAN="The Bad Packets Stop Here"

# Where the ipcop specific config files will be installed (this is the path on a running ipcop system)
CONFIG_ROOT=/var/ipcop

# What's the kernel version we're building (this is not the host kernel)
KVER=`grep --max-count=1 VER lfs/linux | awk '{ print $3 }'`

# Let's see what's our host architecture
MACHINE=`uname -m`

# Debian specific settings
if [ ! -e /etc/debian_version ]; then
	FULLPATH=`which ${0}`
else
	if [ -x /usr/bin/realpath ]; then
		FULLPATH=`/usr/bin/realpath ${0}`
	else
		echo "ERROR: Need to do apt-get install realpath"
		exit 1
	fi
fi

PWD=`pwd`

# This is just a temporary variable to help us decide what our current working directory is
BASENAME=`basename ${0}`

# The directory where make.sh is.  Needed by every script
BASEDIR=`echo ${FULLPATH} | sed "s/\/${BASENAME}//g"`
export BASEDIR

# This is used to test source file download on the web with ./make.sh check (erase tags with ./make.sh checkclean)
DIR_CHK=${BASEDIR}/cache/check

# This is an optional .config file with variables overriding the default values
IPCOP_CONFIG=${BASEDIR}/.config

# Set up INSTALLER_DIR
INSTALLER_DIR=installer

# This is the LFS branch we're using.  Can be either stable or development
#LFS_BRANCH=stable
LFS_BRANCH=development

# This is the URL that lists the versions of packages for the LFS branch we are using as the basis for IPCop
LFS_PACKAGES_URL=http://www.linuxfromscratch.org/lfs/view/${LFS_BRANCH}/chapter03/packages.html

# HOST SYSTEM REQUIREMENTS
# IPCop-2.0 is based on LFS-6.3.
# Check http://www.linuxfromscratch.org/lfs/view/stable/prologue/hostreqs.html for a list of
# host system requirements.  The values below are based on LFS-6.3

# If you absolutely know what you're doing and want to ignore the prerequisites, set this variable to "yes"
# OVERRIDING THIS IS STRONGLY DISCOURAGED
IGNORE_PREREQUISITES=no

# NOTE!!!
# When building toolchain and base we require at least kernel 2.6.5.
# Nobody should be using a really really old kernel: http://lkml.org/lkml/2007/11/14/6
# The benefit (size and compile time) of increasing kernel compatiblity is small.
# A glibc compiled with --enable-kernel=2.6.5 is very unlikely to run on a system with kernel < 2.6.5
# If you have a kernel that is older than REQUIRED_KERNEL below, upgrade now!
REQUIRED_BASH=2.05a
REQUIRED_BINUTILS=2.12
REQUIRED_BISON=1.875
REQUIRED_BZIP2=1.0.2
REQUIRED_COREUTILS=5.0
REQUIRED_DIFFUTILS=2.8
REQUIRED_FINDUTILS=4.1.20
REQUIRED_GAWK=3.1.5
REQUIRED_GCC=3.0.1
REQUIRED_GLIBC=2.2.5
REQUIRED_GREP=2.5
REQUIRED_GZIP=1.2.4
REQUIRED_KERNEL=2.6.5
REQUIRED_MAKE=3.79.1
REQUIRED_PATCH=2.5.4
REQUIRED_SED=3.0.2
REQUIRED_TAR=1.14
REQUIRED_TEXINFO=4.8
# END OF HOST SYSTEM REQUIREMENTS

# This array holds the names of all customizable variables
declare -a CUSTOMIZABLE_VARIABLES

# This is the file that tells us there's a package building in progress
RUNNING_MARKER=${BASEDIR}/.running

# This is the file that tells us a package has failed to build
FAILED_MARKER=${BASEDIR}/.failed

# Store 'now' in YYYYMMDD-HHMMSS and in seconds since 1970 format
BUILDDATE=`date +"%Y%m%d-%H%M%S"`
BUILDSTART=`date +"%s"`

#########################################################################################################
# End of important non-customizable variables								#
#########################################################################################################


#########################################################################################################
# The following variables can be customized to fit your preferences					#
#########################################################################################################
END=no
COUNTER=0
until [ x"${END}" == x"yes" ];
do
	# How nice should we be?
	NICE=10 && CUSTOMIZABLE_VARIABLES[${COUNTER}]="NICE" && COUNTER=$[ ${COUNTER} + 1 ]

	# How many concurrent jobs to run (NOTE: Don't use too high a value, or packages may have timing issues.
	# At least openssh fails if this is too high.  It's safe to set this to number of CPUs + 1)
	PARALLELISM=3 && CUSTOMIZABLE_VARIABLES[${COUNTER}]="PARALLELISM" && COUNTER=$[ ${COUNTER} + 1 ]

	# How many times should prefetch try to download a package. Don't set to too high a value and hammer sites
	MAX_RETRIES=3 && CUSTOMIZABLE_VARIABLES[${COUNTER}]="MAX_RETRIES" && COUNTER=$[ ${COUNTER} + 1 ]

	# Make distcc optional, by default don't use it.  Override this in .config
	USE_DISTCC=no && CUSTOMIZABLE_VARIABLES[${COUNTER}]="USE_DISTCC" && COUNTER=$[ ${COUNTER} + 1 ]

	# To what servers should distcc distribute the load
	DISTCC_HOSTS=localhost && CUSTOMIZABLE_VARIABLES[${COUNTER}]="DISTCC_HOSTS" && COUNTER=$[ ${COUNTER} + 1 ]

	# Should we skip creation of USB images
	SKIP_USB_IMAGES=yes && CUSTOMIZABLE_VARIABLES[${COUNTER}]="SKIP_USB_IMAGES" && COUNTER=$[ ${COUNTER} + 1 ]

	# Should we skip creation of floppy images
	SKIP_FLOPPY_IMAGES=yes && CUSTOMIZABLE_VARIABLES[${COUNTER}]="SKIP_FLOPPY_IMAGES" && COUNTER=$[ ${COUNTER} + 1 ]

	# If you *absolutely* want to build ipcop as root, override this in .config. STRONGLY DISCOURAGED
	ALLOW_ROOT_TO_BUILD=no && CUSTOMIZABLE_VARIABLES[${COUNTER}]="ALLOW_ROOT_TO_BUILD" && COUNTER=$[ ${COUNTER} +1 ]

	# A timeout variable (in seconds) for when we need user input within a specified amount of time
	TIMEOUT=5 && CUSTOMIZABLE_VARIABLES[${COUNTER}]="TIMEOUT" && COUNTER=$[ ${COUNTER} +1 ]

	# Should make.sh show you a nice progress meter
	SHOW_PROGRESS=no && CUSTOMIZABLE_VARIABLES[${COUNTER}]="SHOW_PROGRESS" && COUNTER=$[ ${COUNTER} +1 ]

	# Some brave soul might want to skip the test for available diskspace
	SKIP_CHECK_DISKSPACE=no && CUSTOMIZABLE_VARIABLES[${COUNTER}]="SKIP_CHECK_DISKSPACE" && COUNTER=$[ ${COUNTER} +1 ]

    # Add new variables before this line
	END=yes
done
unset COUNTER
unset END

#########################################################################################################
# End of customizable variables										#
#########################################################################################################


#########################################################################################################
# Beautifying variables & presentation & input/output interface						#
#########################################################################################################

## Screen dimensions
# Find current screen size
if [ -z "${COLUMNS}" ]; then
	COLUMNS=$(stty size)
	COLUMNS=${COLUMNS##* }
fi

# When using remote connections, such as a serial port, stty size returns 0
if [ "${COLUMNS}" = "0" ]; then
	COLUMNS=80
fi

## Measurements for positioning result messages
RESULT_WIDTH=4
WGET_WIDTH=4
ARCH_WIDTH=5
FOUND_WIDTH=9
REQUIRED_WIDTH=10
TIME_WIDTH=9
OPT_WIDTH=6
VER_WIDTH=10

RESULT_COL=$((${COLUMNS} - ${RESULT_WIDTH} - 4))
WGET_COL=$((${RESULT_COL} - ${WGET_WIDTH} - 5))
ARCH_COL=$((${WGET_COL} - ${ARCH_WIDTH} - 5))
FOUND_COL=$((${RESULT_COL} - ${FOUND_WIDTH} - 5))
REQUIRED_COL=$((${FOUND_COL} - ${REQUIRED_WIDTH} - 5))
TIME_COL=$((${RESULT_COL} - ${TIME_WIDTH} - 5))
VER_COL=$((${TIME_COL} - ${VER_WIDTH} - 5))
OPT_COL=$((${VER_COL} - ${OPT_WIDTH} - 5))

## Set Cursor Position Commands, used via echo -e
SET_RESULT_COL="\\033[${RESULT_COL}G"
SET_WGET_COL="\\033[${WGET_COL}G"
SET_ARCH_COL="\\033[${ARCH_COL}G"
SET_FOUND_COL="\\033[${FOUND_COL}G"
SET_REQUIRED_COL="\\033[${REQUIRED_COL}G"
SET_TIME_COL="\\033[${TIME_COL}G"
SET_OPT_COL="\\033[${OPT_COL}G"
SET_VER_COL="\\033[${VER_COL}G"

# Define color for messages
BOLD="\\033[1;39m"
DONE="\\033[0;32m"
SKIP="\\033[0;34m"
WARN="\\033[0;35m"
FAIL="\\033[0;31m"
NORMAL="\\033[0;39m"
STOP="\\033[0;33m"
INFO="\\033[0;36m"

#########################################################################################################
# End of beautifying variables & presentation & input/output interface					#
#########################################################################################################


#########################################################################################################
# Now that .config is optional, set the toolchain variables here so they				#
# are available globally										#
# THESE ARE EXTREMELY IMPORTANT VARIABLES.								#
#########################################################################################################

# MACHINE is the target for userspace, so 32b (except alpha wich know only 64b world)
# MACHINE_REAL is the target for kernel compilation
# If MACHINE != MACHINE_REAL, gcc-x-cross-compile is build.
# Currently sparc64 and ppc64 use gcc-x-cross-compile
case ${MACHINE} in
	i?86)
		MACHINE=i486
		MACHINE_REAL=i486
		LINKER=/lib/ld-linux.so.2
		CFLAGS="-Os -fomit-frame-pointer -march=i486 -mtune=pentium -pipe"
		CXXFLAGS="-Os -fomit-frame-pointer -march=i486 -mtune=pentium -pipe"
		;;
	x86_64)
		MACHINE_REAL=i486
		MACHINE=i486
		WRAPPER_32BIT=linux32
		LINKER=/lib/ld-linux.so.2
		CFLAGS="-Os -fomit-frame-pointer -march=i486 -mtune=pentium -pipe"
		CXXFLAGS="-Os -fomit-frame-pointer -march=i486 -mtune=pentium -pipe"
		;;
	alpha)
		MACHINE=alpha
		MACHINE_REAL=alpha
		LINKER=/lib/ld-linux.so.2
		CFLAGS="-O2 -march=ev4 -mtune=ev56 -mieee -pipe"
		CXXFLAGS="-O2 -march=ev4 -mtune=ev56 -mieee -pipe"
		;;
	sparc|sparc64)
		MACHINE_REAL=${MACHINE}
		MACHINE=sparc
		WRAPPER_32BIT=linux32
		LINKER=/lib/ld-linux.so.2
		CFLAGS="-O2 -pipe -mcpu=ultrasparc -mtune=ultrasparc"
		CXXFLAGS="-O2 -pipe -mcpu=ultrasparc -mtune=ultrasparc"
		;;
	ppc|ppc64)
		MACHINE_REAL=${MACHINE}
		MACHINE=ppc
		WRAPPER_32BIT=linux32
		LINKER=/lib/ld.so.1
		CFLAGS="-O2 -pipe"
		CXXFLAGS="-O2 -pipe"
		;;
	*)
		echo -ne "${FAIL}Can't determine your architecture - ${MACHINE}${NORMAL}\n"
		exit 1
		;;
esac

export CCACHE_DIR=${BASEDIR}/ccache
export CCACHE_HASHDIR=1

if [ x"${USE_DISTCC}" == x"yes" -a ! -z "${DISTCC_HOSTS}" ]; then
	export CCACHE_PREFIX="distcc"
	export DISTCC_DIR=${BASEDIR}/distcc
fi

# This is the directory that holds the newly built ipcop system
LFS=${BASEDIR}/build_${MACHINE}/ipcop

# Set up what used to be /tools
TOOLS_DIR=tools_${MACHINE}

# Save the original PATH to a temporary variable.  We reset the PATH for the next few commands, just in case
ORG_PATH=${PATH}

# On some systems /sbin and /usr/sbin are not in non-root users' PATH.  Use a temporary PATH to include them
export PATH=/bin:/sbin:/usr/bin:/usr/sbin

# Find out the location of some programs. /etc/sudoers needs an absolute path to the executable for which
# users are granted access (ie you can't type mount, you have to type /bin/mount)
# We use bash's type as which may not be available
# Don't allow bash to use hashing for the commands, or this might not work
CHMOD=`bash +h -c "type chmod" | cut -d" " -f3`
CHROOT=`bash +h -c "type chroot" | cut -d" " -f3`
DU=`bash +h -c "type du" | cut -d" " -f3`
LN="`bash +h -c "type ln" | cut -d" " -f3` -sf"
LOSETUP=`bash +h -c "type losetup" | cut -d" " -f3`
MKDIR="`bash +h -c "type mkdir" | cut -d" " -f3` -p"
MOUNT=`bash +h -c "type mount" | cut -d" " -f3` > /dev/null 2>&1
BIND="${MOUNT} --bind"
MV=`bash +h -c "type mv" | cut -d" " -f3`
RENICE=`bash +h -c "type renice" | cut -d" " -f3`
RM="`bash +h -c "type rm" | cut -d" " -f3` -fr"
UMOUNT=`bash +h -c "type umount" | cut -d" " -f3`

# Who's running this script?
CURRENT_USER=`id -un`
CURRENT_USER_GROUP=`id -gn`

# Find where sudo is if we're doing a non-root build
if [ x"${CURRENT_USER}" != x"root" ]; then
	# Do we have sudo?
	if type sudo > /dev/null 2>&1; then
		SUDO=`bash +h -c "type sudo" | cut -d" " -f3`
	fi
else
	SUDO=
fi

# Reset PATH to the original
export PATH=${ORG_PATH}

# include machine in TOOLCHAINNAME
TOOLCHAINNAME=${SNAME}-${TOOLCHAINVERSION}-toolchain-${MACHINE}.tar.gz


#########################################################################################################
# Ok, now's your chance to override any of the "customizable" variables					#
#													#
# NO MORE VARIABLE OVERRIDING BELOW THIS LINE								#
#########################################################################################################
if [ -e ${IPCOP_CONFIG} ]; then
	echo -ne "${BOLD}*** IPCop build .config found. Parsing ...${NORMAL}\n"

	#################################################################################
	# Make sure the log directory exists						#
	#################################################################################
	${MKDIR} $BASEDIR/log_${MACHINE}

	# Now write the list of overwritten variables to _build_00_preparation.log
	echo "=== LIST OF OVERWRITTEN VARIABLES ===" >> ${BASEDIR}/log_${MACHINE}/_build_00_preparation.log

	ORG_IFS=${IFS}
	IFS=$'\n'

	# Take care of garbage
	sed -i "s@^ .*@#& ## DISABLED by make.sh@g" ${IPCOP_CONFIG}

	for LINE in `cat ${IPCOP_CONFIG} | grep -v "^#"`
	do
		VARIABLE=`echo "${LINE}" | cut -d= -f 1`
		VALUE=`echo "${LINE}" | cut -d= -f 2-`

		if [ -z $(eval echo "\${${VARIABLE}}") ]; then
			echo -ne "*** ${WARN}Ignoring invalid variable ${VARIABLE} and disabling in .config\n"
			sed -i "s@^.*${VARIABLE}.*@#& ## DISABLED by make.sh@g" ${IPCOP_CONFIG}
		elif echo ${CUSTOMIZABLE_VARIABLES[*]} | grep -qEi "${VARIABLE}"; then
			echo -ne "*** Setting ${BOLD}${VARIABLE}${NORMAL} to '${BOLD}${VALUE}${NORMAL}'\n"

			if [ x"${VARIABLE}" == x"PARALLELISM" ]; then
				if [ ${VALUE} -gt ${PARALLELISM} ]; then
					echo -ne "*** ${WARN}Setting PARALLELISM too high "
					echo -ne "may break the build${NORMAL}\n"
				fi
			fi

			export `eval echo ${VARIABLE}`=${VALUE}

			echo -ne "Variable ${VARIABLE} was overwritten and is now ${VALUE}\n" \
				>> ${BASEDIR}/log_${MACHINE}/_build_00_preparation.log

		else
			echo -ne "*** ${WARN}Skipping non-customizable variable ${VARIABLE} and disabling in .config\n"
			sed -i "s@^.*${VARIABLE}.*@#& ## DISABLED by make.sh@g" ${IPCOP_CONFIG}
		fi
	done
	IFS=${ORG_IFS}
	echo "=== END OF LIST OF OVERWRITTEN VARIABLES ===" >> ${BASEDIR}/log_${MACHINE}/_build_00_preparation.log

	echo -ne "${BOLD}*** Done parsing .config${NORMAL}\n"
else
	echo -ne "${BOLD}*** IPCop build .config not found. Using defaults${NORMAL}\n"
fi

#########################################################################################################
# NO MORE VARIABLE OVERRIDING BELOW THIS LINE								#
#########################################################################################################

#########################################################################################################
#########################################################################################################
# End of BLOCK 1 -- Variables										#
#########################################################################################################
#########################################################################################################




#########################################################################################################
#########################################################################################################
# BLOCK 2 -- Functions											#
#########################################################################################################
#########################################################################################################

#########################################################################################################
# This function show a nice spinning bar figuring build progress					#
#########################################################################################################
show_progress()
{
	# Hide the cursor
	echo -ne "\\033[?25l"

	# Move the cursor to the results column
	echo -ne "${SET_TIME_COL}[           ]"
	echo -ne "\\033[11D"

	while [ -f ${RUNNING_MARKER} -a ! -f ${FAILED_MARKER} ]
	do
		local PKG_TIME_END=`date +%s`
		beautify result PENDING $[ ${PKG_TIME_END} - ${PKG_TIME_START} ] ${1} ${PKG_VER} ${STAGE_ORDER} ${STAGE}
		echo -ne "\\033[11D" 2> /dev/null
		# on restart, this allow 10 lfs scripts ALREADY done per second
		sleep 0.1
	done

	# Show the cursor
	echo -ne "\\033[?25h"
} # End of show_progress()



#########################################################################################################
# This is the function that helps beautify() do its job							#
#########################################################################################################
position_cursor()
{
	# ARG1=starting position on screen
	# ARG2=string to be printed
	# ARG3=offset, negative for left movement, positive for right movement, relative to ARG1
	# For example if your starting position is column 50 and you want to print Hello three columns to the right
	# of your starting position, your call will look like this:
	# position_cursor 50 "Hello" 3 (you'll get the string Hello at position 53 (= 50 + 3)
	# If on the other hand you want your string "Hello" to end three columns to the left of position 50,
	# your call will look like this:
	# position_cursor 50 "Hello" -3 (you'll get the string Hello at position 42 (= 50 - 5 -3)
	# If you want to start printing at the exact starting location, use offset 0

	START=${1}
	STRING=${2}
	OFFSET=${3}

	STRING_LENGTH=${#STRING}

	if [ ${OFFSET} -lt 0 ]; then
		COL=$((${START} + ${OFFSET} - ${STRING_LENGTH}))
	else
		COL=$((${START} + ${OFFSET}))
	fi

	SET_COL="\\033[${COL}G"

	echo ${SET_COL}
} # End of position_cursor()



#########################################################################################################
# This is the function that makes the build process output pretty					#
#########################################################################################################
beautify()
{
	# Commands: build_stage, make_pkg, message, result
	case "${1}" in
		message)
			local SET_COL
			local MESSAGE
			if [ "${3}" -a x"${3}" == x"wget" ]; then
				local SET_COL=${SET_WGET_COL}
				MESSAGE=
			else
				local SET_COL=${SET_RESULT_COL}
				MESSAGE="${3}"
			fi

			case "${2}" in
				DONE)
					echo -ne "${BOLD}${MESSAGE}${NORMAL}${SET_COL}[${DONE} DONE ${NORMAL}]"
					;;
				WARN)
					echo -ne "${WARN}${MESSAGE}${NORMAL}${SET_COL}[${WARN} WARN ${NORMAL}]"
					;;
				FAIL)
					echo -ne "${BOLD}${MESSAGE}${NORMAL}${SET_COL}[${FAIL} FAIL ${NORMAL}]"
					;;
				SKIP)
					echo -ne "${BOLD}${MESSAGE}${NORMAL}${SET_COL}[${SKIP} SKIP ${NORMAL}]"
					;;
				STOP)
					echo -ne "${BOLD}${MESSAGE}${NORMAL}${SET_COL}[${STOP} STOP ${NORMAL}]"
					;;
				INFO)
					echo -ne "${INFO}${MESSAGE}${NORMAL}${SET_COL}[${INFO} INFO ${NORMAL}]"
					;;
				ARCH)
					SET_ARCH_COL_REAL=`position_cursor ${WGET_COL} ${3} -3`
					echo -ne "${SET_ARCH_COL}[ ${BOLD}${SET_ARCH_COL_REAL}${3}${NORMAL} ]"
					;;
			esac

			if [ x"${3}" != x"wget" -a x"${2}" != x"ARCH" ]; then
				echo -ne "\n"
			fi
			;;
		build_stage)
			MESSAGE=${2}
			echo -ne "${BOLD}*** ${MESSAGE}${SET_OPT_COL} options${SET_VER_COL}    version"
			echo -ne "${SET_TIME_COL} time (sec)${SET_RESULT_COL} status${NORMAL}\n"
			;;
		make_pkg)
			echo "${2}" | while read PKG_VER PROGRAM OPTIONS
			do
				SET_VER_COL_REAL=`position_cursor ${TIME_COL} ${PKG_VER} -3`

				if [ x"${OPTIONS}" == x"" ]; then
					echo -ne "${PROGRAM}${SET_VER_COL}[ ${BOLD}${SET_VER_COL_REAL}${PKG_VER}"
					echo -ne "${NORMAL} ]"
				else
					echo -ne "${PROGRAM}${SET_OPT_COL}[ ${BOLD}${OPTIONS}${NORMAL} ]"
					echo -ne "${SET_VER_COL}[ ${BOLD}${SET_VER_COL_REAL}${PKG_VER}"
					echo -ne "${NORMAL} ]${SET_RESULT_COL}"
				fi
			done
			;;
		result)
			RESULT=${2}

			if [ ! ${3} ]; then
				PKG_TIME=0
			else
				PKG_TIME=${3}
			fi

			local PROGRAM=${4}
			local VERSION=${5}
			local STAGE_ORDER=${6}
			local STAGE=${7}

			SET_TIME_COL_REAL=`position_cursor ${RESULT_COL} ${PKG_TIME} -3`
			case "${RESULT}" in
				DONE)
					echo -ne "${SET_TIME_COL}[ ${BOLD}${SET_TIME_COL_REAL}$PKG_TIME${NORMAL} ]"
					echo -ne "${SET_RESULT_COL}[${DONE} DONE ${NORMAL}]\n"
					;;
				FAIL)
					echo -ne "${SET_TIME_COL}[ ${BOLD}${SET_TIME_COL_REAL}$PKG_TIME${NORMAL} ]"
					echo -ne "${SET_RESULT_COL}[${FAIL} FAIL ${NORMAL}]\n"
					;;
				SKIP)
					echo -ne "${SET_TIME_COL}[ ${BOLD}${SET_TIME_COL_REAL}$PKG_TIME${NORMAL} ]"
					echo -ne "${SET_RESULT_COL}[${SKIP} SKIP ${NORMAL}]\n"
					;;
				PENDING)
					echo -ne "${SET_TIME_COL}[ ${BOLD}${SET_TIME_COL_REAL}$PKG_TIME${NORMAL} ]"
					;;
			esac
			;;
	esac
} # End of beautify()



#########################################################################################################
# This is the function that checks for required version of a prerequisite				#
#########################################################################################################
check_version()
{
	local PKG_NAME="${1}"
	local REAL_REQUIRED="${2}"
	local REAL_FOUND="${3}"
	local REQUIRED=`echo "${REAL_REQUIRED}" | sed "s,\.,DOT,g" | tr '[:punct:]' '-' | sed "s,DOT,.,g" | tr -d '[:alpha:]' | cut -d"-" -f1`
	local FOUND=`echo "${REAL_FOUND}" | sed "s,\.,DOT,g" | tr '[:punct:]' '-' | sed "s,DOT,.,g" | tr -d '[:alpha:]' | cut -d"-" -f1`
	local FAILURE=0
	local i

	echo -ne "Checking for ${PKG_NAME}" | tee -a ${LOGFILE}

	for i in `seq 1 4`; do
		if [ 0`echo ${FOUND} | cut -d"." -f ${i}` -gt 0`echo ${REQUIRED} | cut -d"." -f ${i}` ]; then
			FAILURE=0
			break
		elif [ 0`echo ${FOUND} | cut -d"." -f ${i}` -eq 0`echo ${REQUIRED} | cut -d"." -f ${i}` ]; then
			FAILURE=0
		else
			FAILURE=1
			break
		fi
	done

	SET_REQUIRED_COL_REAL=`position_cursor ${FOUND_COL} ${REAL_REQUIRED} -3`
	SET_FOUND_COL_REAL=`position_cursor ${RESULT_COL} ${REAL_FOUND} -3`

	echo -ne "${SET_REQUIRED_COL}[ ${BOLD}${SET_REQUIRED_COL_REAL}${REAL_REQUIRED}${NORMAL} ]"
	echo -ne "${SET_FOUND_COL}[ ${BOLD}${SET_FOUND_COL_REAL}${REAL_FOUND}${NORMAL} ]"

	if [ ${FAILURE} -eq 0 ]; then
		beautify message DONE
	else
		beautify message FAIL
	fi

	return ${FAILURE}
} # End of check_version()



#########################################################################################################
# This is the function that returns the version number of a package					#
#########################################################################################################
get_pkg_ver()
{
	PKG_VER=`grep ^VER ${1} | awk '{print $3}'`

	if [ -z ${PKG_VER} ]; then
		PKG_VER="svn-$(grep "# \$Id:" ${1} | awk '{print $4}')"
	fi

	if [ ${#PKG_VER} -gt ${VER_WIDTH} ]; then
		# If a package version number is greater than ${VER_WIDTH}, we keep the first 4 characters
		# and replace enough characters to fit the resulting string on the screen.  We'll replace
		# the extra character with .. (two dots).  That's why the "+ 2" in the formula below.
		# Example: if we have a 21-long version number that we want to fit into a 10-long space,
		# we have to remove 11 characters.  But if we replace 11 characters with 2 characters, we'll
		# end up with a 12-character long string.  That's why we replace 12 characters with ..
		REMOVE=`expr substr "${PKG_VER}" 4 $[ ${#PKG_VER} - ${VER_WIDTH} + 2 ]`
		PKG_VER=`echo ${PKG_VER/${REMOVE}/..}`
	fi

	echo "${PKG_VER}"
} # End of get_pkg_ver()



#########################################################################################################
# This is the function that unmounts all directories needed for the build				#
#########################################################################################################
stdumount()
{
	# First find and unmount any loop-mounted filesystems
	${MOUNT} | grep ".*${BASEDIR}.*loop.*" | sed 's/^.* on \(.[^ ]*\).*loop=\(.*\))/\1 \2/' \
	| while read MOUNT_POINT DEVICE
	do
		${SUDO} ${UMOUNT} ${MOUNT_POINT} > /dev/null 2>&1
		${SUDO} ${LOSETUP} -d ${DEVICE} > /dev/null 2>&1
	done

	# Now find and unmount any of the bound filesystems we need for the build
	for i in `${MOUNT} | grep ${BASEDIR} | sed 's/^.* on \(.[^ ]*\).*$/\1/'`
	do
		${SUDO} ${UMOUNT} ${i} > /dev/null 2>&1

		# If the unmount command failed, give it 3 more chances, 5 seconds apart
		# This is necessary if a child process is keeping a bound filesystem busy
		if [ ${?} -ne 0 ]; then

			# Print the next warning message on a new line
			echo -ne "\n"
			beautify message WARN "Retrying to unmount ${i}"

			for SEQ in `seq 1 3`
			do
				sleep 5
				echo -ne "[ Attempt ${SEQ} ]"
				${SUDO} ${UMOUNT} ${i} > /dev/null 2>&1

				if [ ${?} -eq 0 ]; then
					beautify message DONE
					break
				else
					beautify message FAIL
				fi
			done
		fi
	done

	# If after all our attempts to unmount we still fail, quit, but show a failure message
	if ${MOUNT} | grep ${BASEDIR} > /dev/null 2>&1; then
		beautify message FAIL "Some bound filesystems could not be unmounted. Unmount them manually"
		${MOUNT} | grep ${BASEDIR} | sed 's/^.* on \(.[^ ]*\).*$/\1/' | sort | uniq
	fi
} # End of stdumount()



#########################################################################################################
# This is the function that gracefully ends the build process						#
#########################################################################################################
exiterror()
{
	if echo "$*" | grep -i interrupted > /dev/null 2>&1; then
		beautify message STOP
		parentcounter=1
		childcounter=1
		MORECHILD=true
		# the first make.sh shell is this script
		# the second make.sh is already gone by ctrl/c
		# this is the parent script that make.sh is running
		pidarray[1]=`ps -ef| grep --max-count=1 ' make -f ' | awk '{ print $2 }'`
		# ps -ef > ../ps.log
		# first parent look sometime spontaneously killed by ctrl/c, so avoid if not needed
		if [ ! -z $pidarray[1] ]; then
			while [ x"$MORECHILD" = x"true" ]; do
				MORECHILD=false
				# for all parents
				for i in `seq $parentcounter $childcounter`; do
					# find each child
					for j in `ps -ef| grep -v 'ps ' | awk '$3 == '${pidarray[$i]}' { print $2 }'`; do
						childcounter=`expr $childcounter + 1`
						pidarray[$childcounter]=$j
						MORECHILD=true
					done
					parentcounter=`expr $parentcounter + 1`
				done
			done
			# having the full list, kill them all
			for i in `seq $childcounter`;do
				# there is the first killed message that I don't know how to avoid
				# echo "Killing ${pidarray[$i]}"
				kill -9 "${pidarray[$i]}" &>/dev/null
			done
		fi
	fi
	beautify message FAIL "\n${FAIL}ERROR${NORMAL}: ${BOLD}$*${NORMAL}"
	echo -ne "Check ${LOGFILE} for errors if applicable"

	# In case of an abnormal ending, make sure the cursor is visible
	echo -ne "\\033[?25h\n"

	# Then unmount everything
	stdumount

	exit 1
} # End of exiterror()



#########################################################################################################
# This is the function that makes sure our build environment meets the requirements for building ipcop	#
#########################################################################################################
check_build_env()
{
	echo -ne "${BOLD}*** Checking for host prerequisites${NORMAL}\n"

	#################################################################################
	# Are we running the right shell?						#
	#################################################################################
	echo -ne "Checking if we're using bash"
	if [ ! "${BASH}" ]; then
		exiterror "BASH environment variable is not set.  You're probably running the wrong shell."
	fi

	if [ -z "${BASH_VERSION}" ]; then
		exiterror "Not running BASH shell."
	fi

	beautify message DONE

	#################################################################################
	# Checking if running as non-root user						#
	#################################################################################
	echo -ne "Checking if we're running as non-root user"
	if [ `id -u` -eq 0 -a x"${ALLOW_ROOT_TO_BUILD}" != x"yes" ]; then
		exiterror "Building as root is no longer supported.\n Please use another account or set ALLOW_ROOT_TO_BUILD=yes in .config"
	elif [ `id -u` -eq 0 -a x"${ALLOW_ROOT_TO_BUILD}" == x"yes" ]; then
		beautify message WARN " You're on your own, Mr. root"
	else
		beautify message DONE

		#################################################################################
		# Checking if sudo was found							#
		#################################################################################
		echo -ne "Checking if we have sudo"
		if [ ! ${SUDO} -o -z ${SUDO} -a x"${ALLOW_ROOT_TO_BUILD}" != x"yes" ]; then
			exiterror "sudo not found but is required!"
		else
			beautify message DONE
		fi

		#################################################################################
		# Checking if sudo is configured						#
		#################################################################################
		echo -ne "Checking if sudo is configured ${BOLD}"
		SUDO_ERROR=0
		SUDO_NUMBER=12 # number of registered command, need to be adjusted in case of change
		if (! ${SUDO} -p " if you see a login prompt, type enter " -l 1>/dev/null); then
			SUDO_ERROR=1
		else
			SUDO_COUNT=`${SUDO} -l | wc -l`
			if [ $SUDO_COUNT -lt ${SUDO_NUMBER} ]; then
				echo -ne "${NORMAL} found only $SUDO_COUNT sudo command instead of mini ${SUDO_NUMBER}"
				SUDO_ERROR=1
			fi
		fi
		echo -ne "${NORMAL}"
		if [ ${SUDO_ERROR} -eq 0 ]; then
			echo -ne "${NORMAL}"
			beautify message DONE
		else
			beautify message FAIL
			echo -ne "${BOLD}"
			echo -ne "\nAs root, use visudo to add these lines to /etc/sudoers\n"
			echo -ne "Make sure you don't break the lines!\n\n"
			echo -ne "# *** IPCop configuration ***\n"
			echo -ne "User_Alias IPCOP_BUILDER = ${CURRENT_USER}\n\n"
			echo -ne "Cmnd_Alias BIND = ${BIND} /dev ${LFS}/dev, \\ \n"
			echo -ne "\t\t${BIND} /proc ${LFS}/proc, \\ \n"
			echo -ne "\t\t${BIND} ${BASEDIR}/* ${LFS}/*\n"
			echo -ne "Cmnd_Alias CHMOD = ${CHMOD} [0-9]* ${LFS}/*\n"
			echo -ne "Cmnd_Alias CHROOT = ${CHROOT} ${LFS} *\n"
			echo -ne "Cmnd_Alias DU = ${DU} -skx ${BASEDIR}\n"
			echo -ne "Cmnd_Alias LN = ${LN} * ${LFS}/*, \\ \n"
			echo -ne "\t\t${LN} ${BASEDIR}/build_${MACHINE}/${TOOLS_DIR} /\n"
			echo -ne "Cmnd_Alias LOSETUP = ${LOSETUP} -d /dev/loop*\n"
			echo -ne "Cmnd_Alias MKDIR = ${MKDIR} ${LFS}/*, \\ \n"
			echo -ne "\t\t${MKDIR} ${BASEDIR}/build_${MACHINE}/${INSTALLER_DIR}\n"
			echo -ne "Cmnd_Alias MV = ${MV} ${LFS}/tmp/* \\ \n"
			echo -ne "\t\t\t\t${LFS}/tmp/*, \\ \n"
			echo -ne "\t\t${MV} ${BASEDIR}/build_${MACHINE}/${INSTALLER_DIR}/images/ipcop-* \\ \n"
			echo -ne "\t\t\t\t${BASEDIR}/\n"
			echo -ne "Cmnd_Alias RENICE = ${RENICE} * *\n"
			echo -ne "Cmnd_Alias RM = ${RM} ${BASEDIR}/*, \\ \n"
			echo -ne "\t\t${RM} /${TOOLS_DIR}\n"
			echo -ne "Cmnd_Alias UMOUNT = ${UMOUNT} ${LFS}/*, \\ \n"
			echo -ne "\t\t${UMOUNT} /dev/loop*\n\n"
			echo -ne "IPCOP_BUILDER	ALL = NOPASSWD: BIND,CHMOD,CHROOT,DU,LN,LOSETUP,MKDIR,MV,RENICE,RM,UMOUNT\n"
			echo -ne "# *** End of IPCop configuration ***\n"
			echo -ne "${NORMAL}"

			exiterror "sudo is not configured for user \"${CURRENT_USER}\"."
		fi
	fi

	#################################################################################
	# Checking for necessary temporary space					#
	#################################################################################
	if [[ (x"${SKIP_CHECK_DISKSPACE}" == x"no") && (x"${ACTION}" = x'build' || x"${ACTION}" = x'prefetch' || x"${ACTION}" = x'toolchain') ]]; then
		BASE_DEV=`df -P -k ${BASEDIR} | tail -n 1 | awk '{ print $1 }'`
		echo -ne "Checking for necessary space on disk ${BASE_DEV}"
		BASE_ASPACE=`df -P -k ${BASEDIR} | tail -n 1 | awk '{ print $4 }'`
		if (( 2202000 > ${BASE_ASPACE} )); then
			BASE_USPACE=`${SUDO} ${DU} -skx ${BASEDIR} | awk '{print $1}'`
			if (( 2202000 - ${BASE_USPACE} > ${BASE_ASPACE} )); then
				exiterror "Not enough temporary space available, need at least 2.1GB on ${BASE_DEV}"
			else
				beautify message DONE
			fi
		else
			beautify message DONE
		fi
	fi
} # End of check_build_env()



#########################################################################################################
# This is the function that prepares the build environment						#
#########################################################################################################
prepareenv()
{
	echo -ne "${BOLD}*** Setting up our build environment${NORMAL}\n"

	#################################################################################
	# Make sure the log directory exists						#
	#################################################################################
	${MKDIR} $BASEDIR/log_${MACHINE}

	LOGFILE=${BASEDIR}/log_${MACHINE}/_build_00_preparation.log
	export LOGFILE

	#################################################################################
	# System configuration								#
	#################################################################################
	echo "System configuration" >> ${LOGFILE}

	# Set umask
	umask 022

	#################################################################################
	# Trap on emergency exit							#
	#################################################################################
	trap "exiterror 'Build process interrupted'" SIGINT SIGTERM SIGKILL SIGSTOP SIGQUIT

	#################################################################################
	# Setting our nice level							#
	#################################################################################
	if [ x`nice` != x"${NICE}" ]; then
		echo "Setting our nice level to ${NICE}" >> ${LOGFILE}
		echo -ne "Setting our nice level to ${NICE}"
		${SUDO} ${RENICE} ${NICE} $$ > /dev/null

		if [ x`nice` != x"${NICE}" ]; then
			exiterror "Failed to set correct nice level"
		else
			beautify message DONE
		fi
	fi

	#################################################################################
	# Set SCHED_BATCH								#
	#################################################################################
	if [ -x /usr/bin/schedtool ]; then
		echo -ne "Setting kernel schedular to SCHED_BATCH"
		/usr/bin/schedtool -B $$

		if [ ${?} -eq 0 ]; then
			beautify message DONE
		else
			beautify message FAIL
		fi
	fi

	#################################################################################
	# Check TOOLS_DIR symlink							#
	#################################################################################
	if [ -h /${TOOLS_DIR} ]; then
		${SUDO} ${RM} /${TOOLS_DIR}
	fi

	if [ ! -a /${TOOLS_DIR} ]; then
		${SUDO} ${LN} ${BASEDIR}/build_${MACHINE}/${TOOLS_DIR} /
	fi

	if [ ! -h /${TOOLS_DIR} ]; then
		exiterror "Could not create /${TOOLS_DIR} symbolic link."
	fi

	#################################################################################
	# Setup environment								#
	#################################################################################
	set +h
	LC_ALL=POSIX
	export LC_ALL
	unset CC CXX CPP LD_LIBRARY_PATH LD_PRELOAD

	#################################################################################
	# Make some extra directories							#
	#################################################################################
	${MKDIR} ${BASEDIR}/{cache,ccache}
	${MKDIR} ${BASEDIR}/build_${MACHINE}/${TOOLS_DIR}/usr
	if [ x"${RUNNING_TEST}" == x"yes" ]; then
		${MKDIR} ${BASEDIR}/test_${MACHINE}/${BUILDDATE}
	fi

	if [ x"${USE_DISTCC}" == x"yes" -a ! -z "${DISTCC_HOSTS}" ]; then
		${MKDIR} ${DISTCC_DIR}
		${SUDO} ${MKDIR} ${LFS}/usr/src/distcc
	fi

	${SUDO} ${MKDIR} ${BASEDIR}/build_${MACHINE}/${INSTALLER_DIR}
	${SUDO} ${MKDIR} ${LFS}/bin
	${SUDO} ${MKDIR} ${LFS}/${TOOLS_DIR}
	${SUDO} ${MKDIR} ${LFS}/dev/pts
	${SUDO} ${MKDIR} ${LFS}/proc
	${SUDO} ${MKDIR} ${LFS}/usr/src/{cache,ccache,config,doc,html,langs,lfs,log_${MACHINE},src,test_${MACHINE}}
	${SUDO} ${MKDIR} ${LFS}/${INSTALLER_DIR}
	${SUDO} ${MKDIR} ${LFS}/etc

	# If everyone can write to ${LFS}/usr/src, at least set the sticky bit
	${SUDO} ${CHMOD} 1777 ${LFS}/usr/src

	#################################################################################
	# Make sure ${LFS}/bin/bash exists.  We used to do this check in lfs/bash, but	#
	# it's better to do it here so that the toolchain becomes completely		#
	# self-sufficient (ie you can start with no build_MACHINE/ipcop and lfs/stage2)	#
	#################################################################################
	if [ ! -f ${LFS}/bin/bash ]; then
		${SUDO} ${LN} /${TOOLS_DIR}/bin/bash ${LFS}/bin/bash
		${SUDO} ${LN} bash ${LFS}/bin/sh
	fi

	#################################################################################
	# Make all sources and proc available under lfs build				#
	#################################################################################
	${SUDO} ${BIND} /dev							${LFS}/dev
	${SUDO} ${BIND} /proc							${LFS}/proc
	${SUDO} ${BIND} ${BASEDIR}/cache					${LFS}/usr/src/cache
	${SUDO} ${BIND} ${BASEDIR}/ccache					${LFS}/usr/src/ccache
	${SUDO} ${BIND} ${BASEDIR}/config					${LFS}/usr/src/config
	${SUDO} ${BIND} ${BASEDIR}/doc						${LFS}/usr/src/doc
	${SUDO} ${BIND} ${BASEDIR}/html						${LFS}/usr/src/html
	${SUDO} ${BIND} ${BASEDIR}/langs					${LFS}/usr/src/langs
	${SUDO} ${BIND} ${BASEDIR}/lfs						${LFS}/usr/src/lfs
	${SUDO} ${BIND} ${BASEDIR}/log_${MACHINE}				${LFS}/usr/src/log_${MACHINE}
	${SUDO} ${BIND} ${BASEDIR}/src						${LFS}/usr/src/src
	${SUDO} ${BIND} ${BASEDIR}/test_${MACHINE}				${LFS}/usr/src/test_${MACHINE}
	${SUDO} ${BIND} ${BASEDIR}/build_${MACHINE}/${TOOLS_DIR}		${LFS}/${TOOLS_DIR}
	${SUDO} ${BIND} ${BASEDIR}/build_${MACHINE}/${INSTALLER_DIR}		${LFS}/${INSTALLER_DIR}

	#################################################################################
	# Write the distcc hosts only if we're using distcc and we've specified the hosts
	#################################################################################
	if [ x"${USE_DISTCC}" == x"yes" -a ! -z "${DISTCC_HOSTS}" ]; then
		echo "${DISTCC_HOSTS}" > ${DISTCC_DIR}/hosts
		${SUDO} ${BIND} ${DISTCC_DIR}					${LFS}/usr/src/distcc
	fi

	echo -n "Running compilation tests : "
	beautify message INFO "$RUNNING_TEST"

	# Remove pre-install list of installed files in case user erase some files before to build again
	${SUDO} ${RM} ${LFS}/usr/src/lsalr 2>/dev/null
	# List of packages is totally remade at each build
	${RM} ${LFS}/usr/src/log_${MACHINE}/_build_06_othersrc-list.log 2>/dev/null
	# This log is totally remade at each build, so don't accumulate log
	${RM} ${LFS}/usr/src/log_${MACHINE}/_build_05_packages.log 2>/dev/null
} # End of prepareenv()



#########################################################################################################
# This is the function that checks for toolchain prerequisites						#
#########################################################################################################
check_toolchain_prerequisites()
{
	LOGFILE=${BASEDIR}/log_${MACHINE}/_build_00_preparation.log
	export LOGFILE

	local SUCCESS=1
	local RESULT

	echo -ne "${BOLD}*** Checking for toolchain prerequisites${SET_REQUIRED_COL}   required${SET_FOUND_COL}    found"
	echo -ne "${SET_RESULT_COL} status${NORMAL}\n"

	check_version "GNU bash" ${REQUIRED_BASH} `bash --version | head -n1 | cut -d" " -f4 | cut -d"(" -f1`
	RESULT=${?}
	SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
	if [ ${RESULT} -eq 0 ]; then
		echo ": DONE" >> ${LOGFILE}
	else
		echo ": FAIL" >> ${LOGFILE}
	fi

	if bash +h -c "type ld" > /dev/null 2>&1; then
		check_version "GNU binutils" ${REQUIRED_BINUTILS} \
			`ld --version | head -n1 | sed "s,.* \([0-9]*\.[0-9][^ ]*\).*$,\1,g"`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU binutils not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	if bash +h -c "type bison" > /dev/null 2>&1; then
		check_version "GNU bison" ${REQUIRED_BISON} `bison --version | head -n1 | cut -d" " -f4`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU bison not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	if bash +h -c "type bzip2" > /dev/null 2>&1; then
		check_version "GNU bzip2" ${REQUIRED_BZIP2} \
			`bzip2 --version 2>&1 < /dev/null | head -n1 | cut -d" " -f8 | cut -d, -f1`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU bzip2 not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	if bash +h -c "type chown" > /dev/null 2>&1; then
		check_version "GNU coreutils" ${REQUIRED_COREUTILS} `chown --version | head -n1 | cut -d")" -f2 | cut -d" " -f2`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU coreutils not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	if bash +h -c "type diff" > /dev/null 2>&1; then
		check_version "GNU diffutils" ${REQUIRED_DIFFUTILS} `diff --version | head -n1 | cut -d" " -f4`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU diffutils not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	if bash +h -c "type find" > /dev/null 2>&1; then
		check_version "GNU findutils" ${REQUIRED_FINDUTILS} `find --version | head -n1 | cut -d" " -f4`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU findutils not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	if bash +h -c "type gawk" > /dev/null 2>&1; then
		check_version "GNU awk" ${REQUIRED_GAWK} `gawk --version | head -n1 | cut -d" " -f3`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU awk not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	if bash +h -c "type gcc" > /dev/null 2>&1; then
		check_version "GNU CC" ${REQUIRED_GCC} `gcc -dumpversion`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU CC not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	check_version "GNU libc" ${REQUIRED_GLIBC} `/lib/libc.so.6* | head -n1 | cut -d" " -f7 | cut -d"," -f1`
	RESULT=${?}
	SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
	if [ ${RESULT} -eq 0 ]; then
		echo ": DONE" >> ${LOGFILE}
	else
		echo ": FAIL" >> ${LOGFILE}
	fi

	if bash +h -c "type grep" > /dev/null 2>&1; then
		check_version "GNU grep" ${REQUIRED_GREP} \
			`grep --version | head -n1 | sed "s,.* \([0-9]*\.[0-9][^ ]*\).*$,\1,g"`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU grep not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	if bash +h -c "type gzip" > /dev/null 2>&1; then
		check_version "GNU gzip" ${REQUIRED_GZIP} `gzip --version | head -n1 | cut -d" " -f2`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU gzip not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	check_version "Linux Kernel" ${REQUIRED_KERNEL} `uname -r`
	RESULT=${?}
	SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
	if [ ${RESULT} -eq 0 ]; then
		echo ": DONE" >> ${LOGFILE}
	else
		echo ": FAIL" >> ${LOGFILE}
	fi

	if bash +h -c "type make" > /dev/null 2>&1; then
		check_version "GNU make" ${REQUIRED_MAKE} `make --version | head -n1 | cut -d" " -f3`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU make not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	if bash +h -c "type patch" > /dev/null 2>&1; then
		check_version "GNU patch" ${REQUIRED_PATCH} `patch --version | head -n1 | cut -d" " -f2`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU patch not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	if bash +h -c "type sed" > /dev/null 2>&1; then
		check_version "GNU sed" ${REQUIRED_SED} `sed --version | head -n1 | cut -d" " -f4`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU sed not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	if bash +h -c "type tar" > /dev/null 2>&1; then
		check_version "GNU tar" ${REQUIRED_TAR} `tar --version | head -n1 | cut -d" " -f4`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU tar not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	if bash +h -c "type makeinfo" > /dev/null 2>&1; then
		check_version "GNU texinfo" ${REQUIRED_TEXINFO} `makeinfo --version | head -n1 | cut -d" " -f4`
		RESULT=${?}
		SUCCESS=$[ ${SUCCESS} - ${RESULT} ]
		if [ ${RESULT} -eq 0 ]; then
			echo ": DONE" >> ${LOGFILE}
		else
			echo ": FAIL" >> ${LOGFILE}
		fi
	else
		beautify message FAIL "GNU texinfo not found but it's required"
		SUCCESS=$[ ${SUCCESS} - 1 ]
	fi

	if [ ${SUCCESS} -lt 1 ]; then
		if [ x${IGNORE_PREREQUISITES} == x"yes" ]; then
			beautify message WARN "You've ignored failed host requirements!  You're on your own."
		else
			exiterror "Some host system requirements not met!  See the output above and rectify."
		fi
	fi
} # End of check_toolchain_prerequisites()



#########################################################################################################
# This is the function that sets the environment of a chroot and enters it				#
#########################################################################################################
entershell()
{
	PATH=/${TOOLS_DIR}/usr/bin:/bin:/usr/bin:/sbin:/usr/sbin:/${TOOLS_DIR}/bin:/usr/${MACHINE_REAL}-linux/bin

	if [ ! -e ${LFS}/usr/src/lfs/ ]; then
		exiterror "No such file or directory: ${LFS}/usr/src/lfs/"
	fi

	echo -ne "Entering ${BOLD}${MACHINE_REAL}${NORMAL} LFS chroot, type exit to return to host environment\n"

	${SUDO} ${CHROOT} ${LFS} /${TOOLS_DIR}/bin/env -i \
		HOME=/root \
		TERM=${TERM} \
		PS1="\[${BOLD}\][chroot-${MACHINE_REAL}]\[${NORMAL}\] \u:\w\$ " \
		PATH=${PATH} \
		CONFIG_ROOT=${CONFIG_ROOT} \
		VERSION=${VERSION} \
		NAME=${NAME} \
		SNAME=${SNAME} \
		SLOGAN="${SLOGAN}" \
		CCACHE_DIR=/usr/src/ccache \
		CCACHE_PREFIX=${CCACHE_PREFIX} \
		CCACHE_HASHDIR=${CCACHE_HASHDIR} \
		DISTCC_DIR=/usr/src/distcc \
		PARALLELISM=${PARALLELISM} \
		LINKER=${LINKER} \
		TOOLS_DIR=${TOOLS_DIR} \
		INSTALLER_DIR=${INSTALLER_DIR} \
		MACHINE="${MACHINE}" \
		MACHINE_REAL="${MACHINE_REAL}" \
		CFLAGS="${CFLAGS}" \
		CXXFLAGS="${CXXFLAGS}" \
		KVER=${KVER} \
		STAGE=${STAGE} \
		STAGE_ORDER=${STAGE_ORDER} \
		LOGFILE=`echo ${LOGFILE} | sed "s,${BASEDIR},/usr/src,g"` \
		${WRAPPER_32BIT} bash

	if [ ${?} -ne 0 ]; then
		exiterror "chroot error"
	else
		stdumount
	fi
} # End of entershell()



#########################################################################################################
# This is the function that downloads and checks md5sum of every package				#
# Return:0 caller can continue										#
#	:1 skip (nothing to do)										#
#	or fail if no script file found									#
#########################################################################################################
lfsmakecommoncheck()
{
	# Script present?
	if [ ! -f ${BASEDIR}/lfs/${1} ]; then
		exiterror "No such file or directory: ${BASEDIR}/lfs/${1}"
	fi

	if grep -E "^HOST_ARCH.*${MACHINE}|^HOST_ARCH.*all" ${BASEDIR}/lfs/${1} >/dev/null; then
		local PKG_VER=`get_pkg_ver ${BASEDIR}/lfs/${1}`

		beautify make_pkg "${PKG_VER} $*"

		echo -e "`date -u '+%b %e %T'`: Building $* " >> ${LOGFILE}

		cd ${BASEDIR}/lfs && make -s -f $* MACHINE=${MACHINE} LFS_BASEDIR=${BASEDIR} MESSAGE="${1}\t " \
			download  >> ${LOGFILE} 2>&1
		if [ ${?} -ne 0 ]; then
			exiterror "Download error in ${1}"
		fi

		cd ${BASEDIR}/lfs && make -s -f $* MACHINE=${MACHINE} LFS_BASEDIR=${BASEDIR} MESSAGE="${1}\t md5sum" \
			md5  >> ${LOGFILE} 2>&1
		if [ ${?} -ne 0 ]; then
			exiterror "md5sum error in ${1}, check file in cache or signature"
		fi

		return 0	# pass all!
	else
		return 1
	fi
} # End of lfsmakecommoncheck()



#########################################################################################################
# This is the function that builds every package in stage "toolchain"					#
#########################################################################################################
toolchain_make()
{
	lfsmakecommoncheck $*
	[ ${?} -eq 1 ] && return 0

	local PKG_TIME_START=`date +%s`
	local PKG_VER=`get_pkg_ver ${BASEDIR}/lfs/${1}`

	local WRAPPER_32BIT=${WRAPPER_32BIT}

	# On both 32bit and 64bit machines, we first compile linux32, a wrapper that emulates 32bit userland
	# on alpha, WRAPPER_32BIT is not set
	[ x"${1}" == x"linux32" ] && unset WRAPPER_32BIT

	# Create some progress markers
	${RM} ${FAILED_MARKER}
	${RM} ${RUNNING_MARKER}
	touch ${RUNNING_MARKER}

	# CFLAGS and CXXFLAGS are not set on purpose (recommended to be empty by lfs)
	cd ${BASEDIR}/lfs && (if ${WRAPPER_32BIT} make -f $* \
		CONFIG_ROOT=${CONFIG_ROOT} \
		LINKER=${LINKER} \
		TOOLS_DIR=${TOOLS_DIR} \
		REQUIRED_KERNEL=${REQUIRED_KERNEL} \
		MACHINE="${MACHINE}" \
		BUILDDATE="${BUILDDATE}" \
		LFS_BASEDIR=${BASEDIR} \
		LFS=${LFS} \
		PARALLELISM=${PARALLELISM} \
		RUNNING_TEST=${RUNNING_TEST} \
		STAGE=${STAGE} \
		STAGE_ORDER=${STAGE_ORDER} \
		SHELL='/bin/bash' \
		install >> ${LOGFILE} 2>&1; then \
				${RM} ${RUNNING_MARKER}; \
			else \
				${RM} ${RUNNING_MARKER}; \
				touch ${FAILED_MARKER}; \
			fi ) &

	[ x"${SHOW_PROGRESS}" == x"yes" ] && show_progress

	# Wait until the child processes have finished
	wait

	local PKG_TIME_END=`date +%s`

	if [ -f ${FAILED_MARKER} ]; then
		${RM} ${FAILED_MARKER}
		beautify result FAIL $[ ${PKG_TIME_END} - ${PKG_TIME_START} ] ${1} ${PKG_VER} ${STAGE_ORDER} ${STAGE}
		exiterror "Building $*";
	else
		beautify result DONE $[ ${PKG_TIME_END} - ${PKG_TIME_START} ] ${1} ${PKG_VER} ${STAGE_ORDER} ${STAGE}
	fi

	return 0
} # End of toolchain_make()



#########################################################################################################
# This is the function that builds every package in stage "base"					#
#########################################################################################################
chroot_make()
{
	lfsmakecommoncheck $*
	[ ${?} -eq 1 ] && return 0

	local PKG_TIME_START=`date +%s`
	local PKG_VER=`get_pkg_ver ${BASEDIR}/lfs/${1}`

	local WRAPPER_32BIT=${WRAPPER_32BIT}
	# When cross-compiling, make sure the kernel is compiled for the target
	[ x"${MACHINE_REAL}" == x"sparc64" -a x"${1}" == x"linux" ] && unset WRAPPER_32BIT
	# don't play 32bit on alpha
	[ x${MACHINE} == x"alpha" ] && unset WRAPPER_32BIT

	# Also, make sure external kernel modules are compiled 64bit
	if grep -qEi 'KERNEL_MOD = yes' ${1} ; then
		[ x"${MACHINE_REAL}" == x"sparc64" ] && unset WRAPPER_32BIT
	fi

	# Create some progress markers
	${RM} ${FAILED_MARKER}
	${RM} ${RUNNING_MARKER}
	touch ${RUNNING_MARKER}

	(if ${SUDO} ${CHROOT} ${LFS} /${TOOLS_DIR}/bin/env -i \
		HOME=/root \
		TERM=${TERM} \
		PS1='\u:\w\$ ' \
		PATH=${PATH} \
		CONFIG_ROOT=${CONFIG_ROOT} \
		VERSION=${VERSION} \
		NAME=${NAME} \
		SNAME=${SNAME} \
		SLOGAN="${SLOGAN}" \
		CCACHE_DIR=/usr/src/ccache \
		CCACHE_PREFIX=${CCACHE_PREFIX} \
		CCACHE_HASHDIR=${CCACHE_HASHDIR} \
		DISTCC_DIR=/usr/src/distcc \
		PARALLELISM=${PARALLELISM} \
		LINKER=${LINKER} \
		TOOLS_DIR=${TOOLS_DIR} \
		INSTALLER_DIR=${INSTALLER_DIR} \
		REQUIRED_KERNEL=${REQUIRED_KERNEL} \
		MACHINE="${MACHINE}" \
		MACHINE_REAL="${MACHINE_REAL}" \
		BUILDDATE="${BUILDDATE}" \
		CFLAGS="${CFLAGS}" \
		CXXFLAGS="${CXXFLAGS}" \
		RUNNING_TEST=${RUNNING_TEST} \
		KVER=${KVER} \
		STAGE=${STAGE} \
		STAGE_ORDER=${STAGE_ORDER} \
		LOGFILE=`echo ${LOGFILE} | sed "s,${BASEDIR},/usr/src,g"` \
		bash -x -c "cd /usr/src/lfs && \
		${WRAPPER_32BIT} make -f $* LFS_BASEDIR=/usr/src install" >> ${LOGFILE} 2>&1; then \
				${RM} ${RUNNING_MARKER}; \
			else \
				${RM} ${RUNNING_MARKER}; \
				touch ${FAILED_MARKER}; \
			fi ) &

	[ x"${SHOW_PROGRESS}" == x"yes" ] && show_progress

	# Wait until the child processes have finished
	wait

	local PKG_TIME_END=`date +%s`

	if [ -f ${FAILED_MARKER} ]; then
		${RM} ${FAILED_MARKER}
		beautify result FAIL $[ ${PKG_TIME_END} - ${PKG_TIME_START} ] ${1} ${PKG_VER} ${STAGE_ORDER} ${STAGE}
		exiterror "Building $*";
	else
		beautify result DONE $[ ${PKG_TIME_END} - ${PKG_TIME_START} ] ${1} ${PKG_VER} ${STAGE_ORDER} ${STAGE}
	fi

	return 0
} # End of chroot_make()



#########################################################################################################
# This builds the entire stage "toolchain"								#
#########################################################################################################
toolchain_build()
{
	beautify build_stage "Building toolchain"

	ORG_PATH=${PATH}
	export PATH=/${TOOLS_DIR}/usr/bin:/${TOOLS_DIR}/bin:${PATH}
	STAGE_ORDER=01
	STAGE=toolchain

	${MKDIR} ${BASEDIR}/log_${MACHINE}/01_toolchain

	LOGFILE="${BASEDIR}/log_${MACHINE}/_build_01_toolchain.log"
	export LOGFILE

	if [ -f $BASEDIR/log_${MACHINE}/01_toolchain/ccache-* ]; then
		beautify message INFO "You can't partially rebuild some parts of the toolchain cleanly."
		beautify message INFO "Preferably before each toolchain build, use:\n./make.sh clean"
	fi

	if [ x`uname -m` == x"x86_64" ]; then
		exiterror "Toolchain build does not work actually on x86_64.\n" \
			"You need to use one made on a i686\n or load one with ./make.sh gettoolchain to be able to continue\n"
	fi

	# For 64bit hosts except alpha, we compile 32bit userland and 64bit kernel.  We build a small helper that
	# fools the system into thinking we're on a 32bit architecture. This works for x86_64, ppc64 and sparc64
	# We use linux32 even on 32bit machines, so 32bit and 64bit could use exactly the same toolchain package
	if [ x"${MACHINE}" != x"alpha" ]; then
		toolchain_make linux32
	fi

	# make distcc first so that CCACHE_PREFIX works immediately
	if [ x"${USE_DISTCC}" == x"yes" -a ! -z "${DISTCC_HOSTS}" ]; then
		toolchain_make distcc
	fi

	toolchain_make ccache
	toolchain_make binutils						PASS=1
	toolchain_make gcc						PASS=1
	toolchain_make linux-headers
	toolchain_make glibc
	toolchain_make adjust-toolchain
	toolchain_make tcl	# tcl,expect,dejagnu only needed to run gcc tests
	toolchain_make expect
	toolchain_make dejagnu
	toolchain_make gcc						PASS=2
	toolchain_make binutils						PASS=2
	toolchain_make ncurses
	toolchain_make bash
	toolchain_make bzip2
	toolchain_make lzmautils
	toolchain_make coreutils
	toolchain_make diffutils
	toolchain_make e2fsprogs
	toolchain_make findutils
	toolchain_make gawk
	toolchain_make gettext
	toolchain_make grep
	toolchain_make gzip
	toolchain_make make
	toolchain_make patch
	toolchain_make perl
	toolchain_make sed
	toolchain_make tar
	toolchain_make texinfo
	toolchain_make util-linux-ng
	toolchain_make strip
	export PATH=${ORG_PATH}
} # End of toolchain_build()



#########################################################################################################
# This builds the entire stage "base"									#
#########################################################################################################
base_build()
{
	beautify build_stage "Building base"

	PATH=/${TOOLS_DIR}/usr/bin:/bin:/usr/bin:/sbin:/usr/sbin:/${TOOLS_DIR}/bin
	STAGE_ORDER=02
	STAGE=base

	${MKDIR} ${BASEDIR}/log_${MACHINE}/02_base

	LOGFILE="${BASEDIR}/log_${MACHINE}/_build_02_base.log"
	export LOGFILE

	chroot_make stage2
	chroot_make linux-headers
	chroot_make glibc
	chroot_make adjust-toolchain
	chroot_make binutils
	chroot_make gcc
	chroot_make sed
	chroot_make e2fsprogs
	chroot_make coreutils
	chroot_make iana-etc
	chroot_make m4
	chroot_make bison
	chroot_make ncurses
	chroot_make procps
	chroot_make libtool
	chroot_make zlib
	chroot_make perl
	chroot_make readline
	chroot_make autoconf
	chroot_make automake
	chroot_make bash
	chroot_make bzip2
	chroot_make lzmautils
	chroot_make diffutils
	chroot_make file
	chroot_make findutils
	chroot_make flex
	chroot_make aboot
	chroot_make sparc-utils
	chroot_make silo
	chroot_make powerpc-utils
	chroot_make hfsutils
	chroot_make yaboot
	chroot_make quik
	chroot_make gawk
	chroot_make gettext
	chroot_make grep
	chroot_make groff
	chroot_make gzip
	chroot_make iproute2
	chroot_make kbd
	chroot_make less
	chroot_make make
	chroot_make module-init-tools
	chroot_make patch
	chroot_make pkg-config
	chroot_make psmisc
	chroot_make shadow
	chroot_make sysklogd
	chroot_make sysvinit
	chroot_make tar
	chroot_make texinfo
	chroot_make udev
	chroot_make util-linux-ng
	chroot_make vim
} # End of base_build()



#########################################################################################################
# This builds the entire stage "ipcop"									#
#########################################################################################################
ipcop_build()
{
	beautify build_stage "Building ipcop"

	PATH=/${TOOLS_DIR}/usr/bin:/bin:/usr/bin:/sbin:/usr/sbin:/${TOOLS_DIR}/bin:/usr/${MACHINE_REAL}-linux/bin
	STAGE_ORDER=03
	STAGE=ipcop

	${MKDIR} ${BASEDIR}/log_${MACHINE}/03_ipcop

	LOGFILE="${BASEDIR}/log_${MACHINE}/_build_03_ipcop.log"
	export LOGFILE

	# Build these first as some of the kernel packages below rely on
	# these for some of their client program functionality
	chroot_make ipcop
	chroot_make sysfsutils
	chroot_make which
	chroot_make net-tools
	chroot_make libusb
	chroot_make libpcap
	chroot_make libxml2
	chroot_make linux-atm
	chroot_make ppp
	chroot_make rp-pppoe
	chroot_make unzip
	# Build cross-compile mini-toolchain (binutils and gcc) so we can build a kernel for 64bit machines
	chroot_make binutils-cross-compile
	chroot_make gcc-cross-compile
	chroot_make linux
	chroot_make binary-firmware-all
	#chroot_make 3cp4218    # DEBUG -- broken with kernel 2.6.27
	#chroot_make amedyn2    # DEBUG -- broken with kernel 2.6.27
	#chroot_make CnxADSL	# DEBUG -- driver warn to not be SMP safe but CHANGELOG say this is fixed
	#chroot_make wanpipe    # DEBUG -- broken with kernel 2.6.25
	#chroot_make fcdsl    # DEBUG -- broken with kernel 2.6.27
	#chroot_make fcdsl2    # DEBUG -- broken with kernel 2.6.27
	#chroot_make fcdslsl    # DEBUG -- broken with kernel 2.6.27
	chroot_make pulsar
	chroot_make pcmciautils
	chroot_make eciadsl-usermode
	chroot_make cpio
	chroot_make expat
	chroot_make gmp
	chroot_make bc			# needed to run some openssl tests
	chroot_make openssl
	chroot_make libgpg-error	# OpenVPN
	chroot_make libgcrypt		# OpenVPN
	chroot_make libnet
	chroot_make libpng
	chroot_make glib
	chroot_make dejavu
	chroot_make freetype
	chroot_make libgd
	chroot_make fontconfig
	chroot_make pixman
	chroot_make cairo
	chroot_make pango
	chroot_make popt
	chroot_make libcap
	chroot_make pciutils
	chroot_make pcre
	chroot_make usbutils
	chroot_make acpid
	chroot_make apache
	chroot_make arping
	chroot_make beep
	chroot_make bind
	chroot_make capi4k-utils
	chroot_make dhcpcd
	chroot_make dnsmasq
	chroot_make dosfstools
	chroot_make ethtool
	chroot_make ez-ipupdate
	chroot_make fcron
	chroot_make gnupg
	chroot_make hdparm
	chroot_make ibod
	chroot_make iptables
	chroot_make libnfnetlink
	chroot_make libnetfilter_conntrack
	chroot_make conntrack-tools
	#chroot_make iptstate		# DEBUG -- wants libnetfilter and libstdc++
	chroot_make iftop
	chroot_make ipcop-gui
	chroot_make iputils
	chroot_make isdn4k-utils
	chroot_make logrotate
	chroot_make logwatch
	chroot_make lzo			# OpenVPN
	chroot_make mdadm
	chroot_make misc-progs
	chroot_make nano
	chroot_make nasm
	# PERL CPAN packages
	chroot_make Archive-Zip		# OpenVPN
	chroot_make URI
	chroot_make HTML-Tagset
	chroot_make HTML-Parser
	chroot_make Compress-Zlib
	chroot_make DBI
	chroot_make DBD-SQLite
	chroot_make Digest-SHA1
	chroot_make Digest-HMAC
	chroot_make GD
	chroot_make libwww-perl
	chroot_make Locale-Maketext-Gettext
	chroot_make Net-IP
	chroot_make Net-DNS
	chroot_make Net-Netmask		# OpenVPN
	chroot_make Net-IPv4Addr
	chroot_make Net-SSLeay
	chroot_make Text-Parsewords
	chroot_make XML-Parser
	chroot_make XML-Simple
	# end of CPAN
	chroot_make ntp
	chroot_make openldap
	chroot_make openssh
	chroot_make openswan
	chroot_make openvpn
	chroot_make pptp
	chroot_make rrdtool
	chroot_make setserial
	#chroot_make smartmontools	# DEBUG -- wants libstdc++
	chroot_make sqlite
	chroot_make squid
	chroot_make squid-graph
	chroot_make strace
	chroot_make tcpdump
	chroot_make traceroute
	chroot_make ulogd
	chroot_make vlan
	chroot_make wireless_tools
	chroot_make 3c5x9setup
} # End of ipcop_build()



#########################################################################################################
# This builds the entire stage "misc"									#
#########################################################################################################
misc_build()
{
	beautify build_stage "Building miscellaneous"

	PATH=/${TOOLS_DIR}/usr/bin:/bin:/usr/bin:/sbin:/usr/sbin:/${TOOLS_DIR}/bin:/usr/${MACHINE_REAL}-linux/bin
	STAGE_ORDER=04
	STAGE=misc

	${MKDIR} ${BASEDIR}/log_${MACHINE}/04_misc

	LOGFILE="${BASEDIR}/log_${MACHINE}/_build_04_misc.log"
	export LOGFILE

	chroot_make cdrtools
	chroot_make memtest
	chroot_make syslinux
	chroot_make macutils
	chroot_make rsrce
	chroot_make miboot
	chroot_make mtools
	chroot_make parted
	chroot_make slang
	chroot_make newt
	chroot_make busybox
	chroot_make ipcop-lang
	chroot_make discover
	chroot_make Python
	chroot_make mklibs
	chroot_make klibc
	#chroot_make gdb		# DEBUG -- requested by Robert Kerr for debugging. Enable only in your tree
} # End of misc_build()



#########################################################################################################
# This builds the entire stage "packages"								#
#########################################################################################################
packages_build()
{
	beautify build_stage "Building packages"

	PATH=/${TOOLS_DIR}/usr/bin:/bin:/usr/bin:/sbin:/usr/sbin:/${TOOLS_DIR}/bin:/usr/${MACHINE_REAL}-linux/bin
	STAGE_ORDER=05
	STAGE=packages

	${MKDIR} ${BASEDIR}/log_${MACHINE}/05_packages

	LOGFILE="${BASEDIR}/log_${MACHINE}/_build_05_packages.log"
	export LOGFILE

	chroot_make stage5
	chroot_make installer
	chroot_make initramfs
	if [ x"${SKIP_FLOPPY_IMAGES}" != x"yes" ]; then
		chroot_make boot.img
	else
		echo "Skip floppy images"
	fi
	#chroot_make avmdrv

	# Create update for this version
	#echo "Building update ${VERSION} tgz" | tee -a ${LOGFILE}
	#tar -cz -C ${LFS} --files-from=${BASEDIR}/updates/${VERSION}/ROOTFILES.${MACHINE}-${VERSION} -f ${BASEDIR}/updates/${VERSION}/patch.tar.gz --exclude='#*';
	#chmod 755 ${BASEDIR}/updates/${VERSION}/setup
	#tar -cz -C ${BASEDIR}/updates/${VERSION} -f ${LFS}/install/images/update-${VERSION}.tgz patch.tar.gz setup information

	chroot_make packages_list PASS=1
	chroot_make cdrom
	chroot_make netboot
	chroot_make packages_list PASS=2

	if [ x"${SKIP_USB_IMAGES}" != x"yes" ]; then
		chroot_make usb-key
	else
		echo "Skip usb images to save time"
	fi

	# Cleanup
	stdumount
	${SUDO} ${RM} ${LFS}/tmp/*

	cd ${PWD}
} # End of packages_build()



#########################################################################################################
# This downloads all the external sources								#
#########################################################################################################
loadsrc()
{
	LOGFILE=${BASEDIR}/log_${MACHINE}/_build_00_preparation.log

	${MKDIR} ${BASEDIR}/log_${MACHINE}
	${MKDIR} ${BASEDIR}/cache
	local COUNTER=0
	local ALL_PACKAGES=0
	local ARCHES=${1}
	local FINISHED=0
	local WGET_FAIL_COUNTER=0
	local MD5SUM_FAIL_COUNTER=0
	declare -a FAILED
	cd ${BASEDIR}/lfs

	if [ x"${ARCHES}" == x"all" ]; then
		echo -ne "${BOLD}*** Prefetching files for all supported architectures"
		echo -ne "${SET_ARCH_COL}   arch${SET_WGET_COL}  wget${SET_RESULT_COL} md5sum${NORMAL}\n"
	else
		echo -ne "${BOLD}*** Prefetching files for ${MACHINE} only"
		echo -ne "${SET_ARCH_COL}   arch${SET_WGET_COL}  wget${SET_RESULT_COL} md5sum${NORMAL}\n"
	fi

	ALL_PACKAGES=`grep -l -E "^objects.*=" * | wc -l`

	for i in `grep -l -E "^objects.*=" *`
	do
		grep -E "^HOST_ARCH.*${MACHINE}|^HOST_ARCH.*all" ${i} >/dev/null

		if [ $? -eq 0 -o x"${ARCHES}" == x"all" ]; then
			COUNTER=$[ ${COUNTER} + 1 ]
			echo -ne "${i}"
			local HOST_ARCH=`grep "^HOST_ARCH" ${i} | sed "s,^HOST_ARCH.*= ,,"`

			if echo "${HOST_ARCH}" | grep -E "," > /dev/null 2>&1; then
				HOST_ARCH="some"
			fi

			beautify message ARCH ${HOST_ARCH}

			make -s -f ${i} MACHINE=${MACHINE} LFS_BASEDIR=${BASEDIR} MESSAGE="${i}\t" download >> ${LOGFILE} 2>&1
			if [ ${?} -ne 0 ]; then
				beautify message FAIL wget
				echo "${1} : wget error in '${i}'" >> ${LOGFILE} 2>&1
				FINISHED=0
				FAILED[${WGET_FAIL_COUNTER}]=${i}
				WGET_FAIL_COUNTER=$[ ${WGET_FAIL_COUNTER} + 1]
			else
				beautify message DONE wget
			fi

			make -s -f ${i} MACHINE=${MACHINE} LFS_BASEDIR=${BASEDIR} MESSAGE="${i}\t " md5 >> ${LOGFILE} 2>&1
			if [ ${?} -ne 0 ]; then
				beautify message FAIL
				MD5SUM_FAIL_COUNTER=$[ ${MD5SUM_FAIL_COUNTER} + 1]
			else
				beautify message DONE
			fi
		else
			echo "${i} : skipping '${i}' on ${MACHINE}" >> ${LOGFILE} 2>&1
		fi
	done

	if [ ${#FAILED[*]} -gt 0 ]; then
		echo -ne "${BOLD}*** Retrying failed files${SET_WGET_COL}  wget${SET_RESULT_COL} md5sum${NORMAL}\n"

		for c in `seq 2 ${MAX_RETRIES}`
		do
			for i in ${FAILED[@]}
			do
				if [ ${FINISHED} -eq 1 ]; then
					break
				fi

				FINISHED=1

				echo -ne "${BOLD}${i} (Attempt: ${c})${NORMAL}"
				make -s -f ${i} MACHINE=${MACHINE} LFS_BASEDIR=${BASEDIR} MESSAGE="${i}\t (${c}/${MAX_RETRIES})" download >> ${LOGFILE} 2>&1
				if [ ${?} -ne 0 ]; then
					beautify message FAIL wget
					echo "${1} : wget error in '${i}'" >> ${LOGFILE} 2>&1
					FINISHED=0
				else
					beautify message DONE wget
					WGET_FAIL_COUNTER=$[ ${WGET_FAIL_COUNTER} - 1]
				fi

				make -s -f ${i} MACHINE=${MACHINE} LFS_BASEDIR=${BASEDIR} MESSAGE="${i}\t " md5 >> ${LOGFILE} 2>&1
				if [ ${?} -ne 0 ]; then
					beautify message FAIL
				else
					beautify message DONE
					MD5SUM_FAIL_COUNTER=$[ ${MD5SUM_FAIL_COUNTER} - 1]
				fi
			done
		done
	fi

	beautify message INFO "Attempted to download ${COUNTER} of ${ALL_PACKAGES} packages"

	if [ ${WGET_FAIL_COUNTER} -gt 0 -o ${MD5SUM_FAIL_COUNTER} -gt 0 ]; then
		echo -ne "${BOLD}*** ${WGET_FAIL_COUNTER} package(s) could not be downloaded${NORMAL}\n"
		echo -ne "${BOLD}*** ${MD5SUM_FAIL_COUNTER} package(s) had incorrect md5sum${NORMAL}\n"
		return 1
	else
		echo -ne "${BOLD}*** All packages successfully downloaded with correct md5 sums${NORMAL}\n"
		return 0
	fi

	unset FAILED
	cd - > /dev/null
} # End of loadsrc()



#########################################################################################################
# This is the function that will create the toolchain tar archive					#
#########################################################################################################
package_toolchain()
{
	if [ -f ${BASEDIR}/log_${MACHINE}/02_base/stage2 ]; then
		exiterror "Too late, the toolchain has been altered and made unfit for future rebuilding"
	else
		TOOLCHAINFILES="01_toolchain.lst"
		echo "*** Packaging the ${MACHINE} toolchain" >> ${LOGFILE}
		echo -ne "${BOLD}*** Packaging the ${MACHINE} toolchain${NORMAL}\n"

		# Remove the man pages from the toolchain
		${RM} /${TOOLS_DIR}/man /${TOOLS_DIR}/usr/man

		echo "Creating the ${MACHINE} toolchain tar archive" >> ${LOGFILE}
		echo  -ne "Creating the ${MACHINE} toolchain tar archive"
		cd ${BASEDIR} && tar --create --gzip --verbose \
					--exclude="log_${MACHINE}/_build_0[0,2-6]_*.log" \
					--exclude="log_${MACHINE}/_build_${TOOLCHAINFILES}" \
					--exclude="log_${MACHINE}/0[2-6]_*" \
					--file=cache/${TOOLCHAINNAME} \
					build_${MACHINE}/${TOOLS_DIR} \
					log_${MACHINE} > log_${MACHINE}/_build_${TOOLCHAINFILES}

		if [ $? -eq 0 ]; then
			beautify message DONE
		else
			beautify message FAIL
		fi

		echo "Calculating the ${MACHINE} toolchain tar archive md5sum" >> ${LOGFILE}
		echo -ne "Calculating the ${MACHINE} toolchain tar archive md5sum"
		md5sum cache/${TOOLCHAINNAME} > cache/${TOOLCHAINNAME}.md5

		if [ $? -eq 0 ]; then
			beautify message DONE
		else
			beautify message FAIL
		fi
	fi

} # End of package_toolchain()

#########################################################################################################
#########################################################################################################
# End of BLOCK 2 -- Functions										#
#########################################################################################################
#########################################################################################################




#########################################################################################################
#########################################################################################################
# BLOCK 3 -- THIS IS WHERE EXECUTION STARTS								#
#########################################################################################################
#########################################################################################################

# See  what we're supposed to do
ACTION=${1}
case "${ACTION}" in
build)
	# First check if we meet the build environment requirements
	check_build_env

	# not running building tests by default
	RUNNING_TEST='no'
	if [ "${2}" -a x"${2}" == x"test" ]; then
		RUNNING_TEST='yes'
	fi

	# On first build attempt, better prefetch everything first.
	# This will allow to find packages URL that may have moved as soon as possible.
	if [ ! -d ${BASEDIR}/cache ]; then
		beautify message WARN "You should use './make.sh prefetch' to load all files before building"
		loadsrc
	fi

	echo -ne "Building ${BOLD}${NAME}-${VERSION}${NORMAL} for ${BOLD}${MACHINE} on ${MACHINE_REAL}${NORMAL}\n"

	if [ -f ${BASEDIR}/log_${MACHINE}/01_toolchain/strip ]; then
		prepareenv
		beautify message DONE "Stage toolchain already built (found log_${MACHINE}/01_toolchain/strip)"
	elif [ -f ${BASEDIR}/cache/${TOOLCHAINNAME} -a -f ${BASEDIR}/cache/${TOOLCHAINNAME}.md5 ]; then
		echo -ne "${BOLD}*** Restore from ${TOOLCHAINNAME}${NORMAL}\n"
		echo -ne "Checking md5sum"
		TOOLCHAIN_MD5_FOUND=`md5sum ${BASEDIR}/cache/${TOOLCHAINNAME} | awk '{print $1}'`
		TOOLCHAIN_MD5_NEEDED=`cat ${BASEDIR}/cache/${TOOLCHAINNAME}.md5 | awk '{print $1}'`

		if [ x"${TOOLCHAIN_MD5_FOUND}" == x"${TOOLCHAIN_MD5_NEEDED}" ]; then
			# toolchain look good, use it
			beautify message DONE

			echo -ne "md5sum matches : "
			beautify message INFO "${TOOLCHAIN_MD5_FOUND}"

			echo -ne "Unpacking toolchain ${TOOLCHAINNAME}"
			tar --no-same-owner --group=${CURRENT_USER_GROUP} -zxf ${BASEDIR}/cache/${TOOLCHAINNAME} \
				-C ${BASEDIR}

			if [ ${?} -eq 0 ]; then
				beautify message DONE
			else
				exiterror "${TOOLCHAINNAME} could not be unpacked. Check downloaded package"
			fi

			# Now prepare the environment
			prepareenv

			# in case of a package more recent in lfs than in the package,
			# uncomment toolchain_build to compile the newer (and hope it will run)
			#toolchain_build
		else
			exiterror "${TOOLCHAINNAME} md5 did not match. Check downloaded package"
		fi
	else
		echo -ne "${BOLD}*** Full toolchain compilation${NORMAL}\n"
		prepareenv

		# Check if host can build the toolchain
		check_toolchain_prerequisites

		# Now build the toolchain
		toolchain_build

		# Now create the toolchain tar archive
		package_toolchain
	fi

	base_build

	ipcop_build

	misc_build

	packages_build

	# Cleanup
	stdumount
	${RM} ${RUNNING_MARKER}
	${RM} ${FAILED_MARKER}
	${SUDO} ${RM} ${LFS}/tmp/*
	${SUDO} ${MV} ${BASEDIR}/build_${MACHINE}/${INSTALLER_DIR}/images/${SNAME}-${VERSION}-install-* ${BASEDIR}/

	cd ${PWD}

	#temporary comment while alpha stage:
	echo ""
	echo "Burn a CD, write ${SNAME}-${VERSION}-boot.img to floppy or use pxe to boot."
	echo -ne "NOTE: Burn the ISO at low speeds (4x is safe) because older CDROMs, especially "
	echo "in old sparc and powerpc machines can't read CD-Rs burnt at higher speeds."
	echo "... and all this hard work for this:"
	${DU} -bsh ${BASEDIR}/${SNAME}-${VERSION}-install-cd.${MACHINE}.iso
	BUILDTIME=$[`date +"%s"` - ${BUILDSTART}]
	BUILDMINUTES=$[${BUILDTIME} / 60]
	BUILDSECONDS=$[${BUILDTIME} - ${BUILDMINUTES}*60]
	echo "... which took: ${BUILDMINUTES} minutes and ${BUILDSECONDS} seconds"
	;;
shell)
	# First check if we meet the build environment requirements
	check_build_env

	# Enter a shell inside LFS chroot
	prepareenv
	# on sparc64 and ppc64, we don't want to set WRAPPER_32BIT to configure the kernel
	[ x"${MACHINE_REAL}" == x"sparc64" ] && unset WRAPPER_32BIT
	[ x"${MACHINE_REAL}" == x"ppc64" ] && unset WRAPPER_32BIT
	entershell
	;;
changelog)
	# First check if we meet the build environment requirements
	check_build_env

	echo "Building doc/Changelog from Subversion"
	# svn2cl script come from http://ch.tudelft.nl/~arthur/svn2cl/

	${BASEDIR}/tools/svn2cl/svn2cl.sh -r 'COMMITTED:3' -o ${BASEDIR}/doc/ChangeLog
	echo
	echo "Commit the change now to update Subversion"
	;;
check)
	# First check if we meet the build environment requirements
	check_build_env

	echo "Checking sources files availability on the web"
	if [ ! -d ${DIR_CHK} ]; then
		${MKDIR} ${DIR_CHK}
	fi

	FINISHED=0
	cd ${BASEDIR}/lfs
	for c in `seq ${MAX_RETRIES}`
	do
		if [ ${FINISHED} -eq 1 ]; then
			break
		fi

		FINISHED=1
		cd ${BASEDIR}/lfs
		for i in *
		do
			if [ -f "${i}" -a x"${i}" != x"Config" ]; then
				make -s -f ${i} MACHINE=${MACHINE} LFS_BASEDIR=${BASEDIR} LFS=${LFS} \
				MESSAGE="${i}\t (${c}/${MAX_RETRIES})" check
				if [ ${?} -ne 0 ]; then
					echo "Check : wget error in lfs/${i}"
					FINISHED=0
				fi
			fi
		done
	done

	cd - > /dev/null
	;;
check_versions)
		echo -ne "${BOLD}*** Comparing LFS-${LFS_BRANCH} & ${NAME}-${VERSION}"
		echo -ne "${SET_REQUIRED_COL}      LFS${SET_FOUND_COL}    IPCOP"
		echo -ne "${SET_RESULT_COL} status${NORMAL}\n"

		wget -q ${LFS_PACKAGES_URL} -O - \
			| grep 'span class="term"' \
			| tr 'A-Z' 'a-z' \
			| sed	-e 's,berkeley ,,g' \
				-e 's,^.*span class="term">\(.*\) (\([0-9]*[^ ]*\)).*$,\1 \2,g' \
			| grep -v -E "documentation|add-on|lfs-bootscripts|man-|configuration|optional" \
			| while IFS= read -r LINE
			do
				PKG_NAME=`echo ${LINE} | awk '{print $1}'`

				if [ ! -f "${BASEDIR}/lfs/${PKG_NAME}" ]; then
					beautify message INFO "${PKG_NAME} not used in ${SNAME}-${VERSION}"
				else
					PKG_LFS=`echo ${LINE} | cut -d" " -f2`
					if [ x"${PKG_NAME}" == x"linux" ]; then
						PKG_IPCOP=`grep ^PATCHLEVEL ${BASEDIR}/lfs/${PKG_NAME}|awk '{print $3}'`
					else
						PKG_IPCOP=`grep ^VER ${BASEDIR}/lfs/${PKG_NAME} | awk '{print $3}'`
					fi

					check_version "${PKG_NAME}" ${PKG_LFS} ${PKG_IPCOP}
					unset PKG_NAME PKG_LFS PKG_IPCOP
				fi
			done

		beautify message INFO "Failures are not necessarily bad."
	;;
checkclean)
	# First check if we meet the build environment requirements
	check_build_env

	echo "Erasing sources files availability tags"
	${RM} ${DIR_CHK}/*
	;;
clean)
	# First check if we meet the build environment requirements
	check_build_env

	echo -ne "Cleaning ${BOLD}${MACHINE}${NORMAL} buildtree${SET_RESULT_COL}"

	stdumount

	${SUDO} ${RM} ${BASEDIR}/build_${MACHINE}
	${SUDO} ${RM} ${BASEDIR}/log_${MACHINE}
	${SUDO} ${RM} ${BASEDIR}/updates/${VERSION}/patch.tar.gz

	if [ -h /${TOOLS_DIR} ]; then
		${SUDO} ${RM} /${TOOLS_DIR}
	fi

	beautify message DONE
	;;
dist)
	# First check if we meet the build environment requirements
	check_build_env

	echo "Building source package from CVS, list of changed files, MD5 of release files"
	if [ ! -s ${BASEDIR}/doc/CVS/Tag ]; then
		BRANCH=""
		BRANCHOPTS="-D `date +'%Y-%m-%d'`"
	else
		BRANCH=`cat ${BASEDIR}/doc/CVS/Tag`
		BRANCH=${BRANCH:1}
		BRANCHOPTS="-r ${BRANCH}"
	fi

	${RM} ${BASEDIR}/doc/release.txt
	${SUDO} ${RM} ${LFS}/tmp/${SNAME}-${VERSION}

	cd ${LFS}/tmp
	# build sources tgz
	echo "Export tree ${BRANCH} ${SNAME}-${VERSION}"
	# SUDO DEBUG: This needs to be configured for sudo
	${SUDO} cvs -z3 -d `cat ${BASEDIR}/CVS/Root` export ${BRANCHOPTS} ipcop

	if [ ${?} -eq 0 ]; then
		${SUDO} ${MV} ipcop ${SNAME}-${VERSION}
		tar cfz ${BASEDIR}/sources-${SNAME}-${VERSION}.tgz ${SNAME}-${VERSION}
		cd ${BASEDIR}
		md5sum update-${VERSION}.tgz ${SNAME}-${VERSION}.${MACHINE}.iso fcdsl-${VERSION}.tgz sources-${SNAME}-${VERSION}.tgz \
			> ${BASEDIR}/doc/release.txt

		if [ ! -d ${LFS}/tmp/${PREVIOUSTAG} ]; then
			# export previous version to be compared with actual,
			#this help to check wich files need to go in update
			cd ${LFS}/tmp
			echo "Export tree ${PREVIOUSTAG}"
			# SUDO DEBUG: This needs to be configured for sudo
			${SUDO} cvs -z3 -d `cat ${BASEDIR}/CVS/Root` export -r ${PREVIOUSTAG} ipcop
			${SUDO} ${MV} ipcop ${PREVIOUSTAG}
		fi

		if [ -d ${LFS}/tmp/${PREVIOUSTAG} -o -d ${LFS}/tmp/${SNAME}-${VERSION} ]; then
			cd ${LFS}/tmp
			echo "diff ${PREVIOUSTAG} <> ${BRANCH} ${SNAME}-${VERSION} >doc/updated-sources.txt"
			diff -rq ${PREVIOUSTAG} ${SNAME}-${VERSION} > ${BASEDIR}/doc/updated-sources.txt
			${MV} ${BASEDIR}/doc/updated-sources.txt ${BASEDIR}/doc/updated-sources.bak
			sed -e "s+Files ${PREVIOUSTAG}\/++" \
				-e "s+ and .*$++" \
				-e "s+src/rc.d+etc/rc.d+" \
				-e "s+^langs/+var/ipcop/langs/+" \
				-e "s+html/cgi-bin+home/httpd/cgi-bin+" ${BASEDIR}/doc/updated-sources.bak \
				> ${BASEDIR}/doc/updated-sources.txt
			${RM} ${BASEDIR}/doc/updated-sources.bak
		fi
	fi
	;;
files_check)
	# Check if we have yet build
	if [ ! -f ${BASEDIR}/doc/${NAME}-${VERSION}-all-files-list.${MACHINE}.txt ]; then
		beautify message FAIL "${NAME}-${VERSION}-all-files-list.${MACHINE}.txt not found, you need to build first"
		exit 1
	fi
	if [ ! -d ${BASEDIR}/log_${MACHINE}/05_packages ]; then
		beautify message FAIL "log_${MACHINE}/05_packages not found, you need to build first"
		exit 1
	fi
	check_build_env
	prepareenv
	PATH=/${TOOLS_DIR}/usr/bin:/bin:/usr/bin:/sbin:/usr/sbin:/${TOOLS_DIR}/bin:/usr/${MACHINE_REAL}-linux/bin
	export PATH
	chroot_make files_check
	stdumount
	;;
prefetch)
	# First check if we meet the build environment requirements
	check_build_env

	# Download all the packages marked as OTHER_SRC=yes
	if [ "${2}" -a x"${2}" == x"all" ]; then
		loadsrc all
	else
		loadsrc
	fi

	if [ $? -eq 1 ]; then
		beautify message WARN "Some files in ${BASEDIR}/cache failed to download or have incorrect md5 sums"
	fi
	;;
othersrc)
	# First check if we meet the build environment requirements
	check_build_env

	loadsrc all

	if [ $? -eq 1 ]; then
		beautify message WARN "Some files in ${BASEDIR}/cache failed to download or have incorrect md5 sums"
	else
		# Create some progress markers
		${RM} ${FAILED_MARKER}
		${RM} ${RUNNING_MARKER}
		touch ${RUNNING_MARKER}

		echo -ne "${BOLD}*** Creating othersrc tarball. This may take some time${NORMAL}\n"
		echo -ne "${OTHERSRC}"

		cd ${BASEDIR}/cache && (if tar -c --files-from=${BASEDIR}/log_${MACHINE}/_build_06_othersrc-list.log \
			-jf ${BASEDIR}/${OTHERSRC} && md5sum ${BASEDIR}/${OTHERSRC} > ${BASEDIR}/${OTHERSRC}.md5; then \
					${RM} ${RUNNING_MARKER}; \
				else \
					${RM} ${RUNNING_MARKER}; \
					touch ${FAILED_MARKER}; \
				fi ) &

	  PKG_TIME_START=`date +%s`
		[ x"${SHOW_PROGRESS}" == x"yes" ] && show_progress

		# Wait until the child processes have finished
		wait

		if [ -f ${FAILED_MARKER} ]; then
			${RM} ${FAILED_MARKER}
			beautify message FAIL
		else
			beautify message DONE
		fi

		cd - > /dev/null
	fi
	;;
toolchain)
	# First check if we meet the build environment requirements
	check_build_env

	# not running building tests by default
	RUNNING_TEST='no'
	if [ "${2}" -a x"${2}" == x"test" ]; then
		RUNNING_TEST='yes'
	fi

	# Prepare the environment
	prepareenv

	# Check if host can build the toolchain
	check_toolchain_prerequisites

	# Now build the toolchain
	toolchain_build

	# Since we're stopping here, run stdumount
	stdumount

	# Now create the toolchain tar archive
	package_toolchain
	BUILDTIME=$[`date +"%s"` - ${BUILDSTART}]
	BUILDMINUTES=$[${BUILDTIME} / 60]
	BUILDSECONDS=$[${BUILDTIME} - ${BUILDMINUTES}*60]
	echo "took: ${BUILDMINUTES} minutes and ${BUILDSECONDS} seconds"
	;;
gettoolchain)
	# First check if we meet the build environment requirements
	check_build_env

	URL_SFNET=`grep URL_SFNET lfs/Config | awk '{ print $3 }'`
	echo "Load toolchain tar.gz for ${MACHINE}"

	cd ${BASEDIR}/cache
	wget -c -nv ${URL_SFNET}/ipcop/${TOOLCHAINNAME} ${URL_SFNET}/ipcop/${TOOLCHAINNAME}.md5

	if [ ${?} -ne 0 ]; then
		echo -ne "Error downloading toolchain for ${MACHINE} machine"
		beautify message FAIL
		echo "Precompiled toolchain not always available for every MACHINE"
	else
		if [ "`md5sum ${TOOLCHAINNAME} | awk '{print $1}'`" = "`cat ${TOOLCHAINNAME}.md5 | awk '{print $1}'`" ]; then
			beautify message DONE
			echo "Toolchain md5 ok"
		else
			exiterror "${TOOLCHAINNAME}.md5 did not match, check downloaded package"
		fi
	fi
	;;
language)
	# First check if we meet the build environment requirements
	check_build_env

	${BASEDIR}/tools/gen_strings.pl ${BASEDIR} langs/
	;;
*)
	echo "Usage: ${0} {prefetch|build|clean|gettoolchain}"
	cat doc/make.sh-usage
	exit 1
	;;
esac

#########################################################################################################
#########################################################################################################
# End of BLOCK 3 -- THIS IS WHERE EXECUTION ENDS							#
#########################################################################################################
#########################################################################################################
