#!/usr/bin/perl
#
# This file is part of the IPCop Firewall.
#
# IPCop is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# IPCop is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with IPCop; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
#
# (c) 2001-2009 The IPCop Team
#
# Over the years many people have changed and contributed to this file.
# Check CVS and SVN for specifics.
#
# $Id: dhcp.cgi 2771 2009-05-02 22:41:23Z owes $
#

use strict;

# enable only the following on debugging purpose
#use warnings; no warnings 'once';
#use CGI::Carp 'fatalsToBrowser';

require '/usr/lib/ipcop/general-functions.pl';
require '/usr/lib/ipcop/lang.pl';
require '/usr/lib/ipcop/header.pl';

&Header::showhttpheaders();

my %dhcpsettings = ();
my %netsettings  = ();
my %mainsettings = ();
my %timesettings = ();

my $buttontext       = $Lang::tr{'add'};
my $errormessage     = '';
my $error_save_main  = '';
my $error_save_fixed = '';
my $warnmessage      = '';

# owes: TODO add Blue only when defined
my @INTERFACEs = ('GREEN', 'BLUE');
my $interface;
my $counter;
my $enabled_count = 0;
my $key;
my $line;
my %checked = ();
my $debug   = 0;

my $disable_main  = 0;          # 1 = only show (non-editable) some vital information in the main box
my $disable_fixed = 1;          # 1 = only show fixed leases, 0 = fields to add fixed lease become usable

# get IPCop settings
&General::readhash('/var/ipcop/ethernet/settings', \%netsettings);
&General::readhash('/var/ipcop/main/settings',     \%mainsettings);
&General::readhash('/var/ipcop/time/settings',     \%timesettings);

# main settings
foreach $interface (@INTERFACEs) {
    for ($counter = 1; $counter <= 1; $counter++) {
        $dhcpsettings{"ENABLED_${interface}_${counter}"}            = 'off';
        $dhcpsettings{"ENABLED_BOOTP_${interface}_${counter}"}      = 'off';
        $dhcpsettings{"START_ADDR_${interface}_${counter}"}         = '';
        $dhcpsettings{"END_ADDR_${interface}_${counter}"}           = '';
        $dhcpsettings{"DOMAIN_NAME_${interface}_${counter}"}        = '';
        $dhcpsettings{"DEFAULT_LEASE_TIME_${interface}_${counter}"} = '60';
        $dhcpsettings{"DNS1_${interface}_${counter}"}               = $netsettings{"${interface}_${counter}_ADDRESS"};
        $dhcpsettings{"DNS2_${interface}_${counter}"}               = '';
        $dhcpsettings{"WINS1_${interface}_${counter}"}              = '';
        $dhcpsettings{"WINS2_${interface}_${counter}"}              = '';
        $dhcpsettings{"NTP1_${interface}_${counter}"}               = '';
        $dhcpsettings{"NTP2_${interface}_${counter}"}               = '';
    }
}
$dhcpsettings{'ACTION'}              = '';
$dhcpsettings{'SORT_FIXEDLEASELIST'} = '';
$dhcpsettings{'FIXED_ENABLED'}       = '';
$dhcpsettings{'FIXED_MAC'}           = '';
$dhcpsettings{'FIXED_IP'}            = '';
$dhcpsettings{'FIXED_REMARK'}        = '';
$dhcpsettings{'FIXED_HOSTNAME'}      = '';
$dhcpsettings{'FIXED_NEXTADDR'}      = '';
$dhcpsettings{'FIXED_FILENAME'}      = '';
$dhcpsettings{'FIXED_ROOTPATH'}      = '';
$dhcpsettings{'KEY_FIXED'}           = '';

# read the fixed leases
my @fixedleases;
open(FILE, '/var/ipcop/dhcp/fixedleases');
@fixedleases = <FILE>;
close(FILE);

&General::getcgihash(\%dhcpsettings);

&Header::openpage($Lang::tr{'dhcp configuration'}, 1, '');
&Header::openbigbox('100%', 'left', '', '');

###############
# DEBUG DEBUG
if ($debug) {
    &Header::openbox('100%', 'left', 'DEBUG');
    my $debugCount = 0;
    foreach my $line (sort keys %dhcpsettings) {
        print "$line = $dhcpsettings{$line}<br />\n";
        $debugCount++;
    }
    print "&nbsp;Count: $debugCount\n";
    &Header::closebox();
}

# DEBUG DEBUG
###############

if ($dhcpsettings{'ACTION'} eq 'SAVE_MAIN') {

    # Verify the options before writing anything
    foreach $interface (@INTERFACEs) {

        # Limit to 1 interface for now
        for ($counter = 1; $counter <= 1; $counter++) {
            if ($dhcpsettings{"ENABLED_${interface}_${counter}"} eq 'on') {

                # define START and END or leave both empty (for static leases only)
                if ($dhcpsettings{"START_ADDR_${interface}_${counter}"}) {
                    if (!&General::validip($dhcpsettings{"START_ADDR_${interface}_${counter}"})) {
                        $errormessage = "DHCP on ${interface}: " . $Lang::tr{'invalid start address'};
                        goto ERROR_SAVE_MAIN;
                    }
                    if (!$dhcpsettings{"END_ADDR_${interface}_${counter}"}) {
                        $errormessage = "DHCP on ${interface}: " . $Lang::tr{'invalid end address'};
                        goto ERROR_SAVE_MAIN;
                    }

                    if (
                        !&General::IpInSubnet(
                            $dhcpsettings{"START_ADDR_${interface}_${counter}"},
                            $netsettings{"${interface}_${counter}_NETADDRESS"},
                            $netsettings{"${interface}_${counter}_NETMASK"}
                        )
                        )
                    {
                        $errormessage = "DHCP on ${interface}: " . $Lang::tr{'invalid start address'};
                        goto ERROR_SAVE_MAIN;
                    }
                }

                if ($dhcpsettings{"END_ADDR_${interface}_${counter}"}) {
                    if (!&General::validip($dhcpsettings{"END_ADDR_${interface}_${counter}"})) {
                        $errormessage = "DHCP on ${interface}: " . $Lang::tr{'invalid end address'};
                        goto ERROR_SAVE_MAIN;
                    }
                    if (!$dhcpsettings{"START_ADDR_${interface}_${counter}"}) {
                        $errormessage = "DHCP on ${interface}: " . $Lang::tr{'invalid start address'};
                        goto ERROR_SAVE_MAIN;
                    }

                    if (
                        !&General::IpInSubnet(
                            $dhcpsettings{"END_ADDR_${interface}_${counter}"},
                            $netsettings{"${interface}_${counter}_NETADDRESS"},
                            $netsettings{"${interface}_${counter}_NETMASK"}
                        )
                        )
                    {
                        $errormessage = "DHCP on ${interface}: " . $Lang::tr{'invalid end address'};
                        goto ERROR_SAVE_MAIN;
                    }
                }

                # Swap Start and End IF End is less than Start
                # only test last octet
                my @startoct = split (/\./, $dhcpsettings{"START_ADDR_${interface}_${counter}"});
                my @endoct   = split (/\./, $dhcpsettings{"END_ADDR_${interface}_${counter}"});
                if ( $endoct[3] < $startoct[3] ) {
                    ($dhcpsettings{"START_ADDR_${interface}_${counter}"},$dhcpsettings{"END_ADDR_${interface}_${counter}"}) =
                    ($dhcpsettings{"END_ADDR_${interface}_${counter}"},$dhcpsettings{"START_ADDR_${interface}_${counter}"});
                }

                # Lease time must be set and be numeric
                if (!($dhcpsettings{"DEFAULT_LEASE_TIME_${interface}_${counter}"} =~ /^\d+$/)) {
                    $errormessage =
                          "DHCP on ${interface}: "
                        . $Lang::tr{'invalid default lease time'} . ' '
                        . $dhcpsettings{'DEFAULT_LEASE_TIME_${interface}_${counter}'};
                    goto ERROR_SAVE_MAIN;
                }

                # Verify DNS1 and DNS2
                if ($dhcpsettings{"DNS1_${interface}_${counter}"}) {
                    if (!&General::validip($dhcpsettings{"DNS1_${interface}_${counter}"})) {
                        $errormessage = "DHCP on ${interface}: " . $Lang::tr{'invalid primary dns'};
                        goto ERROR_SAVE_MAIN;
                    }
                }
                if ($dhcpsettings{"DNS2_${interface}_${counter}"}) {
                    if (!&General::validip($dhcpsettings{"DNS2_${interface}_${counter}"})) {
                        $errormessage = "DHCP on ${interface}: " . $Lang::tr{'invalid secondary dns'};
                        goto ERROR_SAVE_MAIN;
                    }
                    if (!$dhcpsettings{"DNS1_${interface}_${counter}"}) {
                        $errormessage =
                            "DHCP on ${interface}: "
                            . $Lang::tr{'cannot specify secondary dns without specifying primary'};
                        goto ERROR_SAVE_MAIN;
                    }
                }

                # Verify WINS1 and WINS2
                if ($dhcpsettings{"WINS1_${interface}_${counter}"}) {
                    if (!&General::validip($dhcpsettings{"WINS1_${interface}_${counter}"})) {
                        $errormessage = "DHCP on ${interface}: " . $Lang::tr{'invalid wins address'};
                        goto ERROR_SAVE_MAIN;
                    }
                }
                if ($dhcpsettings{"WINS2_${interface}_${counter}"}) {
                    if (!&General::validip($dhcpsettings{"WINS2_${interface}_${counter}"})) {
                        $errormessage = "DHCP on ${interface}: " . $Lang::tr{'invalid wins address'};
                        goto ERROR_SAVE_MAIN;
                    }
                    if (!$dhcpsettings{"WINS1_${interface}_${counter}"}) {
                        $errormessage =
                            "DHCP on ${interface}: "
                            . $Lang::tr{'cannot specify secondary wins without specifying primary'};
                        goto ERROR_SAVE_MAIN;
                    }
                }

                # Verify NTP1 and NTP2
                if ($dhcpsettings{"NTP1_${interface}_${counter}"}) {
                    if (!&General::validip($dhcpsettings{"NTP1_${interface}_${counter}"})) {
                        $errormessage = "DHCP on ${interface}: " . $Lang::tr{'invalid primary ntp'};
                        goto ERROR_SAVE_MAIN;
                    }
                    if (   ($dhcpsettings{"NTP1_${interface}_${counter}"} eq $netsettings{"${interface}_${counter}_ADDRESS"})
                        && ($timesettings{'ENABLED_NTP'} ne 'on'))
                    {
                        $warnmessage =
                            "DHCP on ${interface}: " . $Lang::tr{'local ntp server specified but not enabled'};
                    }
                }
                if ($dhcpsettings{"NTP2_${interface}_${counter}"}) {
                    if (!&General::validip($dhcpsettings{"NTP2_${interface}_${counter}"})) {
                        $errormessage = "DHCP on ${interface}: " . $Lang::tr{'invalid secondary ntp'};
                        goto ERROR_SAVE_MAIN;
                    }
                    if (   ($dhcpsettings{"NTP2_${interface}_${counter}"} eq $netsettings{"${interface}_${counter}_ADDRESS"})
                        && ($timesettings{'ENABLED_NTP'} ne 'on'))
                    {
                        $warnmessage =
                            "DHCP on ${interface}: " . $Lang::tr{'local ntp server specified but not enabled'};
                    }
                    if (!$dhcpsettings{"NTP1_${interface}_${counter}"}) {
                        $errormessage =
                            "DHCP on ${interface}: "
                            . $Lang::tr{'cannot specify secondary ntp without specifying primary'};
                        goto ERROR_SAVE_MAIN;
                    }
                }
            }
        }    # interface count
    }    # foreach interface
    &writeconfig(1);
ERROR_SAVE_MAIN:
    $error_save_main = 'error' if ($errormessage);
}
else {
    &General::readhash('/var/ipcop/dhcp/settings', \%dhcpsettings);
}

if ($dhcpsettings{'ACTION'} eq $Lang::tr{'toggle enable disable'} . '_fixed') {

    # Toggle enable/disable field on specified fixed lease

    chomp(@fixedleases[ $dhcpsettings{'KEY_FIXED'} ]);
    my @temp = split(/\,/, @fixedleases[ $dhcpsettings{'KEY_FIXED'} ]);
    $temp[2] = $temp[2] eq 'on' ? '' : 'on';
    @fixedleases[ $dhcpsettings{'KEY_FIXED'} ] = join(',', @temp) . "\n";
    $dhcpsettings{'KEY_FIXED'} = '';    # forget we were editing something
    &writefixedleases(1);
}

if ($dhcpsettings{'ACTION'} eq $Lang::tr{'edit'} . '_fixed') {

    # Edit fields on specified fixed lease

    $disable_main  = 1;
    $disable_fixed = 0;

    chomp(@fixedleases[ $dhcpsettings{'KEY_FIXED'} ]);
    my @temp = split(/\,/, @fixedleases[ $dhcpsettings{'KEY_FIXED'} ]);
    $dhcpsettings{'FIXED_ENABLED'}  = $temp[2];
    $dhcpsettings{'FIXED_MAC'}      = $temp[0];
    $dhcpsettings{'FIXED_IP'}       = $temp[1];
    $dhcpsettings{'FIXED_REMARK'}   = $temp[6];
    $dhcpsettings{'FIXED_HOSTNAME'} = $temp[7];
    $dhcpsettings{'FIXED_NEXTADDR'} = $temp[3];
    $dhcpsettings{'FIXED_FILENAME'} = $temp[4];
    $dhcpsettings{'FIXED_ROOTPATH'} = $temp[5];
}

if ($dhcpsettings{'ACTION'} eq $Lang::tr{'remove'} . '_fixed') {

    # Remove a fixed lease

    splice(@fixedleases, $dhcpsettings{'KEY_FIXED'}, 1);
    $dhcpsettings{'KEY_FIXED'} = '';    # forget we were editing something
    &writefixedleases(1);
}

if ($dhcpsettings{'ACTION'} eq 'ADD_FIXED_LEASE') {

    # Button to add fixed lease was pressed

    $disable_main                  = 1;
    $disable_fixed                 = 0;
    $dhcpsettings{'FIXED_ENABLED'} = 'on';    # on per default
}

if ($dhcpsettings{'ACTION'} eq 'SAVE_FIXED_LEASE') {

    # Verify the options before writing anything

    $dhcpsettings{'FIXED_REMARK'} = &Header::cleanhtml($dhcpsettings{'FIXED_REMARK'});

    $dhcpsettings{'FIXED_MAC'} =~ tr/-/:/;
    if (!&General::validmac($dhcpsettings{'FIXED_MAC'})) {
        $errormessage = $Lang::tr{'invalid fixed mac address'};
        goto ERROR_SAVE_FIXED;
    }
    if (!&General::validip($dhcpsettings{'FIXED_IP'})) {
        $errormessage = $Lang::tr{'invalid fixed ip address'};
        goto ERROR_SAVE_FIXED;
    }

    my $insubnet = 0;
    # IP must be in green or blue network and dnsmasq must be active there
    foreach $interface (@INTERFACEs) {
        for ($counter = 1; $counter <= $netsettings{"${interface}_COUNT"}; $counter++) {
            next if ($dhcpsettings{"ENABLED_${interface}_${counter}"} ne 'on');

            if (&General::IpInSubnet($dhcpsettings{'FIXED_IP'}, 
                                        $netsettings{"${interface}_${counter}_NETADDRESS"}, 
                                        $netsettings{"${interface}_${counter}_NETMASK"})
               ) {
                $insubnet++;
            }
        }
    }
    if ($insubnet == 0) {
        $errormessage = $Lang::tr{'invalid fixed ip address'};
        goto ERROR_SAVE_FIXED;
    }

    # TODO: test for duplicate MAC addresses
    # Duplicate MAC is OK, as long as the to be assigned IP addresses are in different networks

    if ($dhcpsettings{'KEY_FIXED'} ne '') {

        # replace existing
        @fixedleases[ $dhcpsettings{'KEY_FIXED'} ] =
              "$dhcpsettings{'FIXED_MAC'},$dhcpsettings{'FIXED_IP'},$dhcpsettings{'FIXED_ENABLED'},"
            . "$dhcpsettings{'FIXED_NEXTADDR'},$dhcpsettings{'FIXED_FILENAME'},$dhcpsettings{'FIXED_ROOTPATH'},"
            . "$dhcpsettings{'FIXED_REMARK'},$dhcpsettings{'FIXED_HOSTNAME'}\n";
    }
    else {

        # add new
        unshift(@fixedleases,
                  "$dhcpsettings{'FIXED_MAC'},$dhcpsettings{'FIXED_IP'},$dhcpsettings{'FIXED_ENABLED'},"
                . "$dhcpsettings{'FIXED_NEXTADDR'},$dhcpsettings{'FIXED_FILENAME'},$dhcpsettings{'FIXED_ROOTPATH'},"
                . "$dhcpsettings{'FIXED_REMARK'},$dhcpsettings{'FIXED_HOSTNAME'}\n");
    }
    $dhcpsettings{'KEY_FIXED'} = '';    # forget we were editing something
    &writefixedleases(1);

ERROR_SAVE_FIXED:
    if ($errormessage) {
        $error_save_fixed = 'error';
        $disable_main     = 1;
        $disable_fixed    = 0;
    }
}

#
# Sorting of fixed leases
#
if ($ENV{'QUERY_STRING'} =~ /^FIXEDMAC|^FIXEDIP/) {
    my $newsort = $ENV{'QUERY_STRING'};
    my $act     = $dhcpsettings{'SORT_FIXEDLEASELIST'};

    #Reverse actual sort ?
    if ($act =~ $newsort) {
        my $rev = '';
        if ($act !~ 'Rev') {
            $rev = 'Rev';
        }
        $newsort .= $rev;
    }
    $dhcpsettings{'SORT_FIXEDLEASELIST'} = $newsort;
    &writeconfig(0);
    &writefixedleases(0);
}

#
# display box with warnmessage in case of warning
#
if ($warnmessage) {
    &Header::openbox('100%', 'left', $Lang::tr{'capswarning'}, 'warning');
    print "<font class='base'>$warnmessage&nbsp;</font>\n";
    &Header::closebox();
}

#
# display box with errormessage in case of error
#
if ($errormessage) {
    &Header::openbox('100%', 'left', "$Lang::tr{'error messages'}:", 'error');
    print "<font class='base'>$errormessage&nbsp;</font>\n";
    &Header::closebox();
}

#
# display box with main settings
#
&Header::openbox('100%', 'left', 'DHCP', $error_save_main);
print "<form method='post' name='frm_main' action='$ENV{'SCRIPT_NAME'}'>";
foreach $interface (@INTERFACEs) {
    for ($counter = 1; $counter <= $netsettings{"${interface}_COUNT"}; $counter++) {
        my $lc_interface = lc($interface);
        $checked{'ENABLED'}{'on'} =
            ($dhcpsettings{"ENABLED_${interface}_${counter}"} ne 'on') ? '' : "checked='checked'";
        $checked{'ENABLED_BOOTP'}{'on'} =
            ($dhcpsettings{"ENABLED_BOOTP_${interface}_${counter}"} ne 'on') ? '' : "checked='checked'";
        my $disable_text = '';
        $disable_text = "disabled='disabled'" if ($disable_main == 1);

        print <<END
<table width='100%'>
<tr>
    <td width='25%' class='boldbase'><span class='ipcop_iface_$lc_interface'><b>$Lang::tr{"$lc_interface"}</b></span></td>
    <td width='25%' class='base'>$Lang::tr{'enabled'}:<input type='checkbox' name='ENABLED_${interface}_${counter}' $checked{'ENABLED'}{'on'} $disable_text /></td>
    <td width='25%' class='base'>$Lang::tr{'ip address'}/$Lang::tr{'netmask'}:</td>
    <td width='25%' class='base'><b>$netsettings{"${interface}_${counter}_ADDRESS"}/$netsettings{"${interface}_${counter}_NETMASK"}</b></td>
</tr><tr>
    <td width='25%' class='base'>$Lang::tr{'start address'}:&nbsp;<img src='/blob.gif' alt='*' /></td>
    <td width='25%'><input type='text' name='START_ADDR_${interface}_${counter}' value='$dhcpsettings{"START_ADDR_${interface}_${counter}"}' $disable_text /></td>
    <td width='25%' class='base'>$Lang::tr{'end address'}:&nbsp;<img src='/blob.gif' alt='*' /></td>
    <td width='25%'><input type='text' name='END_ADDR_${interface}_${counter}' value='$dhcpsettings{"END_ADDR_${interface}_${counter}"}' $disable_text /></td>
</tr>
END
            ;
        if ($disable_main == 0) {
            print <<END
<tr>
    <td class='base'>$Lang::tr{'default lease time'}:</td>
    <td><input type='text' name='DEFAULT_LEASE_TIME_${interface}_${counter}' value='$dhcpsettings{"DEFAULT_LEASE_TIME_${interface}_${counter}"}' size='5' /></td>
    <td class='base'>$Lang::tr{'domain name suffix'}:&nbsp;<img src='/blob.gif' alt='*' /></td>
    <td><input type='text' name='DOMAIN_NAME_${interface}_${counter}' value='$dhcpsettings{"DOMAIN_NAME_${interface}_${counter}"}' /></td>
</tr><tr>
    <td>$Lang::tr{'dhcp allow bootp'}:</td>
    <td><input type='checkbox' name='ENABLED_BOOTP_${interface}_${counter}' $checked{'ENABLED_BOOTP'}{'on'} /></td>
    <td>&nbsp;</td><td>&nbsp;</td>
</tr><tr>
    <td class='base'>$Lang::tr{'primary dns'}:</td>
    <td><input type='text' name='DNS1_${interface}_${counter}' value='$dhcpsettings{"DNS1_${interface}_${counter}"}' /></td>
    <td class='base'>$Lang::tr{'secondary dns'}:&nbsp;<img src='/blob.gif' alt='*' /></td>
    <td><input type='text' name='DNS2_${interface}_${counter}' value='$dhcpsettings{"DNS2_${interface}_${counter}"}' /></td>
</tr><tr>
    <td class='base'>$Lang::tr{'primary ntp server'}:&nbsp;<img src='/blob.gif' alt='*' /></td>
    <td><input type='text' name='NTP1_${interface}_${counter}' value='$dhcpsettings{"NTP1_${interface}_${counter}"}' /></td>
    <td class='base'>$Lang::tr{'secondary ntp server'}:&nbsp;<img src='/blob.gif' alt='*' /></td>
    <td><input type='text' name='NTP2_${interface}_${counter}' value='$dhcpsettings{"NTP2_${interface}_${counter}"}' /></td>
</tr><tr>
    <td class='base'>$Lang::tr{'primary wins server address'}:&nbsp;<img src='/blob.gif' alt='*' /></td>
    <td><input type='text' name='WINS1_${interface}_${counter}' value='$dhcpsettings{"WINS1_${interface}_${counter}"}' /></td>
    <td class='base'>$Lang::tr{'secondary wins server address'}:&nbsp;<img src='/blob.gif' alt='*' /></td>
    <td><input type='text' name='WINS2_${interface}_${counter}' value='$dhcpsettings{"WINS2_${interface}_${counter}"}' /></td>
</tr>
END
                ;
        }
        print "</table><hr />";
    }
}

if ($disable_main == 1) {
    print "</form>";
}
else {
    print <<END
<table width='100%'>
<tr>
    <td class='base' width='55%'><img src='/blob.gif' align='top' alt='*' />&nbsp;$Lang::tr{'this field may be blank'}</td>
    <td width='40%' align='center'><input type='hidden' name='ACTION' value='SAVE_MAIN' /><input type='submit' name='SUBMIT' value='$Lang::tr{'save'}' /></td>
    <td width='5%' align='right'>
        <a href='${General::adminmanualurl}/services.html#services_dhcp' target='_blank'>
        <img src='/images/web-support.png' alt='$Lang::tr{'online help en'}' title='$Lang::tr{'online help en'}' /></a></td>
</tr>
</table>
</form>
END
        ;
}
&Header::closebox();

#
# display box with fixed leases
#
&Header::openbox('100%', 'left', "$Lang::tr{'current fixed leases'}:", $error_save_fixed);

if ($disable_fixed == 0) {
    $checked{'FIXED_ENABLED'}{'on'} = ($dhcpsettings{'FIXED_ENABLED'} ne 'on') ? '' : "checked='checked'";
    my $action_text = $Lang::tr{'add new lease'};

    # if KEY_FIXED is set, this is edit/update not add

    if ($dhcpsettings{'KEY_FIXED'} ne '') {
        $buttontext  = $Lang::tr{'update'};
        $action_text = $Lang::tr{'edit an existing lease'};
    }

    print <<END
<form method='post' name='frm_fixed' action='$ENV{'SCRIPT_NAME'}'>
<table width='100%' border='0'>
<tr>
    <td class='boldbase' colspan='3'><b>$action_text</b></td><td>&nbsp;</td>
</tr><tr>
    <td class='base'>$Lang::tr{'enabled'}:</td>
    <td><input type='checkbox' name='FIXED_ENABLED' $checked{'FIXED_ENABLED'}{'on'} /></td>
    <td colspan='2'>&nbsp;</td>
</tr><tr>
    <td width='25%' class='base'>$Lang::tr{'mac address'}:</td>
    <td width='25%'><input type='text' name='FIXED_MAC' value='$dhcpsettings{'FIXED_MAC'}' size='18' /></td>
    <td width='25%' class='base'>$Lang::tr{'ip address'}:</td>
    <td width='25%'><input type='text' name='FIXED_IP' value='$dhcpsettings{'FIXED_IP'}' size='18' /></td>
</tr><tr>
    <td class='base'>$Lang::tr{'hostname'} $Lang::tr{'or'} FQDN:&nbsp;<img src='/blob.gif' alt='*' /></td>
    <td colspan='3'><input type='text' name='FIXED_HOSTNAME' value='$dhcpsettings{'FIXED_HOSTNAME'}' size='40' /></td>
</tr><tr>
    <td width='25%' class='base'>$Lang::tr{'remark'}:&nbsp;<img src='/blob.gif' alt='*' /></td>
    <td colspan='3'><input type='text' name='FIXED_REMARK' value='$dhcpsettings{'FIXED_REMARK'}' size='40' /></td>
</tr><tr>
    <td colspan = '4'><b>$Lang::tr{'dhcp bootp pxe data'}</b></td>
</tr><tr>
    <td class='base'>filename:&nbsp;<img src='/blob.gif' alt='*' /></td>
    <td><input type='text' name='FIXED_FILENAME' value='$dhcpsettings{'FIXED_FILENAME'}' size='18' /></td>
    <td class='base'>root-path:&nbsp;<img src='/blob.gif' alt='*' /></td>
    <td><input type='text' name='FIXED_ROOTPATH' value='$dhcpsettings{'FIXED_ROOTPATH'}' size='18' /></td>
</tr><tr>
    <td class='base'>next-server:&nbsp;<img src='/blob.gif' alt='*' /></td>
    <td><input type='text' name='FIXED_NEXTADDR' value='$dhcpsettings{'FIXED_NEXTADDR'}' size='18' /></td>
    <td colspan='2'>&nbsp;</td>
</tr>
</table>
<hr />
<table width='100%'>
<tr>
    <td class='base' width='55%'><img src='/blob.gif' align='top' alt='*' />&nbsp;$Lang::tr{'this field may be blank'}
        &nbsp;&nbsp;$Lang::tr{'dhcp fixed lease help1'}</td>
    <td width='40%' align='center'><input type='hidden' name='ACTION' value='SAVE_FIXED_LEASE' />
        <input type='hidden' name='KEY_FIXED' value='$dhcpsettings{'KEY_FIXED'}' />
        <input type='submit' name='SUBMIT' value='$buttontext' /></td>
</tr>
</table>
</form>
END
        ;
}
else {
    print <<END
<form method='post' name='frm_fixed_add' action='$ENV{'SCRIPT_NAME'}'>
&nbsp;&nbsp;<img src='/images/null.gif' width='21' height='1' alt='' /><input type='hidden' name='ACTION' value='ADD_FIXED_LEASE' /><input type='submit' name='SUBMIT' value='$Lang::tr{'add new lease'}' />
</form>
END
        ;
}

# Add visual indicators to column headings to show sort order - EO
my $sortarrow1 = '';
my $sortarrow2 = '';

if ($dhcpsettings{'SORT_FIXEDLEASELIST'} eq 'FIXEDMACRev') {
    $sortarrow1 = $Header::sortdn;
}
elsif ($dhcpsettings{'SORT_FIXEDLEASELIST'} eq 'FIXEDMAC') {
    $sortarrow1 = $Header::sortup;
}
elsif ($dhcpsettings{'SORT_FIXEDLEASELIST'} eq 'FIXEDIPRev') {
    $sortarrow2 = $Header::sortdn;
}
else {
    $sortarrow2 = $Header::sortup;
}

print <<END
<hr /><table width='100%'>
<tr>
    <td width='13%' align='center' nowrap='nowrap'><a href='$ENV{'SCRIPT_NAME'}?FIXEDMAC'><b>$Lang::tr{'mac address'}</b></a> $sortarrow1</td>
    <td width='13%' align='center' nowrap='nowrap'><a href='$ENV{'SCRIPT_NAME'}?FIXEDIP'><b>$Lang::tr{'ip address'}</b></a> $sortarrow2</td>
    <td width='14%' class='boldbase' align='center'><b>$Lang::tr{'hostname'}</b></td>
    <td width='15%' align='center'><b>$Lang::tr{'remark'}</b></td>
    <td width='15%' class='boldbase' align='center'><b>next-server</b></td>
    <td width='15%' class='boldbase' align='center'><b>filename</b></td>
    <td width='15%' class='boldbase' align='center'><b>root-path</b></td>
    <td colspan='3' class='boldbase' align='center'><b>$Lang::tr{'action'}</b></td>
</tr>
END
    ;

$key = 0;
foreach $line (@fixedleases) {
    my $gif   = '';
    my $gdesc = '';

    chomp($line);
    my @temp = split(/\,/, $line);

    if ($temp[2] eq "on") {
        $gif   = 'on.gif';
        $gdesc = $Lang::tr{'click to disable'};
    }
    else {
        $gif   = 'off.gif';
        $gdesc = $Lang::tr{'click to enable'};
    }

    if ($dhcpsettings{'KEY_FIXED'} eq $key) {
        print "<tr bgcolor='${Header::colouryellow}'>";
    }
    elsif ($key % 2) {
        print "<tr bgcolor='${Header::table2colour}'>";
    }
    else {
        print "<tr bgcolor='${Header::table1colour}'>";
    }
    print <<END
<td align='center'>$temp[0]</td>
<td align='center'>$temp[1]</td>
<td align='center'>$temp[7]&nbsp;</td>
<td align='center'>$temp[6]&nbsp;</td>
<td align='center'>$temp[3]&nbsp;</td>
<td align='center'>$temp[4]&nbsp;</td>
<td align='center'>$temp[5]&nbsp;</td>

<td align='center'>
    <form method='post' name='frm_fixed_ted_$key' action='$ENV{'SCRIPT_NAME'}'>
    <input type='hidden' name='ACTION' value='$Lang::tr{'toggle enable disable'}_fixed' />
    <input type='image' name='$Lang::tr{'toggle enable disable'}' src='/images/$gif' alt='$gdesc' title='$gdesc' />
    <input type='hidden' name='KEY_FIXED' value='$key' />
    </form>
</td>

<td align='center'>
    <form method='post' name='frm_fixed_e_$key' action='$ENV{'SCRIPT_NAME'}'>
    <input type='hidden' name='ACTION' value='$Lang::tr{'edit'}_fixed' />
    <input type='image' name='$Lang::tr{'edit'}' src='/images/edit.gif' alt='$Lang::tr{'edit'}' title='$Lang::tr{'edit'}' />
    <input type='hidden' name='KEY_FIXED' value='$key' />
    </form>
</td>

<td align='center'>
    <form method='post' name='frm_fixed_r_$key' action='$ENV{'SCRIPT_NAME'}'>
    <input type='hidden' name='ACTION' value='$Lang::tr{'remove'}_fixed' />
    <input type='image' name='$Lang::tr{'remove'}' src='/images/delete.gif' alt='$Lang::tr{'remove'}' title='$Lang::tr{'remove'}' />
    <input type='hidden' name='KEY_FIXED' value='$key' />
    </form>
</td>
</tr>
END
        ;
    $key++;
}    # for all fixed leases

print "</table>";

# If the fixed leases file contains entries, print a legend
if ($key && $disable_fixed) {
    print <<END
<table>
<tr>
    <td class='boldbase'>&nbsp;<b>$Lang::tr{'legend'}:&nbsp;</b></td>
    <td><img src='/images/on.gif' alt='$Lang::tr{'click to disable'}' /></td>
    <td class='base'>$Lang::tr{'click to disable'}</td>
    <td>&nbsp;&nbsp;</td>
    <td><img src='/images/off.gif' alt='$Lang::tr{'click to enable'}' /></td>
    <td class='base'>$Lang::tr{'click to enable'}</td>
    <td>&nbsp;&nbsp;</td>
    <td><img src='/images/edit.gif' alt='$Lang::tr{'edit'}' /></td>
    <td class='base'>$Lang::tr{'edit'}</td>
    <td>&nbsp;&nbsp;</td>
    <td><img src='/images/delete.gif' alt='$Lang::tr{'remove'}' /></td>
    <td class='base'>$Lang::tr{'remove'}</td>
</tr>
</table>
END
        ;
}

&Header::closebox();

#
# display box with dynamic leases if we have one or more enabled interface(s)
#
$enabled_count = 0;
foreach $interface (@INTERFACEs) {
    for ($counter = 1; $counter <= $netsettings{"${interface}_COUNT"}; $counter++) {
        if ($dhcpsettings{"ENABLED_${interface}_${counter}"} eq 'on') {
            $enabled_count++;
        }
    }
}
if ($enabled_count > 0) {
    &General::PrintActualLeases;
}

&Header::closebigbox();
&Header::closepage();

#
# write config files, dhcpd.conf etc.
#
sub writeconfig {
    my %savesettings = ();

    # copy the relevant settings into a duplicate hash, otherwise we'd need to undef loads of stuff
    foreach $interface (@INTERFACEs) {
        for ($counter = 1; $counter <= 1; $counter++) {
            $savesettings{"ENABLED_${interface}_${counter}"} = $dhcpsettings{"ENABLED_${interface}_${counter}"};
            $savesettings{"ENABLED_BOOTP_${interface}_${counter}"} =
                $dhcpsettings{"ENABLED_BOOTP_${interface}_${counter}"};
            $savesettings{"START_ADDR_${interface}_${counter}"}  = $dhcpsettings{"START_ADDR_${interface}_${counter}"};
            $savesettings{"END_ADDR_${interface}_${counter}"}    = $dhcpsettings{"END_ADDR_${interface}_${counter}"};
            $savesettings{"DOMAIN_NAME_${interface}_${counter}"} = $dhcpsettings{"DOMAIN_NAME_${interface}_${counter}"};
            $savesettings{"DEFAULT_LEASE_TIME_${interface}_${counter}"} =
                $dhcpsettings{"DEFAULT_LEASE_TIME_${interface}_${counter}"};
            $savesettings{"DNS1_${interface}_${counter}"}  = $dhcpsettings{"DNS1_${interface}_${counter}"};
            $savesettings{"DNS2_${interface}_${counter}"}  = $dhcpsettings{"DNS2_${interface}_${counter}"};
            $savesettings{"WINS1_${interface}_${counter}"} = $dhcpsettings{"WINS1_${interface}_${counter}"};
            $savesettings{"WINS2_${interface}_${counter}"} = $dhcpsettings{"WINS2_${interface}_${counter}"};
            $savesettings{"NTP1_${interface}_${counter}"}  = $dhcpsettings{"NTP1_${interface}_${counter}"};
            $savesettings{"NTP2_${interface}_${counter}"}  = $dhcpsettings{"NTP2_${interface}_${counter}"};
        }
    }
    $savesettings{'SORT_FIXEDLEASELIST'} = $dhcpsettings{'SORT_FIXEDLEASELIST'};
    &General::writehash('/var/ipcop/dhcp/settings', \%savesettings);

    # only wanted to change (and save) sort order, no need to touch the dnsmasq file
    return if ($_[0] == 0);

    open(FILE, ">/var/ipcop/dhcp/dnsmasq.conf") or die "Unable to write dhcp server conf file";
    flock(FILE, 2);

    # Global settings
    print FILE <<END
# Do not modify '/var/ipcop/dhcp/dnsmasq.conf' directly since any changes
# you make will be overwritten whenever you resave dhcp settings using the
# web interface! 
# Instead modify the file '/var/ipcop/dhcp/dnsmasq.local' and then restart 
# the DHCP server using the web interface. Changes made to the 'local' file
# will then propagate to the DHCP server.

# some global settings
pid-file=/var/run/dnsmasq/dnsmasq.pid
dhcp-authoritative
dhcp-leasefile=/var/run/dnsmasq/dnsmasq.leases
conf-file=/var/ipcop/dhcp/dnsmasq.static
conf-file=/var/ipcop/dhcp/dnsmasq.local

END
        ;

    # Interface definitions
    foreach $interface (@INTERFACEs) {
        for ($counter = 1; $counter <= $netsettings{"${interface}_COUNT"}; $counter++) {
            if ($savesettings{"ENABLED_${interface}_${counter}"} eq 'on') {
                my $lease = $savesettings{"DEFAULT_LEASE_TIME_${interface}_${counter}"} * 60;

                print FILE "# network: ${interface} - ${counter}\n";
                if ($savesettings{"START_ADDR_${interface}_${counter}"}) {
                    print FILE "dhcp-range=${interface}_${counter},"
                        . $savesettings{"START_ADDR_${interface}_${counter}"} . ","
                        . $savesettings{"END_ADDR_${interface}_${counter}"}
                        . ",$lease\n";
                }
                else {
                    print FILE "dhcp-range=${interface}_${counter},"
                        . $netsettings{"${interface}_${counter}_ADDRESS"}
                        . ",static,$lease\n";
                }
                if ($savesettings{"DOMAIN_NAME_${interface}_${counter}"}) {
                    print FILE "dhcp-option=${interface}_${counter},option:domain-name,"
                        . $savesettings{"DOMAIN_NAME_${interface}_${counter}"} . "\n";
                }

                # bootp enabled
                if ($savesettings{"ENABLED_BOOTP_${interface}_${counter}"}) {
                    print FILE "bootp-dynamic=${interface}_${counter}\n";
                }

                # DNS server(s)
                print FILE "dhcp-option=${interface}_${counter},option:dns-server,"
                    . $savesettings{"DNS1_${interface}_${counter}"};
                print FILE "," . $savesettings{"DNS2_${interface}_${counter}"} . "\n"
                    if ($savesettings{"DNS2_${interface}_${counter}"});
                print FILE "\n";

                # WINS server(s)
                if ($savesettings{"WINS1_${interface}_${counter}"}) {
                    print FILE "dhcp-option=${interface}_${counter},option:netbios-ns,"
                        . $savesettings{"WINS1_${interface}_${counter}"};
                    print FILE "," . $savesettings{"WINS2_${interface}_${counter}"} . "\n"
                        if ($savesettings{"WINS2_${interface}_${counter}"});
                    print FILE "\n";
                }

                # NTP server(s)
                if ($savesettings{"NTP1_${interface}_${counter}"}) {
                    print FILE "dhcp-option=${interface}_${counter},option:ntp-server,"
                        . $savesettings{"NTP1_${interface}_${counter}"};
                    print FILE "," . $savesettings{"NTP2_${interface}_${counter}"} . "\n"
                        if ($savesettings{"NTP2_${interface}_${counter}"});
                    print FILE "\n";
                }
            }
        }    # interface counter
    }    # for some interfaces
    close FILE;

    system '/usr/local/bin/restartdhcp';
}

sub writefixedleases {
    open(FILE, ">/var/ipcop/dhcp/fixedleases") or die 'Unable to open fixed leases file.';
    print FILE @fixedleases;
    close(FILE);

    # sort
    if ($dhcpsettings{'SORT_FIXEDLEASELIST'} eq 'FIXEDMAC') {
        system "/usr/bin/sort -t ',' -k 1,1 /var/ipcop/dhcp/fixedleases -o /var/ipcop/dhcp/fixedleases";
    }
    elsif ($dhcpsettings{'SORT_FIXEDLEASELIST'} eq 'FIXEDMACRev') {
        system "/usr/bin/sort -r -t ',' -k 1,1 /var/ipcop/dhcp/fixedleases -o /var/ipcop/dhcp/fixedleases";
    }
    elsif ($dhcpsettings{'SORT_FIXEDLEASELIST'} eq 'FIXEDIPRev') {
        system "/usr/bin/sort -r -n -t ',' -k 2,2 /var/ipcop/dhcp/fixedleases -o /var/ipcop/dhcp/fixedleases";
    }
    else {
        # FIXEDIP is also default when no sorting selected (yet)
        system "/usr/bin/sort -n -t ',' -k 2,2 /var/ipcop/dhcp/fixedleases -o /var/ipcop/dhcp/fixedleases";
    }

    open(FILE, '/var/ipcop/dhcp/fixedleases');
    @fixedleases = <FILE>;
    close(FILE);

    # only wanted to change (and save) sort order, no need to touch the dnsmasq file
    return if ($_[0] == 0);

    # now write the fixed leases file for dnsmasq
    open(FILE, ">/var/ipcop/dhcp/dnsmasq.static") or die "Unable to write dhcp server conf file";
    $key = 0;
    foreach $line (@fixedleases) {
        chomp($line);
        my @temp = split(/\,/, $line);
        if ($temp[2] eq "on") {
            my @fqdn = split(/\./, $temp[7], 2);
            print FILE "dhcp-host=$temp[0],net:STATIC_$key,$temp[1],$fqdn[0]\n";

            if ($temp[4]) {
                print FILE "dhcp-boot=net:STATIC_$key,$temp[4]";
                print FILE ",,$temp[3]" if ($temp[3]);
                print FILE "\n";
            }
            if ($temp[5]) {
                print FILE "dhcp-option-force=net:STATIC_$key,option:root-path,$temp[5]\n";
            }
        }
        $key++;
    }
    close FILE;

    system '/usr/local/bin/restartdhcp';
}
