#!/usr/bin/perl
#
# This file is part of the IPCop Firewall.
#
# IPCop is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# IPCop is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with IPCop; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
#
# $Id: aggregatetraffic.pl 4078 2010-01-06 11:24:40Z owes $
#

use DBI;
use strict;

#use warnings;

require '/usr/lib/ipcop/traffic-lib.pl';

# Debug level:
#	0 - aggregate traffic, no print
#	1 - aggregate traffic, print
#	2 - only print traffic data
my $debugLevel = 0;

# Detailed aggregation on/off
my $detailed = 'off';
if($TRAFFIC::settings{'DETAIL_LEVEL'} eq 'High')
{
    $detailed = 'on';
}

# FIXME/TODO: as ulogd locks the database we have to stop it here and restart after aggregation
# As long as ulogd is not running the traffic flow is not logged, we lose some data!
system("/usr/bin/killall ulogd");

my $dbh = DBI->connect("dbi:SQLite:dbname=/var/log/ulogd/account.db", "", "", {RaiseError => 1, AutoCommit => 0});
my $groupDetailed = "";
my $selectColumns = "";

# Set temp directory to /tmp instead of /var/tmp
eval { $dbh->do("PRAGMA temp_store_directory=\"/tmp\";") };

if ($detailed eq 'on') {
    ### Aggregate detailed
    $selectColumns = " ip_saddr, ip_protocol, tcp_dport, udp_dport, ";
    $groupDetailed = " , ip_saddr, ip_protocol, tcp_dport, udp_dport ";
}
else {
    ### Aggregate simple
    $selectColumns = " NULL AS ip_saddr, NULL AS ip_protocol, NULL AS tcp_dport, NULL AS udp_dport, ";
    $groupDetailed = "";
}

#my $statementSelect = "select oob_prefix, count(*), sum(ip_totlen) from ulog group by oob_prefix;";
my $statementSelect = "SELECT strftime(\"%Y%m%d\", oob_time_sec, 'unixepoch') AS date, oob_prefix, ";
$statementSelect .= $selectColumns;
$statementSelect .= " SUM(ip_totlen) ";
$statementSelect .= " FROM ulog ";
$statementSelect .= " GROUP BY date, oob_prefix ";
$statementSelect .= $groupDetailed;
$statementSelect .= ";";

my $whereClause = q{
    WHERE (date = ? OR (date IS NULL AND ? IS NULL) )
    AND (oob_prefix = ? OR (oob_prefix IS NULL AND ? IS NULL) )
    AND (ip_saddr = ? OR (ip_saddr IS NULL AND ? IS NULL) )
    AND (ip_protocol = ? OR (ip_protocol IS NULL AND ? IS NULL) )
    AND (tcp_dport = ? OR (tcp_dport IS NULL AND ? IS NULL) )
    AND (udp_dport = ? OR (udp_dport IS NULL AND ? IS NULL) )
};

eval {
    my $sthSelect = $dbh->prepare($statementSelect);
    my $sthInsert = $dbh->prepare(
        q{
          INSERT INTO daily (date, oob_prefix, ip_saddr, ip_protocol, tcp_dport, udp_dport, ip_totlen) VALUES (?, ?, ?, ?, ?, ?, ?);
      }
    );
    my $sthCheckDailyCount = $dbh->prepare("SELECT count(*) FROM daily $whereClause;");
    my $sthCheckDaily      = $dbh->prepare("SELECT ip_totlen FROM daily $whereClause;");
    my $sthUpdate          = $dbh->prepare("UPDATE daily set ip_totlen = ? $whereClause;");
    my $sthDelete          = $dbh->prepare("DELETE FROM ulog;");

    $sthSelect->execute();

    # Bind Perl variables to columns:
    my ($date, $prefix, $srcAdr, $proto, $tcpPort, $udpPort, $bytes);

    $sthSelect->bind_columns(\$date, \$prefix, \$srcAdr, \$proto, \$tcpPort, \$udpPort, \$bytes);

    my $countDaily = 0;
    $sthCheckDailyCount->bind_columns(\$countDaily);
    my $bytesCheck;
    $sthCheckDaily->bind_columns(\$bytesCheck);

    while ($sthSelect->fetch()) {
        my @whereData =
            ($date, $date, $prefix, $prefix, $srcAdr, $srcAdr, $proto, $proto, $tcpPort, $tcpPort, $udpPort, $udpPort);

        $sthCheckDailyCount->execute(@whereData);
        $sthCheckDailyCount->fetch();
        if ($countDaily > 0) {
            $sthCheckDaily->execute(@whereData);
            $sthCheckDaily->fetch();

            # There is already an traffic entry with this data
            $bytes = $bytesCheck + $bytes;
            if ($debugLevel < 2) {
                $sthUpdate->execute($bytes, @whereData);
            }
            print "Update: " if ($debugLevel > 0);
        }
        else {

            # New Entry
            if ($debugLevel < 2) {
                $sthInsert->execute($date, $prefix, $srcAdr, $proto, $tcpPort, $udpPort, $bytes);
            }
            print "Insert: " if ($debugLevel > 0);
        }

        print " $date|$prefix|$srcAdr|$proto|$tcpPort|$udpPort|$bytes \n" if ($debugLevel > 0);
    }

    if ($debugLevel < 2) {
        $sthDelete->execute();
    }

    $sthSelect->finish();
    $sthInsert->finish();
    $sthCheckDailyCount->finish();
    $sthCheckDaily->finish();
    $sthUpdate->finish();

    $dbh->commit();    # commit the changes if we get this far
};
if ($@) {
    warn "Transaction aborted because $@";

    # now rollback to undo the incomplete changes
    # but do it in an eval{} as it may also fail
    eval { $dbh->rollback() };

    # add other application on-error-clean-up code here
}

$dbh->disconnect() or warn $dbh->errstr;

# reconnect for "VACUUM"
$dbh = DBI->connect("dbi:SQLite:dbname=/var/log/ulogd/account.db", "", "");

# shrink database file
eval { $dbh->do("VACUUM;") };
$dbh->disconnect() or warn $dbh->errstr;

# FIXME/TODO: Restart ulogd
system("/usr/sbin/ulogd -d");

