/* 
 * restore.c: restore from some backup during installation
 *
 * This file is part of the IPCop Firewall.
 *
 * IPCop is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * IPCop is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with IPCop; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 * (c) 2008, the IPCop team
 * 
 * $Id: restore.c 4157 2010-01-21 09:58:34Z owes $
 * 
 */


#include <newt.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include "common.h"
#include "common_backup.h"
#include "common_newt.h"
#include "arch_defs.h"


// tweak for errorbox
#define  gettext  ipcop_gettext

#define TMP_RESTORE_PATH_FULL      "/harddisk/tmp/restore"
#define TMP_RESTORE_PATH_CHROOT    "/tmp/restore"
#define MOUNT_BACKUP_FULL           "/harddisk/mnt/usb"
#define MOUNT_BACKUP_CHROOT         "/mnt/usb"
#define DATFILE                     "/home/httpd/html/backup/ipcop-xxxx-xx-xx_xx-xx-xx.dat"


static char command[STRING_SIZE];
/* these are module global, to make callback function work */
static newtComponent restoreform;
static newtComponent radiofloppy, radiousb, radionetwork;
static newtComponent entryhostname, entrypassword;


/* */
static int copy_change_files(void)
{
    if (access(TMP_RESTORE_PATH_FULL "/var/ipcop/main/settings", 0)) {
        errorbox(ipcop_gettext("TR_NO_MAIN_SETTINGS_IN_BACKUP"));
        return FAILURE;
    }

    if (testbackupversion(TMP_RESTORE_PATH_FULL) != SUCCESS) {
        errorbox(ipcop_gettext("TR_NO_VALID_VERSION_IN_BACKUP"));
        return FAILURE;
    }

    mysystem("chroot /harddisk /bin/cp -af " TMP_RESTORE_PATH_CHROOT "/. /");

    /* TODO: here we will need to add mechanism for upgrade from 1.4.xx configuration files */
    mysystem("chroot /harddisk /usr/local/bin/upgrade.sh");

    return SUCCESS;
}


/* Return SUCCESS when dev contains a backup key and backup dat file.
 * Leave dev mounted */
static int test_backup_key(char *dev, char *hostname)
{
    mysystem("/bin/umount " MOUNT_BACKUP_FULL " 2>/dev/null");
    snprintf(command, STRING_SIZE, "/bin/mount -t vfat -o ro %s " MOUNT_BACKUP_FULL, dev);
    if (mysystem(command)) {
        return FAILURE;                 /* no mountable dev */
    }

    /* Test backup .key */
    snprintf(command, STRING_SIZE, MOUNT_BACKUP_FULL "/backup.%s.key", hostname);
    if (access(command, 0)) {
        return FAILURE;                 /* no backup key on this dev */
    }
    /* Test backup .dat */
    snprintf(command, STRING_SIZE, MOUNT_BACKUP_FULL "/%s.dat", hostname);
    if (access(command, 0)) {
        return FAILURE;                 /* no backup dat on this dev */
    }
    /* */
    snprintf(command, STRING_SIZE, "cp -f " MOUNT_BACKUP_FULL "/%s.dat /harddisk/" DATFILE, hostname);
    mysystem(command);

    return SUCCESS;
}


/* Try to mount usb device until backup.<hostname>.key is found */
static int mountusb(char *hostname)
{
    char sourcedev[30];
    int i, j;

    /* TODO: instead of scanning sda, sda1 ... sdz3, sdz4 it is probably better to look at /proc/partitions */
    for (i = 'a'; i <= 'z'; i++) {
        for (j = 0; j < 5; j++) {
            if (j) {
                sprintf(sourcedev, "/dev/sd%c%d", i, j);
            }
            else {
                sprintf(sourcedev, "/dev/sd%c", i);
            }
            if (test_backup_key(sourcedev, hostname) == SUCCESS) return SUCCESS;
        }
    }
    return FAILURE;
}


/* Try and grab from /dev/fd0 (1st floppy)
   USB floppy is /dev/sd[a-z], need some magic to walk through sd devices */
static int restorefromfloppy(void)
{
    char device[STRING_SIZE];
    struct stat st;
    int i;

    /* since we do not have floppy.ko, grab from already installed IPCop */
#if defined(__powerpc__) || defined(__powerpc64__)
    mysystem("chroot /harddisk /sbin/modprobe swim3");
#else
    mysystem("chroot /harddisk /sbin/modprobe floppy");
#endif
    /* give the device some time to settle */
    sleep(1);


    /* /dev/fd0 first */
    strcpy(device, "/dev/fd0");
    lstat(device, &st);
    if (S_ISBLK(st.st_mode)) {
        if (mysystem
            ("chroot /harddisk /bin/tar -X /var/ipcop/backup/exclude.system -C " TMP_RESTORE_PATH_CHROOT
             " -xvzf /dev/fd0") == 0) {
            newtPopWindow();    // Pop status window

            return copy_change_files();
        }
    }

    /* Now USB floppy (needs work) */
    for (i = 'a'; i <= 'z'; i++) {
    }

    newtPopWindow();            // Pop status window
    errorbox(ipcop_gettext("TR_UNABLE_TO_INSTALL_FILES"));
    return FAILURE;
}


/* */
static int restorefromusb(char *hostname, char *password)
{
    int rc;

    if (mountusb(hostname) == FAILURE) {
        newtPopWindow();
        errorbox(ipcop_gettext("TR_NO_BACKUP_ON_USB_FOUND"));
        return FAILURE;
    }

    /*  device is mounted and contains .key and .dat 
        extract .key first */
    snprintf(command,  STRING_SIZE, "chroot /harddisk /usr/bin/openssl enc"
                        " -a -d -aes256 -salt"
                        " -pass pass:%s"
                        " -in " MOUNT_BACKUP_CHROOT "/backup.%s.key"
                        " -out /var/ipcop/backup/backup.key",
                        password, hostname);
    if (mysystem(command)) {
        newtPopWindow();
        errorbox(ipcop_gettext("TR_WRONG_PASSWORD_OR_KEYFILE"));
        return FAILURE;
    }

    /* adjust mode */
    mysystem("chroot /harddisk /bin/chmod 400 /var/ipcop/backup/backup.key");

    snprintf(command, STRING_SIZE, "chroot /harddisk /usr/local/bin/ipcoprestore"
        " --restore=%s --hostname=ipcop --hardware", DATFILE);
    if ((rc = mysystem(command)) != 0) {
        newtPopWindow();
        fprintf(flog, "ipcoprestore returned errorcode: %d (%d)\n", (rc >> 8), rc);
        if (rc == (BACKUP_ERR_VERSION << 8)) {
            /* Special case, inform with some more detail */
            errorbox(ipcop_gettext("TR_NO_VALID_VERSION_IN_BACKUP"));
        }
        else {
            errorbox(ipcop_gettext("TR_UNABLE_TO_INSTALL_FILES"));
        }
        return FAILURE;
    }

    newtPopWindow();            // Pop status window
    return SUCCESS;
}


/* */
static int restorefromnetwork(char *hostname)
{
    newtPopWindow();            // Pop status window
    newtWinMessage(ipcop_gettext("TR_RESTORE"), ipcop_gettext("TR_OK"), "Sorry, not implemented yet.");
    return FAILURE;
}


/* Change disbabled hostname & password depending on radio selection */
static void restorecallback(newtComponent cm, void *data)
{
    newtComponent selected = newtRadioGetCurrent(radiofloppy);

    if ((selected == radiousb) || (selected == radionetwork)) {
        newtEntrySetFlags(entryhostname, NEWT_FLAG_DISABLED, NEWT_FLAGS_RESET);
        newtEntrySetFlags(entrypassword, NEWT_FLAG_DISABLED, NEWT_FLAGS_RESET);
    }
    else {
        newtEntrySetFlags(entryhostname, NEWT_FLAG_DISABLED, NEWT_FLAGS_SET);
        newtEntrySetFlags(entrypassword, NEWT_FLAG_DISABLED, NEWT_FLAGS_SET);
    }
    newtRefresh();
    newtDrawForm(restoreform);
}


/* Selection screen for source of backup */
int handlerestore(void)
{
    newtComponent text;
    newtComponent ok, skip;
    newtComponent labelhostname, labelpassword;
    char hostnameinitvalue[STRING_SIZE];
    char passwordinitvalue[STRING_SIZE];
    char typevalue[32];
    const char *hostnamevalue;
    const char *passwordvalue;
    struct newtExitStruct exitstruct;
    char message[STRING_SIZE_LARGE];
    int numLines;
    int error;
    int userskip;

    strcpy(hostnameinitvalue, "ipcop.localdomain");
    strcpy(typevalue, "floppy");

    do {
        snprintf(message, STRING_SIZE, ipcop_gettext("TR_RESTORE_CONFIGURATION"), NAME);
        text = newtTextboxReflowed(1, 1, message, 68, 0, 0, 0);
        numLines = newtTextboxGetNumLines(text);

        newtCenteredWindow(72, 14 + numLines, ipcop_gettext("TR_RESTORE"));
        restoreform = newtForm(NULL, NULL, 0);
        newtFormAddComponent(restoreform, text);

        /* selections: floppy, usb */
        radiofloppy = newtRadiobutton(12, 2 + numLines, ipcop_gettext("TR_FLOPPY"), !strcmp(typevalue, "floppy"), NULL);
        radiousb = newtRadiobutton(12, 3 + numLines, ipcop_gettext("TR_USB_KEY"), !strcmp(typevalue, "usb"), radiofloppy);

        newtComponentAddCallback(radiofloppy, restorecallback, NULL);
        newtComponentAddCallback(radiousb, restorecallback, NULL);

        if (medium_sources == network) {
            radionetwork = newtRadiobutton(12, 4 + numLines, "http/ftp", !strcmp(typevalue, "http"), radiousb);
            newtComponentAddCallback(radionetwork, restorecallback, NULL);
            newtFormAddComponents(restoreform, radiofloppy, radiousb, radionetwork, NULL);
        }
        else {
            /* when not installing from network source, there is no active and usable network card */
            radionetwork = NULL;
            newtFormAddComponents(restoreform, radiofloppy, radiousb, NULL);
        }

        /* hostname for network restore */
        labelhostname = newtTextbox(2, 6 + numLines, 35, 1, 0);
        newtTextboxSetText(labelhostname, ipcop_gettext("TR_HOSTNAME"));
        newtFormAddComponent(restoreform, labelhostname);
        entryhostname = newtEntry(25, 6 + numLines, hostnameinitvalue, 35, &hostnamevalue, 0);
        newtFormAddComponent(restoreform, entryhostname);
        /* password */
        labelpassword = newtTextbox(2, 7 + numLines, 35, 1, 0);
        newtTextboxSetText(labelpassword, ipcop_gettext("TR_BACKUP_PASSWORD"));
        newtFormAddComponent(restoreform, labelpassword);
        entrypassword = newtEntry(25, 7 + numLines, "", 20, &passwordvalue, 0);
        newtEntrySetFlags(entrypassword, NEWT_FLAG_PASSWORD, NEWT_FLAGS_SET);
        newtFormAddComponent(restoreform, entrypassword);

        if (!strcmp(typevalue, "floppy")) {
            /* disabled for default selection */
            newtEntrySetFlags(entryhostname, NEWT_FLAG_DISABLED, NEWT_FLAGS_SET);
            newtEntrySetFlags(entrypassword, NEWT_FLAG_DISABLED, NEWT_FLAGS_SET);
        }

        ok = newtButton(6, 9 + numLines, ipcop_gettext("TR_OK"));
        skip = newtButton(26, 9 + numLines, gettext("TR_SKIP"));
        newtFormAddComponents(restoreform, ok, skip, NULL);

        newtRefresh();
        newtDrawForm(restoreform);

        error = FAILURE;
        userskip = 0;
        newtFormRun(restoreform, &exitstruct);
        newtPopWindow();
        strcpy(hostnameinitvalue, (char *)hostnamevalue);
        strcpy(passwordinitvalue, (char *)passwordvalue);
        newtFormDestroy(restoreform);

        if (exitstruct.u.co == skip) {
            userskip = 1;
        }

        if (exitstruct.u.co == ok) {
            newtComponent selected = newtRadioGetCurrent(radiofloppy);

            statuswindow(72, 5, ipcop_gettext("TR_RESTORE"), ipcop_gettext("TR_READING_BACKUP"));
            /* cleanout possible leftovers and (re)create temp path */
            mysystem("/bin/rm -rf " TMP_RESTORE_PATH_FULL);
            mkdir(TMP_RESTORE_PATH_FULL, S_IRWXU | S_IRWXG | S_IRWXO);
            mysystem("/bin/rm -rf " MOUNT_BACKUP_FULL);
            mkdir(MOUNT_BACKUP_FULL, S_IRWXU|S_IRWXG|S_IRWXO);

            if (selected == radiofloppy) {
                strcpy(typevalue, "floppy");
                error = restorefromfloppy();
            }
            else if (selected == radiousb) {
                strcpy(typevalue, "usb");
                if (!strcmp(passwordinitvalue, "")) {
                    /* password is mandatory to decrypt the key */
                    newtPopWindow();
                    errorbox(ipcop_gettext("TR_PASSWORD_CANNOT_BE_BLANK"));
                    error = FAILURE;
                }
                else {
                    error = restorefromusb(hostnameinitvalue, passwordinitvalue);
                }
            }
            else {
                strcpy(typevalue, "http");
                error = restorefromnetwork(hostnameinitvalue);
            }
        }
    }
    while ((error != SUCCESS) && (userskip == 0));

    return (error);

    // ipcop_gettext("TR_FILE_NOT_FOUND")
    // ipcop_gettext("TR_UNABLE_TO_INSTALL_FILES")
}
