#!/usr/bin/env python
# -*- coding: utf-8 -*-
import os
import sys
import subprocess
import tempfile
import shutil
import json
import string
import random
from contextlib import contextmanager


PKG_CACHE_DIR = '/var/cache/pkglist.tmp'
SPRITE_DIR = os.path.join(PKG_CACHE_DIR, 'sprite')
ICON_DIR = os.path.join(PKG_CACHE_DIR, 'icon')
SPRITING_FLAG = os.path.join(SPRITE_DIR, 'spriting')
INDEX_FILE = os.path.join(SPRITE_DIR, 'icon_index.json')
PRESERVE_SPRITE_COUNT = 2
IDENTIFY = '/bin/identify'
CONVERT = '/bin/convert'


class Spriter(object):
    def __init__(self, img_size):
        self.img_size = img_size
        self.icon_suffix = '{}.png'.format(img_size)
        self.tmp_dir = tempfile.mkdtemp(dir='/tmp', prefix='sprite_')
        self.len_key = 'index{}'.format(img_size)
        self.image_key = 'sprite{}'.format(img_size)
        self.sprite_prefix = 'sprite_{}_'.format(img_size)
        self.image = '{}{}.png'.format(self.sprite_prefix, gen_random_string(6))
        self.image = os.path.join(SPRITE_DIR, self.image)
        self.new_icons = []
        self.len = 0
        self.index = {}
        self.index_conf = {}
        self.exists_old_sprite = self.init_from_existing_sprite()

    def __enter__(self):
        return self

    def __exit__(self, type, value, traceback):
        shutil.rmtree(self.tmp_dir)

    def init_from_existing_sprite(self):
        try:
            with open(INDEX_FILE) as file:
                self.index_conf = json.load(file)
                self.index = self.index_conf['index']
                self.old_image = os.path.join(SPRITE_DIR, self.index_conf[self.image_key])
                if not os.path.isfile(self.old_image):
                    return False
                self.len = self.index_conf[self.len_key]
        except (IOError, KeyError, AttributeError):
            return False

        return True

    def sprite(self):
        self.iterate_icons()
        if self.new_icons:
            self.gen_sprite_image()
            self.save_conf()
            self.remove_old_sprites()

    def remove_old_sprites(self):
        files = [os.path.join(SPRITE_DIR, _) for _ in os.listdir(SPRITE_DIR) if _.startswith(self.sprite_prefix)]
        files.sort(key=os.path.getmtime, reverse=True)
        for file in files[2:]:
            os.remove(file)

    def gen_sprite_image(self):
        command = [CONVERT]
        if self.exists_old_sprite:
            command.append(self.old_image)
        command.extend(self.new_icons)
        command.extend(['-append', self.image])
        subprocess.call(command)

    def save_conf(self):
        self.index_conf[self.len_key] = self.len
        self.index_conf[self.image_key] = os.path.basename(self.image)
        self.index_conf['index'] = self.index

        with open(INDEX_FILE, 'w') as file:
            json.dump(self.index_conf, file, separators=[',', ':'])

    def iterate_icons(self):
        for dirpath, dirnames, filenames in os.walk(ICON_DIR):
            for filename in filenames:
                if filename.endswith(self.icon_suffix):
                    img_path = os.path.join(dirpath, filename)
                    splitted = img_path.split('/')
                    pkgname = splitted[-3]
                    version = splitted[-2]
                    index_key = '{}{}-{}'.format(pkgname, version, self.img_size)

                    if index_key in self.index or not is_image_file(img_path):
                        continue

                    new_icon = self.convert_icon(img_path, index_key)
                    self.new_icons.append(new_icon)
                    self.index[index_key] = self.len
                    self.len += 1

    def convert_icon(self, path, index_key):
        icon_copy = os.path.join(self.tmp_dir, '{}.png'.format(self.len))
        shrink_size = self.img_size - 2
        resize_command = [CONVERT, path, '-resize', 'x'.join([str(shrink_size)] * 2), icon_copy]
        background_command = [CONVERT, icon_copy, '-background', 'none', '-gravity', 'center', '-extent', 'x'.join([str(self.img_size)] * 2), icon_copy]

        with open(os.devnull, 'w') as FNULL:
            subprocess.call(resize_command, stdout=FNULL, stderr=subprocess.STDOUT)
            subprocess.call(background_command, stdout=FNULL, stderr=subprocess.STDOUT)

        return icon_copy


def is_image_file(path):
    with open(os.devnull, 'w') as FNULL:
        return 0 == subprocess.call([IDENTIFY, '-ping', path], stdout=FNULL, stderr=subprocess.STDOUT)


def gen_random_string(length):
    return ''.join(random.choice(string.ascii_letters + string.digits) for _ in range(length))


def is_spriting():
    return os.path.isfile(SPRITING_FLAG)


def mkdir_p(path):
    try:
        os.makedirs(path)
    except OSError:
        pass


def touch(path):
    with open(path, 'a'):
        os.utime(path, None)


@contextmanager
def spriting_flag():
    touch(SPRITING_FLAG)
    yield
    os.remove(SPRITING_FLAG)


def main():
    if is_spriting():
        exit(0)

    try:
        append_only = 'append' == sys.argv[1]
    except IndexError:
        append_only = False

    mkdir_p(SPRITE_DIR)
    with spriting_flag():
        for size in [72, 256]:
            with Spriter(img_size=size) as spriter:
                if append_only and not spriter.exists_old_sprite:
                    continue
                spriter.sprite()

if __name__ == "__main__":
    main()
