#!/bin/sh
# Copyright (c) 2000-2015 Synology Inc. All rights reserved.

SYNOPKG=/usr/syno/bin/synopkg
SYNOGETKEYVALUE=/usr/syno/bin/synogetkeyvalue

syslog() {
	local ret=$?
	logger -p user.err -t $(basename $0) "$@"
	return $ret
}

resolve_pkgname_from_spk()
{
	local spk="$1"
	local pkgname

	pkgname="$(tar xOf "$spk" INFO | grep "^package=" | cut -d'"' -f2)"
	if [ -n "$pkgname" ]; then
		echo "$pkgname"
	else
		echo "$spk" | sed -n 's,^.*/\([^-]\+\)-.*$,\1,p'
	fi
}

get_display_name()
{
	local spk="$1"
	local pkgname="$2"
	local tmpdir
	local displayname_key=displayname

	tmpdir="$(mktemp -d "${pkgname}XXX" -p /tmp)"
	/bin/tar xf "$spk" -C "$tmpdir" INFO

	lang="$($SYNOGETKEYVALUE /etc/synoinfo.conf maillang)"
	if [ -n "$lang" ]; then
		displayname_key="displayname_$lang"
	fi

	display_name=$($SYNOGETKEYVALUE "$tmpdir/INFO" "$displayname_key")
	[ -n "$display_name" ] && echo "$display_name" || echo "$pkgname"

	rm -r "$tmpdir"
}

send_notification()
{
	local app="SYNO.SDS.PkgManApp.Instance"
	local send_to="@administrators"
	local title="tree:leaf_packagemanage"

	/usr/syno/bin/synodsmnotify -c "$app" -t "true" "$send_to" "$title" "$1" "$2" "" ""
}

is_package_installed()
{
	local packageName=$(resolve_pkgname_from_spk $1)

	if [ -d "/var/packages/$packageName" ]; then
		true
	else
		false
	fi
}

# Is package $1 built-in package for freshly installed DSM?
#
# PROTO TYPE:
#         is_in_fresh_install spk_filename
is_in_fresh_install()
{
	local packageName=$(resolve_pkgname_from_spk $1)
	local package_list="FileStation SynoFinder USBCopy"
	local is_nvr=`/bin/get_key_value /etc.defaults/synoinfo.conf nvr`

	# For NVR series, SurveillanceStation will become buildin package to facilitate
	# users to use NVR. (Save download time and world wide network connection)
	if [ "xyes" == "x${is_nvr}" ]; then
		package_list="$package_list SurveillanceStation"
	fi

	for i in ${package_list}
	do
		if [ "x$packageName" = "x${i}" ]; then
			return 0
		fi
	done
	return 1
}

# Is packge $2 built-in package for dsm $1?
#
# PROTO TYPE:
#         is_in_dsm previous_dsm_build_number spk_filename
is_in_dsm()
{
	local packageName=$(resolve_pkgname_from_spk $2)
	local package_list="FileStation SynoFinder DisasterRecovery"
	local is_nvr=`/bin/get_key_value /etc.defaults/synoinfo.conf nvr`

	# following if-statements should be something like
	#	if [ $1 -lt xxxx ]; then
	#		package="$package_list yyyy"
	#	fi
	# where
	# 	xxxx: max reserved build number in "previous" release
	#	yyyy: packages become builtin or must install/upgrade in "this" release
	if [ $1 -lt 14600 ]; then  # builtin package introduced in 6.1 beta
		package_list="$package_list USBCopy"
	fi
	if [ $1 -lt 15030 ]; then # check before 6.1 RC
		package_list="$package_list SnapshotReplication"
	fi

	# For NVR series, SurveillanceStation will become buildin package to facilitate
	# users to use NVR. (Save download time and world wide network connection)
	if [ "xyes" == "x${is_nvr}" ]; then
		package_list="$package_list SurveillanceStation"
	fi

	for i in ${package_list}
	do
		if [ "x$packageName" = "x${i}" ]; then
			return 0
		fi
	done
	return 1
}

install_package() {
	local packageName=$(resolve_pkgname_from_spk $1)
	local ret=
	local is_upgrade=false
	local msg=

	if is_package_installed $1; then
		is_upgrade=true
	fi

	if [ "$packageName" = "FileStation" ] || [ "$packageName" = "SynoFinder" ]; then
		:
		# always install and upgrade FileStation.
		# always install and upgrade SynoFinder.
	elif [ "$packageName" = "DisasterRecovery" ]; then
		if ! $is_upgrade; then
			return  # don't install packages
		fi
	else
		if $is_upgrade; then
			return  # don't upgrade packages
		fi
	fi

	# perform package-specific checks here
	case "$packageName" in
		"USBCopy")
			# if no usbcopy and no sdcopy => do not install
			if [ "yes" != "$(get_key_value /etc.defaults/synoinfo.conf usbcopy)" ] &&
				[ "yes" != "$(get_key_value /etc.defaults/synoinfo.conf sdcopy)" ]; then
				return
			fi
			;;
		"SnapshotReplication")
			python /usr/syno/synodr/sbin/check_snapreplica.py
			chk_ret=$?
			if [ 0 -ne $chk_ret ]; then
				return
			fi
			;;
		"*")
			;;
	esac

	if $is_upgrade; then
		syslog "upgrade package: $1"
	else
		syslog "install package: $1"
	fi
	$SYNOPKG install "$1"
	ret=$?

	# return if install failed
	[ $ret -ne 0 ] && return $ret

	# send notification
	if $is_upgrade; then
		msg="pkgmgr:upgrade_start_pkg_completed"
	else
		msg="pkgmgr:install_start_pkg_completed"
	fi
	send_notification $msg "$(get_display_name "$1" "$packageName")"

	/usr/syno/sbin/synopkgctl enable $packageName
	/usr/syno/sbin/synopkgctl correct-cfg $packageName
	return $ret
}

install_builtin_packages() {
	local package_dir=$1
	local old_bnum=`/bin/get_key_value /.old_patch_info/VERSION buildnumber`
	local fresh_install=true

	[ -d "$package_dir" ] || return 1

	# if no "/.old_patch_info" (new installation), only install packages that pass is_in_fresh_install()
	[ -f "/.old_patch_info/VERSION" ] && fresh_install=false

	if [ ! -z "$(ls -A $package_dir)" ]; then
		for i in $package_dir/*; do
			if $fresh_install && $(is_in_fresh_install "$i"); then
				install_package "$i"
			elif ! $fresh_install && $(is_in_dsm "${old_bnum}" "$i"); then
				install_package "$i"
			fi
		done
	fi
}

usage() {
	cat << EOF
Usage: $(basename $0) [start]
EOF
}

install_manutil_packages()
{
	local package_dir=$1

	[ -d "$package_dir" ] || return 1

	/usr/syno/bin/synosetkeyvalue /etc/synoinfo.conf pkg_source_trust_level 2

	ManutilPackList=`cat $package_dir/install_list`
	for mPackage in $ManutilPackList
	do
		echo install $mPackage
		install_package $package_dir/$mPackage
	done
}

start()
{
	if [ -d /.SynoUpgradePackages ]; then
		install_builtin_packages /.SynoUpgradePackages
		rm -rf /.SynoUpgradePackages
	fi

	# install packages for factory test
	if [ -d /.SynoManutilPackages ]; then
		install_manutil_packages /.SynoManutilPackages
		rm -rf /.SynoManutilPackages
	fi
}

case "$1" in
	start) start;;
	*) usage >&2; exit 1;;
esac
