#!/bin/sh

# VirtualBox package made by seba/dodo-dk/DI4bI0
# package maintained at http://www.diablos-netzwerk.de

# PATH SETTINGS
PATH=/sbin:/bin:/usr/sbin:/usr/bin:$PATH
DEVICE=/dev/vboxdrv
INSMOD=/sbin/insmod
RMMOD=/sbin/rmmod

# PACKAGE VARIABLES
PACKAGE="virtualbox"
NAME="VirtualBox"

# COMMON PACKAGE VARIABLES
PKGDEST=$(readlink "/var/packages/${PACKAGE}/target")
KERNEL=$(uname -r)
LOADED_KERNEL_MODULE=$(lsmod | grep vboxdrv)
DIST=$(get_key_value /etc.defaults/synoinfo.conf unique)
DSMMAJOR=$(get_key_value /etc.defaults/VERSION majorversion)
DSMMINOR=$(get_key_value /etc.defaults/VERSION minorversion)
DSMBUILD=$(get_key_value /etc.defaults/VERSION buildnumber)

# KERNEL SETTINGS
KERNEL_BUILD="15152"

# Use kernel module based on Synology model
if [ $(echo $DIST | grep x86_64) ]; then
	KERNEL_ARCH="x86_64"
elif [ $(echo $DIST | grep cedarview) ]; then
	KERNEL_ARCH="cedarview"
elif [ $(echo $DIST | grep bromolow) ]; then
	KERNEL_ARCH="bromolow"
elif [ $(echo $DIST | grep evansport) ]; then
	KERNEL_ARCH="evansport"
elif [ $(echo $DIST | grep avoton) ]; then
	KERNEL_ARCH="avoton"
elif [ $(echo $DIST | grep braswell) ]; then
	KERNEL_ARCH="braswell"
elif [ $(echo $DIST | grep grantley) ]; then
	KERNEL_ARCH="grantley"
elif [ $(echo $DIST | grep broadwell) ]; then
	KERNEL_ARCH="broadwell"
elif [ $(echo $DIST | grep apollolake) ]; then
	KERNEL_ARCH="apollolake"
else
	KERNEL_ARCH="x86_64"
fi

KERNEL_MODULE_DIR="${PKGDEST}/kernel_module/${KERNEL_BUILD}/${KERNEL_ARCH}/${KERNEL}"

[ -f /etc/vbox/vbox.cfg ] && source /etc/vbox/vbox.cfg

export BUILD_TYPE
export USERNAME
export USER=$USERNAME

VBOXMANAGE="$INSTALL_DIR/VBoxManage"

fail_msg() { echo -e " ...fail!\n${1}"; }
succ_msg() { echo " ...done."; }
begin_msg() { echo -n "$1"; }
failure() { fail_msg "$1"; exit 4; }
running() { lsmod | grep -q "$1[^_-]"; }

start()
{
	begin_msg "Starting VirtualBox kernel modules"
	if ! running vboxdrv; then
		if ! rm -f $DEVICE; then
			failure "Cannot remove $DEVICE"
		fi
		if ! ${INSMOD} ${KERNEL_MODULE_DIR}/vboxdrv.ko > /dev/null 2>&1; then
			failure "modprobe vboxdrv failed. Please use 'dmesg' to find out why"
		fi
		sleep 2
	fi
	# ensure the character special exists
	if [ ! -c $DEVICE ]; then
		MAJOR=`sed -n 's;\([0-9]\+\) vboxdrv$;\1;p' /proc/devices`
		if [ ! -z "$MAJOR" ]; then
			MINOR=0
		else
			MINOR=`sed -n 's;\([0-9]\+\) vboxdrv$;\1;p' /proc/misc`
			if [ ! -z "$MINOR" ]; then
				MAJOR=10
			fi
		fi
		if [ -z "$MAJOR" ]; then
			${RMMOD} ${KERNEL_MODULE_DIR}/vboxdrv.ko 2>/dev/null
			failure "Cannot locate the VirtualBox device"
		fi
		if ! mknod -m 0660 $DEVICE c $MAJOR $MINOR 2>/dev/null; then
			${RMMOD} ${KERNEL_MODULE_DIR}/vboxdrv.ko 2>/dev/null
			failure "Cannot create device $DEVICE with major $MAJOR and minor $MINOR"
		fi
	fi
	# ensure permissions
	if ! chown :root $DEVICE 2>/dev/null; then
		${RMMOD} ${KERNEL_MODULE_DIR}/vboxpci.ko 2>/dev/null
		${RMMOD} ${KERNEL_MODULE_DIR}/vboxnetadp.ko 2>/dev/null
		${RMMOD} ${KERNEL_MODULE_DIR}/vboxnetflt.ko 2>/dev/null
		${RMMOD} ${KERNEL_MODULE_DIR}/vboxdrv.ko 2>/dev/null
		failure "Cannot change group root for device $DEVICE"
	fi
	if ! ${INSMOD} ${KERNEL_MODULE_DIR}/vboxnetflt.ko > /dev/null 2>&1; then
		failure "modprobe vboxnetflt failed. Please use 'dmesg' to find out why"
	fi
	if ! ${INSMOD} ${KERNEL_MODULE_DIR}/vboxnetadp.ko > /dev/null 2>&1; then
		failure "modprobe vboxnetadp failed. Please use 'dmesg' to find out why"
	fi
	if ! ${INSMOD} ${KERNEL_MODULE_DIR}/vboxpci.ko > /dev/null 2>&1; then
		failure "modprobe vboxpci failed. Please use 'dmesg' to find out why"
	fi
	succ_msg
}

stop()
{
	begin_msg "Stopping VirtualBox kernel modules"
	if running vboxpci; then
		if ! ${RMMOD} ${KERNEL_MODULE_DIR}/vboxpci.ko 2>/dev/null; then
			failure "Cannot unload module vboxpci"
		fi
	fi
	if running vboxnetadp; then
		if ! ${RMMOD} ${KERNEL_MODULE_DIR}/vboxnetadp.ko 2>/dev/null; then
			failure "Cannot unload module vboxnetadp"
		fi
	fi
	if running vboxdrv; then
		if running vboxnetflt; then
			if ! ${RMMOD} ${KERNEL_MODULE_DIR}/vboxnetflt.ko 2>/dev/null; then
				failure "Cannot unload module vboxnetflt"
			fi
		fi
		if ! ${RMMOD} ${KERNEL_MODULE_DIR}/vboxdrv.ko 2>/dev/null; then
			failure "Cannot unload module vboxdrv"
		fi
		if ! rm -f $DEVICE; then
			failure "Cannot unlink $DEVICE"
		fi
	fi
	succ_msg
}

# enter the following variables in vbox.cfg:
#   SHUTDOWN_USERS="foo bar"
#     check for running VMs of user foo and user bar
#   SHUTDOWN=poweroff
#   SHUTDOWN=acpibutton
#   SHUTDOWN=savestate
#     select one of these shutdown methods for running VMs
stop_vms()
{
	wait=0
	for i in $SHUTDOWN_USERS; do
		# don't create the ipcd directory with wrong permissions!
		if [ -d /tmp/.vbox-$i-ipc ]; then
			export VBOX_IPC_SOCKETID="$i"
			VMS=`$VBOXMANAGE --nologo list runningvms | sed -e 's/^".*".*{\(.*\)}/\1/' 2>/dev/null`
			if [ -n "$VMS" ]; then
				if [ "$SHUTDOWN" = "poweroff" ]; then
					begin_msg "Powering off remaining VMs"
					for v in $VMS; do
						$VBOXMANAGE --nologo controlvm $v poweroff
					done
					succ_msg
				elif [ "$SHUTDOWN" = "acpibutton" ]; then
					begin_msg "Sending ACPI power button event to remaining VMs"
					for v in $VMS; do
						$VBOXMANAGE --nologo controlvm $v acpipowerbutton
						wait=60
					done
					succ_msg
				elif [ "$SHUTDOWN" = "savestate" ]; then
					begin_msg "Saving state of remaining VMs"
					for v in $VMS; do
						$VBOXMANAGE --nologo controlvm $v savestate
					done
					succ_msg
				fi
			fi
		fi
	done
	# wait for some seconds when doing ACPI shutdown
	if [ "$wait" -ne 0 ]; then
		begin_msg "Waiting for $wait seconds for VM shutdown"
		sleep $wait
		succ_msg
	fi
}

dmnstatus()
{
	if running vboxdrv; then
		str="vboxdrv"
		if running vboxnetflt; then
			str="$str, vboxnetflt"
			if running vboxnetadp; then
				str="$str, vboxnetadp"
			fi
		fi
		if running vboxpci; then
			str="$str, vboxpci"
		fi
		echo "VirtualBox kernel modules ($str) are loaded."
		for i in $SHUTDOWN_USERS; do
			# don't create the ipcd directory with wrong permissions!
			if [ -d /tmp/.vbox-$i-ipc ]; then
				export VBOX_IPC_SOCKETID="$i"
				VMS=`$VBOXMANAGE --nologo list runningvms | sed -e 's/^".*".*{\(.*\)}/\1/' 2>/dev/null`
				if [ -n "$VMS" ]; then
					echo "The following VMs are currently running:"
					for v in $VMS; do
						echo "  $v"
					done
				fi
			fi
		done
	else
		echo "VirtualBox kernel module is not loaded."
	fi
}

case "$1" in
	start)
		start
	;;
	stop)
		stop_vms
		stop
	;;
	stop_vms)
		stop_vms
	;;
	restart)
		stop && start
	;;
	force-reload)
		stop
		start
	;;
	status)
		dmnstatus
	;;
	*)
		echo "Usage: $0 {start|stop|stop_vms|restart|force-reload|status}"
		exit 1
	;;
esac
	
exit 0
