import json
import subprocess
import constant


def check_response(response_string, query_type, pluginid):
    if not response_string:
        return _err_parse_failed('response is empty')

    try:
        response_obj = json.loads(response_string, strict=False)

    except Exception as e:
        return _err_parse_failed('response format is not json')

    if not response_obj['success']:
        if 'error_code' in response_obj:
            return _err_failed('plugin success is false', response_obj['error_code'])
        return _err_parse_failed('plugin success is false')

    if 'result' not in response_obj:
        return _err_parse_failed('plugin response has no result')

    if len(response_obj['result']) == 0:
        return _err_parse_failed('result length is 0')

    check_list = _get_check_list(query_type, pluginid)
    success, msg = _check_data(check_list, response_obj['result'][0])

    if not success:
        return _err_parse_failed(msg)

    return {'success': True}


def _err_failed(error_msg, error_code):
    return {'success': False, 'error_code': error_code, 'msg': error_msg}


def _err_parse_failed(error_msg):
    return _err_failed(error_msg, constant.ERROR_PLUGIN_PARSE_RESULT_FAIL)


def _create_check_item(path, value_type):
    return {'path': path, 'value_type': value_type}


def _get_check_list(query_type, pluginid):
    check_list = []

    if query_type == 'movie':
        check_list.append(_create_check_item(['title'], str))
        check_list.append(_create_check_item(['original_available'], str))
        check_list.append(_create_check_item(['summary'], str))
        check_list.append(_create_check_item(['genre'], list))
        check_list.append(_create_check_item(['actor'], list))
        check_list.append(_create_check_item(['director'], list))
        check_list.append(_create_check_item(['writer'], list))

    elif query_type == 'tvshow':
        check_list.append(_create_check_item(['title'], str))
        check_list.append(_create_check_item(['original_available'], str))
        check_list.append(_create_check_item(['summary'], str))

    elif query_type == 'tvshow_episode':
        check_list.append(_create_check_item(['title'], str))
        check_list.append(_create_check_item(['original_available'], str))
        check_list.append(_create_check_item(['summary'], str))
        check_list.append(_create_check_item(['genre'], list))
        check_list.append(_create_check_item(['actor'], list))
        check_list.append(_create_check_item(['director'], list))
        check_list.append(_create_check_item(['writer'], list))
        check_list.append(_create_check_item(['season'], int))
        check_list.append(_create_check_item(['episode'], int))
        check_list.append(_create_check_item(['extra', pluginid, 'tvshow', 'title'], str))
        check_list.append(_create_check_item(['extra', pluginid, 'tvshow', 'original_available'], str))
    return check_list


def _check_data(check_list, data_obj):
    for idx, item in enumerate(check_list):
        attribute_list = item['path']
        expect_type = item['value_type']

        success, msg = _check_data_attribute_exist_and_type_valid(data_obj, attribute_list, expect_type)

        if not success:
            return False, msg
    return True, ''


def _check_data_attribute_exist_and_type_valid(data_obj, attribute_list, expect_type):
    # check attribute exist
    is_attribute_exist = True
    now_obj = data_obj
    for attr in attribute_list:
        if not now_obj:
            is_attribute_exist = False
            break

        if isinstance(now_obj, list):
            now_obj = now_obj[attr]
            continue

        if attr not in now_obj:
            is_attribute_exist = False
            break
        now_obj = now_obj[attr]

    if not is_attribute_exist:
        attribute_msg = str(attribute_list) + 'does not exist'
        return False, attribute_msg

    # check type correct
    if not now_obj:
        return False, str(attribute_list) + 'is empty'

    if not isinstance(now_obj, expect_type):
        type_err_msg = 'should be ' + str(expect_type)
        return False, type_err_msg

    return True, ''
