<?php

define('PLUGINID', 'com.synology.TheMovieDb');
define('TV_PLUGINID', 'com.synology.TheMovieDb.Tv');
define('API_URL', 'https://api.themoviedb.org/3/');
define('BANNER_URL', 'https://image.tmdb.org/t/p/w500');
define('BACKDROUP_URL', 'https://image.tmdb.org/t/p/original');
define('DEFAULT_EXPIRED_TIME', 86400);
define('DEFAULT_LONG_EXPIRED_TIME', 30*86400);

$DEFAULT_TYPE = 'movie';
$DEFAULT_LANG = 'enu';

function ConvertToAPILang($lang)
{
	static $map = array(
		'chs' => 'zh-CN', 'cht' => 'zh-TW', 'csy' => 'cs', 'dan' => 'da',
		'enu' => 'en', 'fre' => 'fr', 'ger' => 'de', 'hun' => 'hu',
		'ita' => 'it', 'jpn' => 'ja', 'krn' => 'ko', 'nld' => 'nl',
		'nor' => 'no', 'plk' => 'pl', 'ptb' => 'pt-BR', 'ptg' => 'pt-PT',
		'rus' => 'ru', 'spn' => 'es', 'sve' => 'sv', 'trk' => 'tr',
		'tha' => 'th'
	);

	$ret = isset($map[$lang]) ? $map[$lang] : NULL;
	return $ret;
}

/**
 * @brief download rawdata from website. If we already cache the
 *  	  result, just return cached result
 * @param $url [in] a reuqest url
 * @param $cache_path [in] a expected cache path
 * @param $expire_time [in] a expire time of cache
 * @return [out] a json format result
 */
function DownloadRawdata($url, $cache_path, $expire_time)
{
	$json = FALSE;
	$need_refresh = TRUE;

	//Whether cache file already exist or not
	if (file_exists($cache_path)) {
		$lastupdated = filemtime($cache_path);
		if ($expire_time >= (time() - $lastupdated)) {
			$json = json_decode(@file_get_contents($cache_path));
			if (NULL !== $json) {
				$need_refresh = FALSE;
			}
		}
	}

	//If we need refresh cache file, grab rawdata from url website
	if ($need_refresh) {
		//create dir
		$path_parts = pathinfo($cache_path);
		if (!file_exists($path_parts['dirname'])) {
			mkdir($path_parts['dirname']);
		}

		//download
		$fh = fopen($cache_path, 'w');
		if (FALSE === $fh) {
			syslog(LOG_ERR, "failed to fopen path [$cache_path]");
			throw new Exception();
		}
		$response = HTTPGETDownload($url, $fh);
		fclose($fh);

		if (FALSE === $response) {
			@unlink($cache_path);
		} else {
			$json = json_decode(@file_get_contents($cache_path));
			if (NULL === $json || (isset($json->success) && !$json->success)) {
				$json = FALSE;
				@unlink($cache_path);
			}
		}
	}

	return $json;
}

/**
 * @brief get each kind of rawdata
 * @param $type [in] a kind of rawdata
 * @param $options [in] it may contain parameters for some kind of rawdata
 * @param $expire_time [in] a expire time of cache
 * @return [out] a json format result
 */
function GetRawdata($type, $options, $expire_time)
{
	$url = $cache_path = NULL;
	$apikey = getenv("METADATA_PLUGIN_APIKEY");

	if (0 == strcmp($type, "search")) {
		$lang = $options['lang'];
		$year = $options['year'];
		$query = $options['query'];
		$encoded_query = urlencode($query);
		$queries = array(
			'api_key' => $apikey,
			'language' => $lang,
			'query' => $query
		);
		if ($year) {
			$queries['year'] = $year;
		}
		$url		= API_URL . "search/movie?" . http_build_query($queries);
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/query/{$encoded_query}_{$year}_{$lang}.json";
	} else if (0 == strcmp($type, "translation")) {
		$id			= $options['id'];
		$url 		= API_URL . "movie/{$id}/translations?api_key=" . $apikey;
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/translation.json";
	} else if (0 == strcmp($type, "movie")) {
		$id = $options['id'];
		$lang = $options['lang'];
		$queries = array(
			'api_key' => $apikey,
			'language' => $lang,
			'append_to_response' => 'credits,releases'
		);
		$url		= API_URL . "movie/{$id}?" . http_build_query($queries);
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/{$lang}.json";
	} else if (0 == strcmp($type, "similar")) {
		$id 		= $options['id'];
		$lang 		= $options['lang'];
		$page		= $options['page'];
		$url 		= API_URL . "movie/{$id}/similar?api_key=" . $apikey . "&language={$lang}&page={$page}";
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/{$lang}_{$page}_similar.json";
	} else if (0 == strcmp($type, "collection")) {
		// example: http://api.themoviedb.org/3/collection/8650?api_key=a862673b8d112fc23117e548e7e83916&page=2
		$id 		= $options['id'];
		$lang 		= $options['lang'];
		$url 		= API_URL . "collection/{$id}?api_key=" . $apikey . "&language={$lang}";
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/{$lang}_collection.json";
	}

	return DownloadRawdata($url, $cache_path, $expire_time);
}

function GetTvRawdata($type, $options, $expire_time)
{
	$url = $cache_path = NULL;
	$plugin_data_dir = GetPluginDataDirectory(TV_PLUGINID);
	$apikey = getenv("METADATA_PLUGIN_APIKEY");

	if (0 == strcmp($type, "search")) {
		$lang = $options['lang'];
		$year = $options['year'];
		$query = $options['query'];
		$encoded_query = urlencode($query);
		$queries = array(
			'api_key' => $apikey,
			'language' => $lang,
			'query' => $query
		);

		$url = API_URL . "search/tv?" . http_build_query($queries);
		$cache_path = "{$plugin_data_dir}/query/{$encoded_query}_{$year}_{$lang}.json";
	} else if (0 == strcmp($type, "translation")) {
		$id = $options['id'];
		$url = API_URL . "tv/{$id}/translations?api_key=" . $apikey;
		$cache_path = "{$plugin_data_dir}/{$id}/translation.json";
	} else if (0 == strcmp($type, "tv")) {
		$id = $options['id'];
		$lang = $options['lang'];
		$queries = array(
			'api_key' => $apikey,
			'language' => $lang,
			'append_to_response' => 'credits,content_ratings,external_ids'
		);
		$url = API_URL . "tv/{$id}?" . http_build_query($queries);
		$cache_path = "{$plugin_data_dir}/{$id}/{$lang}.json";
	} else if (0 == strcmp($type, "episode")) {
		$id = $options['id'];
		$lang = $options['lang'];
		$season = $options['season'];
		$episode = $options['episode'];
		$queries = array(
			'api_key' => $apikey,
			'language' => $lang,
			'append_to_response' => 'credits'
		);
		$url = API_URL . "tv/{$id}/season/{$season}/episode/{$episode}?" . http_build_query($queries);
		$cache_path = "{$plugin_data_dir}/{$id}/{$lang}_s{$season}_e{$episode}.json";
	} else {
		syslog(LOG_ERR, "unknown type: $type. failed to get raw data");
		return null;
	}

	return DownloadRawdata($url, $cache_path, $expire_time);
}

function ParseTranslation($translation_data)
{
	$translations = array();
	foreach($translation_data->translations as $item) {
		$translations[] = $item->iso_639_1;
		$translations[] = $item->iso_639_1 . '-' . $item->iso_3166_1;
	}
	return $translations;
}

function IsTranslationAvailable($type, $id, $lang)
{
	if ($type === 'movie') {
		$translation_data = GetRawdata("translation", array('id' => $id), DEFAULT_EXPIRED_TIME);
	} elseif ($type === 'tv') {
		$translation_data = GetTvRawdata("translation", array('id' => $id), DEFAULT_EXPIRED_TIME);
	} else {
		syslog(LOG_ERR, "invalid type: $type");
		return FALSE;
	}

	if ($translation_data) {
		$translations = ParseTranslation($translation_data);

		if (in_array($lang, $translations)) {
			return TRUE;
		}
	}
	return FALSE;
}

/**
 * @brief get a searching result array
 * @param $query [in] a query kwyword
 * @param $lang [in] a language
 * @param $limit [in] a limit number of result array
 * @return [out] a result array
 */
function QueryMovie($query, $year, $lang, $limit)
{
	$result = array();

	$search_data = GetRawdata('search', array('query' => $query, 'lang' => $lang, 'year' => $year), DEFAULT_EXPIRED_TIME);
	if (!$search_data || !isset($search_data->results)) {
		return $result;
	}

	foreach($search_data->results as $item) {
		$data = array();
		$data['id'] 	= $item->id;

		if (!IsTranslationAvailable('movie', $data['id'], $lang)) {
			continue;
		}
		$data['lang'] = $lang;

		$data['diff'] = 1000;
		if (isset($item->release_date)) {
			$item_year = ParseYear((string)$item->release_date);
			$data['diff'] = abs($item_year - $year);
		}

		if ($year && $data['diff'] >= 2) {
			continue;
		}
		$result[] = $data;

		// leave loop when hit limit, to avoid extra translation request
		if ($limit > 0 && count($result) >= $limit) {
			break;
		}
	}

	//If no result
	if (!count($result)) {
		return $result;
	}

	//Get the first $limit items
	$result = array_slice($result, 0, $limit);

	return $result;
}

function QueryTV($query, $year, $lang, $limit)
{
	$result = array();

	$search_data = GetTvRawdata('search', array('query' => $query, 'lang' => $lang), DEFAULT_EXPIRED_TIME);
	if (!$search_data || !isset($search_data->results)) {
		return $result;
	}

	foreach ($search_data->results as $item) {
		$data = array();
		$data['id'] = $item->id;

		$data['diff'] = 1000;
		if (isset($item->first_air_date) && !empty($item->first_air_date)) {
			$item_year = ParseYear((string)$item->first_air_date);
			$data['diff'] = abs($item_year - $year);
		}

		if ($year && $data['diff'] >= 2) {
			continue;
		}

		// Get translation
		if (!IsTranslationAvailable('tv', $data['id'], $lang)) {
			continue;
		}
		$data['lang'] = $lang;

		$result[] = $data;

		// leave loop when hit limit, to avoid extra translation request
		if ($limit > 0 && count($result) >= $limit) {
			break;
		}
	}

	// If no result
	if (!count($result)) {
		return $result;
	}

	// Get the first $limit items
	$result = array_slice($result, 0, $limit);

	return $result;
}
