<?php

require_once(dirname(__FILE__) . '/../util_themoviedb.php');
require_once(dirname(__FILE__) . '/../search.inc.php');

$SUPPORTED_TYPE = array('movie');
$SUPPORTED_PROPERTIES = array();

/**
 * @brief parse movie's data from response in JSON format
 * @param $movie [in] a movie's data in JSON format
 * @return [out] a movies's data in our format
 */
function GetMovieInfo($movie){
	$data['title']				 	= $movie->title;
	$data['original_title']			= $movie->original_title;
	$data['original_available'] 	= $movie->release_date;

	//extra
	$data['extra'] = array();
	$data['extra'][PLUGINID] = array('reference' => array());
	$data['extra'][PLUGINID]['reference']['themoviedb'] = $movie->id;
	if ((float)$movie->vote_average) {
		$data['extra'][PLUGINID]['rating'] = array('themoviedb' => (float)$movie->vote_average);
	}
	if (isset($movie->poster_path)) {
		 $data['extra'][PLUGINID]['poster'] = array(BANNER_URL . $movie->poster_path);
	}
	if (isset($movie->backdrop_path)) {
		 $data['extra'][PLUGINID]['backdrop'] = array(BACKDROUP_URL . $movie->backdrop_path);
	}
	if (isset($movie->popularity)) {
		 $data['extra'][PLUGINID]['popularity'] = array('themoviedb' => (float)$movie->popularity);
	}
	if (isset($movie->belongs_to_collection)) {
		 $data['extra'][PLUGINID]['collection_id'] = $movie->belongs_to_collection->id;
	}

	return $data;
}

/**
 * @brief get the movie's collection id if the movie is in the one of collections
 * @param $id [in] the TMDB id of movie
 * @param $lang [in] a language to set the output language
 * @return [out] the movie's collection id. If the movie isn't in any collections, return -1.
 */
function GetCollectionId($id, $lang){
	$movie_data = GetRawdata("movie", array('id' => $id, 'lang' => $lang), DEFAULT_LONG_EXPIRED_TIME);
	if (!IsValidMovies($movie_data)) {
		/* echo "Error: invalid TMDB id."; */
		return -1;
	}
	$data = GetMovieInfo($movie_data);

	return isset($data['extra'][PLUGINID]['collection_id'])?
		$data['extra'][PLUGINID]['collection_id']: -1;
}

/**
 * @brief parse the result from similar and collection API into the individual movie object
 * @param $movies [in] a array contains multiple movies
 * @param $limit [in] an integer to set the maximum number of output movies
 * @param &$result [out] an output array that contains multiple movie objects
 * @return [out] return 1 if the limit is reached.
 */
function ParseMovies($movies, $limit, &$result){
	static $id_cache = array();
	// Parse multiple similar movies
	foreach ($movies as $movie) {
		$data = GetMovieInfo($movie);
		if (!in_array($data['extra'][PLUGINID]['reference']['themoviedb'], $id_cache, true)) {
			// If it is not duplicate, append to result
			$result[] = $data;
			$id_cache[] = $data['extra'][PLUGINID]['reference']['themoviedb'];
		}

		if (count($result) >= $limit) {
			return 1;
		}
	}
	return 0;
}

/**
 * @brief is a valid movies object responded by similar API
 * @param $response [in] response by similar API
 * @return [out] true: a valid movies object, false: empty response or error message
 */
function IsValidMovies($response)
{
	if (!$response) {
		return false;
	}
	if (isset($response->status_code) && isset($response->status_message)) {
		return false;
	}

	return true;
}

/**
 * @brief get next page number. The page starts from 1, not 0.
 * @param $response [in] response by a TMDB query
 * @return [out] >= 1, next page number. -1: doesn't have next page
 */
function GetNextPage($response)
{
	if($response->total_pages <= $response->page){
		return -1;
	}

	return ($response->page+1);
}

/**
 * @brief get similar movies by collection and similar API
 * @param $query_data [in] a array contains multiple movies
 * @param $lang [in] a language to set the output language
 * @param $limit [in] an integer to set the maximum number of output movies
 * @return [out] a result array
 */
function GetSimilarMovies($query_data, $lang, $limit)
{
	global $DATA_TEMPLATE;

	// Foreach query result
	$result = array();
	foreach($query_data as $item) {
		// If languages are different, skip it
		if (0 != strcmp($item['lang'], $lang)) {
			continue;
		}

		$response = null;
		$isFull = false;

		if (0 >= $item['collection_id']) {
			// It contains two cases:
			//   1. movie isn't belong to any collections
			//   2. movie is belong to a collection but the collection's id isn't stored in db.
			$item['collection_id'] = GetCollectionId($item['id'], $item['lang']);
		}

		if ($item['collection_id'] > 0) {
			// Find the movies in the same collection first
			$response = GetRawdata("collection", array('id' => $item['collection_id'], 'lang' => $item['lang']), DEFAULT_LONG_EXPIRED_TIME);
			if (IsValidMovies($response)) {
				$isFull = ParseMovies($response->parts, $limit, $result);
				if ($isFull){
					break;
				}
			}
		}

		$page = 1;
		//Get similar movies
		do {
			$response = GetRawdata("similar", array('id' => $item['id'], 'lang' => $item['lang'], 'page' => $page), DEFAULT_LONG_EXPIRED_TIME);
			if (IsValidMovies($response)) {
				$isFull = ParseMovies($response->results, $limit, $result);
				if ($isFull) {
					break;
				}
			}
		} while (($page = GetNextPage($response)) > 0);

		if ($isFull){
			break;
		}
	}
	return $result;
}

/**
 * @brief Check the input parameter is valid or not.
 * @param $lang [in] a language to set the output language
 * @param $id [in] tmdb_id
 * @param $collectionId [in] tmdb_collection_id
 * @param $title [in] title
 * @return [out] true: is valid. false: is invalid.
 */
function isValidParams($lang, $id, $collectionId, $title){
	if (!$lang) {
		/* echo "Error: invalid parameter: lang."; */
		return false;
	}

	if (0 >= $id && !$title) {
		/* echo "Error: invalid parameter: you must specify title or tmdb_id."; */
		return false;
	}

	return true;
}

/**
 * @brief process input parameters. Assign default value if empty.
 * @param $lang [in] a language to set the output language
 * @param $input [in] a array contains tmdb_id, tmdb_collection_id, title, original_available
 * @return [out] a result array contains language, tmdb_id, tmdb_collection_id, title, year
 */
function ProcessParams($lang, $input){
	$lang = ConvertToAPILang($lang);
	$id = isSet($input['tmdb_id'])? $input['tmdb_id']: -1;
	$collectionId = isSet($input['tmdb_collection_id'])? $input['tmdb_collection_id']: -1;
	$title = isSet($input['title'])? $input['title']: null;
	$year = ParseYear($input['original_available']);

	return array($lang, $id, $collectionId, $title, $year);
}

/**
 * @brief get similar movies by id or guessing title
 * @param $input [in] a array contains title, original_available, tmdb_id, tmdb_collection_id
 * @param $lang [in] a language to set the output language
 * @param $limit [in] an integer to set the maximum number of output movies
 * @return [out] a result array contains multiple similar movies in Video Station
 */
function Similar($input, $lang, $type, $limit)
{
	list($lang, $id, $collectionId, $title, $year) = ProcessParams($lang, $input);
	if(!IsValidParams($lang, $id, $collectionId, $title)){
		return array();
	}

	if (0 < $id) {
		// If haved tmdb_id, output similar movies directly.
		return GetSimilarMovies(array(array('id' => $id, 'collection_id' => $collectionId, 'lang' => $lang)), $lang, $limit);
	}

	// Doesn't hava tmdb_id, use title tio guess movie's id
	$queryData = array();
	$titles = GetGuessingList($title, true);
	foreach ($titles as $checkTitle) {
		if (empty($checkTitle)) {
			continue;
		}
		$queryData = QueryMovie($checkTitle, $year, $lang, $limit);
		if (0 < count($queryData)) {
			break;
		}
	}

	// Get similar movies
	return GetSimilarMovies($queryData, $lang, $limit);
}

SimilarRun('Similar');
?>
