<?php

define('PLUGINID', 'com.synology.TheMovieDb');
define('API_URL', 'https://api.themoviedb.org/3/');
define('BANNER_URL', 'https://image.tmdb.org/t/p/w500');
define('BACKDROUP_URL', 'https://image.tmdb.org/t/p/original');
define('APIKEY', 'a862673b8d112fc23117e548e7e83916');
define('DEFAULT_EXPIRED_TIME', 86400);
define('DEFAULT_LONG_EXPIRED_TIME', 30*86400);

$DEFAULT_TYPE = 'movie';
$DEFAULT_LANG = 'enu';

function ConvertToAPILang($lang)
{
	static $map = array(
		'chs' => 'zh', 'cht' => 'zh', 'csy' => 'cs', 'dan' => 'da',
		'enu' => 'en', 'fre' => 'fr', 'ger' => 'de', 'hun' => 'hu',
		'ita' => 'it', 'jpn' => 'ja', 'krn' => 'ko', 'nld' => 'nl',
		'nor' => 'no', 'plk' => 'pl', 'ptb' => 'pt', 'ptg' => 'pt',
		'rus' => 'ru', 'spn' => 'es', 'sve' => 'sv', 'trk' => 'tr'
	);

	$ret = isset($map[$lang]) ? $map[$lang] : NULL;
	return $ret;
}

/**
 * @brief download rawdata from website. If we already cache the
 *  	  result, just return cached result
 * @param $url [in] a reuqest url
 * @param $cache_path [in] a expected cache path
 * @param $expire_time [in] a expire time of cache
 * @return [out] a json format result
 */
function DownloadRawdata($url, $cache_path, $expire_time)
{
	$json = FALSE;
	$need_refresh = TRUE;

	//Whether cache file already exist or not
	if (file_exists($cache_path)) {
		$lastupdated = filemtime($cache_path);
		if ($expire_time >= (time() - $lastupdated)) {
			$json = json_decode(@file_get_contents($cache_path));
			if (NULL !== $json) {
				$need_refresh = FALSE;
			}
		}
	}

	//If we need refresh cache file, grab rawdata from url website
	if ($need_refresh) {
		//create dir
		$path_parts = pathinfo($cache_path);
		if (!file_exists($path_parts['dirname'])) {
			mkdir($path_parts['dirname']);
		}

		//download
		$fh = fopen($cache_path, 'w');
		if (FALSE === $fh) {
			throw new Exception();
		}
		$response = HTTPGETDownload($url, $fh);
		fclose($fh);

		if (FALSE === $response) {
			@unlink($cache_path);
		} else {
			$json = json_decode(@file_get_contents($cache_path));
			if (NULL === $json) {
				$json = FALSE;
				@unlink($cache_path);
			}
		}
	}

	return $json;
}

/**
 * @brief get each kind of rawdata
 * @param $type [in] a kind of rawdata
 * @param $options [in] it may contain parameters for some kind of rawdata
 * @param $expire_time [in] a expire time of cache
 * @return [out] a json format result
 */
function GetRawdata($type, $options, $expire_time)
{
	$url = $cache_path = NULL;

	if (0 == strcmp($type, "search")) {
		$query 		= urlencode($options['query']);
		$lang 		= $options['lang'];
		$url 		= API_URL . "search/movie?api_key=" . APIKEY . "&query={$query}&language={$lang}";
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/query/{$query}_{$lang}.json";
	} else if (0 == strcmp($type, "translation")) {
		$id			= $options['id'];
		$url 		= API_URL . "movie/{$id}/translations?api_key=" . APIKEY;
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/translation.json";
	} else if (0 == strcmp($type, "movie")) {
		$id			= $options['id'];
		$lang 		= $options['lang'];
		$url 		= API_URL . "movie/{$id}?api_key=" . APIKEY . "&language={$lang}";
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/{$lang}.json";
	} else if (0 == strcmp($type, "cast")) {
		$id 		= $options['id'];
		$url 		= API_URL . "movie/{$id}/casts?api_key=" . APIKEY;
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/cast.json";
	} else if (0 == strcmp($type, "releases")) {
		$id 		= $options['id'];
		$url 		= API_URL . "movie/{$id}/releases?api_key=" . APIKEY;
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/releases.json";
	} else if (0 == strcmp($type, "similar")) {
		$id 		= $options['id'];
		$lang 		= $options['lang'];
		$page		= $options['page'];
		$url 		= API_URL . "movie/{$id}/similar?api_key=" . APIKEY . "&language={$lang}&page={$page}";
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/{$lang}_{$page}_similar.json";
	} else if (0 == strcmp($type, "collection")) {
		// example: http://api.themoviedb.org/3/collection/8650?api_key=a862673b8d112fc23117e548e7e83916&page=2
		$id 		= $options['id'];
		$lang 		= $options['lang'];
		$url 		= API_URL . "collection/{$id}?api_key=" . APIKEY . "&language={$lang}";
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/{$lang}_collection.json";
	}

	return DownloadRawdata($url, $cache_path, $expire_time);
}

function ParseTranslation($translation_data)
{
	$translations = array();
	foreach($translation_data->translations as $item) {
		$translations[] = $item->iso_639_1;
	}
	return $translations;
}

/**
 * @brief get a searching result array
 * @param $query [in] a query kwyword
 * @param $lang [in] a language
 * @param $limit [in] a limit number of result array
 * @return [out] a result array
 */
function QueryMovie($query, $year, $lang, $limit)
{
	$result = array();

	$search_data = GetRawdata('search', array('query' => $query, 'lang' => $lang), DEFAULT_EXPIRED_TIME);
	if (!$search_data || !isset($search_data->results)) {
		return $result;
	}

	foreach($search_data->results as $item) {
		$data = array();
		$data['id'] 	= $item->id;
		$data['lang'] 	= 'en';

		//Get translation
		$translation_data = GetRawdata("translation", array('id' => $data['id']), DEFAULT_EXPIRED_TIME);
		if ($translation_data) {
			$translations = ParseTranslation($translation_data);
			if (in_array($lang, $translations)) {
				$data['lang'] = $lang;
			}
		}

		$data['diff'] = 1000;
		if (isset($item->release_date)) {
			$item_year = ParseYear((string)$item->release_date);
			$data['diff'] = abs($item_year - $year);
		}

		if ($year && $data['diff'] >= 2) {
			continue;
		}
		$result[] = $data;
	}

	//If no result
	if (!count($result)) {
		return $result;
	}

	//Get the first $limit items
	$result = array_slice($result, 0, $limit);

	return $result;
}

function Query($title, $year, $lang, $limit)
{
	$result = array();

	//Use title with year as keyword
	if ($year) {
		$query = $title . ' ' . $year;
		$result = QueryMovie($query, $year, $lang, $limit);
	}

	//If no result, use title as keyword to search again
	if (!count($result)) {
		$query = $title;
		$result = QueryMovie($query, $year, $lang, $limit);
	}

	return $result;
}

?>
