#!/usr/bin/php
<?php

define('PLUGINID', 'com.synology.TheMovieDb');
define('API_URL', 'http://api.themoviedb.org/3/');
define('BANNER_URL', 'http://image.tmdb.org/t/p/w500');
define('BACKDROUP_URL', 'http://image.tmdb.org/t/p/original');
define('APIKEY', 'a862673b8d112fc23117e548e7e83916');

$DEFAULT_TYPE = 'movie';
$DEFAULT_LANG = 'enu';

$SUPPORTED_TYPE = array('movie');
$SUPPORTED_PROPERTIES = array('title');

require_once(dirname(__FILE__) . '/../search.inc.php');

function ConvertToAPILang($lang)
{
	static $map = array(
		'chs' => 'zh', 'cht' => 'zh', 'csy' => 'cs', 'dan' => 'da',
		'enu' => 'en', 'fre' => 'fr', 'ger' => 'de', 'hun' => 'hu',
		'ita' => 'it', 'jpn' => 'ja', 'krn' => 'ko', 'nld' => 'nl',
		'nor' => 'no', 'plk' => 'pl', 'ptb' => 'pt', 'ptg' => 'pt',
		'rus' => 'ru', 'spn' => 'es', 'sve' => 'sv', 'trk' => 'tr'
	);

	$ret = isset($map[$lang]) ? $map[$lang] : NULL;
	return $ret;
}

/**
 * @brief download rawdata from website. If we already cache the
 *  	  result, just return cached result
 * @param $url [in] a reuqest url
 * @param $cache_path [in] a expected cache path
 * @return [out] a json format result
 */
function DownloadRawdata($url, $cache_path)
{
	$json = FALSE;
	$need_refresh = TRUE;

	//Whether cache file already exist or not
	if (file_exists($cache_path)) {
		$lastupdated = filemtime($cache_path);
		if (86400 >= (time() - $lastupdated)) {
			$json = json_decode(@file_get_contents($cache_path));
			if (NULL !== $json) {
				$need_refresh = FALSE;
			}
		}
	}

	//If we need refresh cache file, grab rawdata from url website
	if ($need_refresh) {
		//create dir
		$path_parts = pathinfo($cache_path);
		if(!file_exists($path_parts['dirname'])) {
			mkdir($path_parts['dirname']);
		}

		//download
		$fh = fopen($cache_path, 'w');
		if (FALSE === $fh) {
			throw new Exception();
		}
		$response = HTTPGETDownload($url, $fh);
		fclose($fh);

		if (FALSE === $response) {
			@unlink($cache_path);
		} else {
			$json = json_decode(@file_get_contents($cache_path));
			if (NULL === $json) {
				$json = FALSE;
				@unlink($cache_path);
			}
		}
	}

	return $json;
}

/**
 * @brief get each kind of rawdata
 * @param $type [in] a kind of rawdata
 * @param $options [in] it may contain parameters for some kind
 *  			   of rawdata
 * @return [out] a json format result
 */
function GetRawdata($type, $options)
{
	$url = $cache_path = NULL;

	if (0 == strcmp($type, "search")) {
		$query 		= urlencode($options['query']);
		$lang 		= $options['lang'];
		$url 		= API_URL . "search/movie?api_key=" . APIKEY . "&query={$query}&language={$lang}";
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/query/{$query}_{$lang}.json";
	} else if (0 == strcmp($type, "translation")) {
		$id			= $options['id'];
		$url 		= API_URL . "movie/{$id}/translations?api_key=" . APIKEY;
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/translation.json";
	} else if (0 == strcmp($type, "movie")) {
		$id			= $options['id'];
		$lang 		= $options['lang'];
		$url 		= API_URL . "movie/{$id}?api_key=" . APIKEY . "&language={$lang}";
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/{$lang}.json";
	} else if (0 == strcmp($type, "cast")) {
		$id 		= $options['id'];
		$url 		= API_URL . "movie/{$id}/casts?api_key=" . APIKEY;
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/cast.json";
	} else if (0 == strcmp($type, "releases")) {
		$id 		= $options['id'];
		$url 		= API_URL . "movie/{$id}/releases?api_key=" . APIKEY;
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$id}/releases.json";
	}

	return DownloadRawdata($url, $cache_path);
}

function GetMovieInfo($movie_data, $data)
{
    $data['title']				 	= $movie_data->title;
	$data['original_title']			= $movie_data->original_title;
    $data['tagline'] 				= $movie_data->tagline;
    $data['original_available'] 	= $movie_data->release_date;
	$data['summary'] 				= $movie_data->overview;

	foreach ($movie_data->genres as $item) {
		if (!in_array($item->name, $data['genre'])) {
			array_push($data['genre'], $item->name);
		}
    }

	//extra
	$data['extra'] = array();
	$data['extra'][PLUGINID] = array('reference' => array());
	$data['extra'][PLUGINID]['reference']['themoviedb'] = $movie_data->id;
	if (isset($movie_data->imdb_id)) {
		 $data['extra'][PLUGINID]['reference']['imdb'] = $movie_data->imdb_id;
	}
	if ((float)$movie_data->vote_average) {
		$data['extra'][PLUGINID]['rating'] = array('themoviedb' => (float)$movie_data->vote_average);
	}
	if (isset($movie_data->poster_path)) {
		 $data['extra'][PLUGINID]['poster'] = array(BANNER_URL . $movie_data->poster_path);
	}
	if (isset($movie_data->backdrop_path)) {
		 $data['extra'][PLUGINID]['backdrop'] = array(BACKDROUP_URL . $movie_data->backdrop_path);
	}

    return $data;
}

function GetCastInfo($cast_data, $data)
{
    // actor
	foreach ($cast_data->cast as $item) {
		if (!in_array($item->name, $data['actor'])) {
			array_push($data['actor'], $item->name);
		}
    }

    // director & writer
	foreach ($cast_data->crew as $item) {
		if (strcasecmp($item->department, 'Directing') == 0) {
			if (!in_array($item->name, $data['director'])) {
				array_push($data['director'], $item->name);
			}
        }
		if (strcasecmp($item->department, 'Writing') == 0) {
			if (!in_array($item->name, $data['writer'])) {
				array_push($data['writer'], $item->name);
			}
        }
    }

    return $data;
}

function GetCertificateInfo($releases_data, $data)
{
	$certificate = array();
	foreach ($releases_data->countries as $item) {
		if ('' === $item->certification) {
			continue;
		}
		$name = strcasecmp($item->iso_3166_1, 'us') == 0 ? 'USA' : $item->iso_3166_1;
		$certificate[$name] = $item->certification;
	}
	$data['certificate'] = $certificate;
    return $data;
}

function ParseTranslation($translation_data)
{
	$translations = array();
	foreach($translation_data->translations as $item) {
		$translations[] = $item->iso_639_1;
	}
	return $translations;
}

/**
 * @brief get metadata for multiple movies
 * @param $query_data [in] a array contains multiple movie item
 * @param $lang [in] a language
 * @return [out] a result array
 */
function GetMetadata($query_data, $lang)
{
	global $DATA_TEMPLATE;

	//Foreach query result
	$result = array();
	foreach($query_data as $item) {
		//If languages are different, skip it
		if (0 != strcmp($item['lang'], $lang)) {
			continue;
		}

        //Copy template
		$data = $DATA_TEMPLATE;

		//Get movie
		$movie_data = GetRawdata("movie", array('id' => $item['id'], 'lang' => $item['lang']));
		if (!$movie_data) {
			continue;
		}
		$data = GetMovieInfo($movie_data, $data);

		//Get cast
		$cast_data = GetRawdata("cast", array('id' => $item['id']));
		if ($cast_data) {
            $data = GetCastInfo($cast_data, $data);
        }

		//Get certificates
		$releases_data = GetRawdata("releases", array('id' => $item['id']));
		if ($releases_data) {
			$data = GetCertificateInfo($releases_data, $data);
		}

		//Append to result
		$result[] = $data;
	}

	return $result;
}

/**
 * @brief get a searching result array
 * @param $query [in] a query kwyword
 * @param $lang [in] a language
 * @param $limit [in] a limit number of result array
 * @return [out] a result array
 */
function QueryMovie($query, $year, $lang, $limit)
{
	$result = array();

	$search_data = GetRawdata('search', array('query' => $query, 'lang' => $lang));
	if (!$search_data) {
		return $result;
	}

	foreach($search_data->results as $item) {
		$data = array();
		$data['id'] 	= $item->id;
		$data['lang'] 	= 'en';

		//Get translation
		$translation_data = GetRawdata("translation", array('id' => $data['id']));
		if ($translation_data) {
			$translations = ParseTranslation($translation_data);
			if (in_array($lang, $translations)) {
				$data['lang'] = $lang;
			}
		}

		$data['diff'] = 1000;
		if (isset($item->release_date)) {
			$item_year = ParseYear((string)$item->release_date);
			$data['diff'] = abs($item_year - $year);
		}

		if ($year && $data['diff'] >= 2) {
			continue;
		}
		$result[] = $data;
	}

	//If no result
	if (!count($result)) {
		return $result;
	}

	//Get the first $limit items
	$result = array_slice($result, 0, $limit);

	return $result;
}


function Query($title, $year, $lang, $limit)
{
	$result = array();

	//Use title with year as keyword
	if ($year) {
		$query = $title . ' ' . $year;
		$result = QueryMovie($query, $year, $lang, $limit);
	}

	//If no result, use title as keyword to search again
	if (!count($result)) {
		$query = $title;
		$result = QueryMovie($query, $year, $lang, $limit);
	}

	return $result;
}

function Process($input, $lang, $type, $limit, $search_properties, $allowguess, $id)
{
	$title 	= $input['title'];
	$year 	= ParseYear($input['original_available']);
	$lang 	= ConvertToAPILang($lang);
	if (!$lang) {
		return array();
	}

	if (0 < $id) {
		// if haved id, output metadata directly.
		return GetMetadata(array(array('id' => $id, 'lang' => $lang)), $lang);
	}

	//Search
	$query_data = array();
	$titles = GetGuessingList($title, $allowguess);
	foreach ($titles as $checkTitle) {
		if (empty($checkTitle)) {
			continue;
		}
		$query_data = Query($checkTitle, $year, $lang, $limit);
		if (0 < count($query_data)) {
			break;
		}
	}

	//Get metadata
	return GetMetadata($query_data, $lang);
}

PluginRun('Process');
?>
