<?php
define('PKG_INSTALL_DIR', dirname(__FILE__));
require_once(dirname(__FILE__) . '/utils.php');

$DATA_TEMPLATE = array(
	'path' => NULL,
	'title' => NULL,
	'tagline' => NULL,
	'original_available' => NULL,
	'summary' => NULL,
	'season' => NULL,
	'episode' => NULL,
	'writer' => array(),
	'director' => array(),
	'actor' => array(),
	'genre' => array(),
	'extra' => NULL
);

function RemovePluginData()
{
	// when this function is called 1000 times, we only
	// remove cache data one time.
	$randval = rand(0, 1000);
	if ($randval != 0) {
		return;
	}

	// remove cache data (mtime is greater than 1 day)
	$path = PKG_INSTALL_DIR . '/plugin_data/';
	$path = escapeshellarg($path);
	@exec("/usr/bin/find $path -mtime +1 -delete");
}

function GetPluginDataDirectory($pluginid)
{
	RemovePluginData();

	$path = PKG_INSTALL_DIR . '/plugin_data/' . $pluginid;
	@mkdir($path, 0777, true);
	return $path;
}

/**
 * @brief calculate the number of valid tokens
 * @param $token [in] a string
 * @return [out] the number of valid tokens
 */
function WordCount($token)
{
	$filter = array('a', 'an', 'the', 'of', 'in', 'on', 'at', 'for', 'by');

	if (!is_array($token)) {
		$token = preg_split('/ /', $token, -1, PREG_SPLIT_NO_EMPTY);
	}

	$count = 0;
	foreach ($token as $term) {
		if (!in_array(strtolower($term), $filter)) {
			$count++;
		}
	}

	return $count;
}

/**
 * @brief omit some tokens with mixed digit and english
 *  	  character. Just remain pure language text
 * @param $text [in]
 * @param $only_english [in]
 * @return [out] a string
 */
function PureLangText($text, $only_english)
{
	$allNum = true;
	$token = array();
	$data = preg_split('/ /', $text, -1, PREG_SPLIT_NO_EMPTY);

	foreach ($data as $term) {
		/* skip "Top001" like terms */
		if (preg_match('/[0-9]/ui', $term) && preg_match('/[a-z]/ui', $term)) {
			continue;
		}

		//all digit characters
		if (preg_match('/^[0-9]+$/', $term)) {
			$token[] = $term;
		//all english characters
		} else if ($only_english && preg_match('/^[a-z]+$/ui', $term)) {
			$allNum = false;
			$token[] = $term;
		//all non english characters
		} else if (!$only_english && !preg_match('/^[a-z]+$/ui', $term)) {
			$allNum = false;
			$token[] = $term;
		}
	}

	if ($allNum) {
		return "";
	}
	return implode(' ', $token);
}

function TextCutoff($text, $max_count, $from_right)
{
	$token = preg_split('/ /', $text, -1, PREG_SPLIT_NO_EMPTY);
	$newWords = $origWords = WordCount($token);

	while (1 < count($token) && $max_count > ($origWords - $newWords)) {
		if ($from_right) {
			array_pop($token);
		} else {
			array_shift($token);
		}
		$newWords = WordCount($token);
	}

	return implode(' ', $token);
}

function GetGuessingList($title, $allowguess)
{
	if (!$allowguess) {
		return array($title);
	}

	$engTitle = PureLangText($title, true);

	if (empty($engTitle)) {
		$engTitle = $title;
	}

	$titles = array(
		$title,
		PureLangText($title, false)
	);

	if (2 <= WordCount($engTitle)) {
		$titles = array_merge($titles, array(
			$engTitle
		));
	}

	if (3 <= WordCount($engTitle)) {
		$titles = array_merge($titles, array(
			TextCutoff($engTitle, 1, true),
			TextCutoff($engTitle, 1, false)
		));
	}

	if (4 <= WordCount($engTitle)) {
		$titles = array_merge($titles, array(
			TextCutoff(TextCutoff($engTitle, 1, false), 1, true),
			TextCutoff($engTitle, 2, true),
			TextCutoff($engTitle, 2, false)
		));
	}

	if (6 <= WordCount($engTitle)) {
		$titles = array_merge($titles, array(
			TextCutoff(TextCutoff($engTitle, 2, false), 2, true)
		));
	}

	return $titles;
}

function HTTPGETDownload($url, $fh)
{
	$ch = curl_init();
	if (FALSE === $ch) {
		return FALSE;
	}

	_SetCurlOptions($ch);
	curl_setopt($ch, CURLOPT_URL, $url);
	curl_setopt($ch, CURLOPT_FILE, $fh);

	$result = _CurlExecRetry($ch, 2);
	curl_close($ch);
	return $result;
}

function HTTPGETRequest($url)
{
	$ch = curl_init();
	if (FALSE === $ch) {
		return FALSE;
	}

	_SetCurlOptions($ch);
	curl_setopt($ch, CURLOPT_URL, $url);
	curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

	$result = _CurlExecRetry($ch, 2);
	curl_close($ch);
	return $result;
}

function ParseYear($original_available)
{
	$data = date_parse($original_available);
	if ($data && !empty($data['year'])) {
		return $data['year'];
	}

	return $original_available;
}

function _CurlExecRetry($ch, $count)
{
	$result = FALSE;

	do {
		$result = curl_exec($ch);
	} while (FALSE === $result && 0 < $count--);

	return $result;
}

function _SetCurlOptions($ch)
{
	global $_cookie_file;

	curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10_5_1; de-de) AppleWebKit/527+ (KHTML, like Gecko) Version/3.1.1 Safari/525.20');
	curl_setopt($ch, CURLOPT_TIMEOUT, 45);
	curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30);
	curl_setopt($ch, CURLOPT_FAILONERROR, false);
	curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
	curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
	curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
	curl_setopt($ch, CURLOPT_COOKIEFILE, $_cookie_file);
	curl_setopt($ch, CURLOPT_COOKIEJAR, $_cookie_file);
}

function _GetSearchType($options)
{
	global $SUPPORTED_TYPE, $DEFAULT_TYPE;

	if (isSet($options['type']) &&
		in_array($options['type'], $SUPPORTED_TYPE)) {
		return $options['type'];
	}
	return $DEFAULT_TYPE;
}

function _GetSearchLang($options)
{
	global $DEFAULT_LANG;

	if (!isSet($options['lang'])) {
		return $DEFAULT_LANG;
	}

	return $options['lang'];
}

function _GetSearchLimit($options)
{
	if (!isSet($options['limit']) || !is_numeric($options['limit'])) {
		return 1;
	}
	return $options['limit'];
}

function _GetAllowGuess($options)
{
	if (isSet($options['allowguess'])) {
		if ('false' == $options['allowguess'] || '0' == $options['allowguess']) {
			return false;
		}
	}
	return true;
}

function _CopyNecessaryInput($data) {
	global $DATA_TEMPLATE;
	$keys = array('path', 'season', 'episode');

	foreach($keys as $key) {
		if (isSet($data[$key])) {
			$DATA_TEMPLATE[$key] = $data[$key];
		}
	}
}

function _GetSearchInput($options)
{
	global $DATA_TEMPLATE;

	if (isSet($options['input'])) {
		$input = $options['input'];
	} else {
		$input = fgets(STDIN);
	}

	$result = json_decode($input, true);

	/* multiple input */
	if (is_array($result) && !_IsDataObject($result)) {
		$result = $result[0];
	}

	/* invalid format */
	if (!_IsDataObject($result)) {
		return $DATA_TEMPLATE;
	}

	_CopyNecessaryInput($result);

	return array_merge($DATA_TEMPLATE, $result);
}

function _GetSearchProperties($options)
{
	global $SUPPORTED_PROPERTIES;

	if (!isSet($options['search'])) {
		return array();
	}

	$properties = explode(',', $options['search']);
	foreach ($properties as $key => $value) {
		if (!in_array($value, $SUPPORTED_PROPERTIES)) {
			unset($properties[$key]);
		}
	}

	return $properties;
}

function _IsDataObject($data)
{
	return is_array($data) && (isSet($data['path']) || isSet($data['title']));
}

function _PluginInit()
{
	global $_cookie_file;

	$_cookie_file = tempnam('/tmp', 'plugin_cookie_');
}

function _PluginDeinit()
{
	global $_cookie_file;

	unlink($_cookie_file);
}

function PluginRun($callback)
{
	$result = array();
	$options = getopt("h", array(
		"help",
		"input:",
		"lang:",
		"type:",
		"limit:",
		"search:",
		"allowguess:"
	));

	if (isSet($options['h']) || isSet($options['help'])) {
		echo $_SERVER['argv'][0] . ' [--input JSON_Object] [--lang enu|cht|...] [--type movie|tvshow|...] [--limit n] [--search title[,tagline]...]' . "\n";
		exit(0);
	}

	$input = _GetSearchInput($options);
	$lang = _GetSearchLang($options);
	$type = _GetSearchType($options);
	$properties = _GetSearchProperties($options);
	$limit = _GetSearchLimit($options);
	$allowguess = _GetAllowGuess($options);

	_PluginInit();
	try {
		$result = $callback($input, $lang, $type, $limit, $properties, $allowguess);
	} catch (Exception $e) {
//		echo $e . "\n";
	}
	_PluginDeinit();

	/* normalize to array */
	if (_IsDataObject($result)) {
		$result = array($result);
	}

	/* workaround for unsupported JSON_UNESCAPED_UNICODE */
	array_walk_recursive($result, function(&$item, $key) {
		if(is_string($item)) {
			$item = urlencode(addcslashes(htmlspecialchars_decode($item), '"\\'));
		}
	});
	echo urldecode(json_encode($result)) . "\n";
}
?>
