#!/usr/bin/php
<?php

define('PLUGINID', 'com.synology.Allocine');
define('API_URL_FR', 'http://api.allocine.fr/rest/v3/');		//France
define('API_URL_EN', 'http://api.screenrush.co.uk/rest/v3/');	//United-Kingdom
define('API_URL_DE', 'http://api.filmstarts.de/rest/v3/');		//Deutschland
define('API_URL_ES', 'http://api.sensacine.com/rest/v3/');		//Espana
define('PARTNER_CODE', 'YW5kcm9pZC12M3M');

$DEFAULT_TYPE = 'movie';
$DEFAULT_LANG = 'fre';

$SUPPORTED_TYPE = array('movie', 'tvshow', 'tvshow_episode');
$SUPPORTED_PROPERTIES = array('title');

require_once(dirname(__FILE__) . '/../search.inc.php');

function ConvertToAPILang($lang)
{
	static $map = array(
		'fre' => 'fr',
		'enu' => 'en',
		'ger' => 'de',
		'spn' => 'es'
	);

	$ret = isset($map[$lang]) ? $map[$lang] : NULL;
	return $ret;
}

function GetApiUrl($lang)
{
	$url = NULL;
	switch ($lang) {
		case "fr":
			$url = API_URL_FR;
			break;
		case "en":
			$url = API_URL_EN;
			break;
		case "de":
			$url = API_URL_DE;
			break;
		case "es":
			$url = API_URL_ES;
			break;
	}
	return $url;
}

/**
 * @brief download rawdata from website. If we already cache the
 *  	  result, just return cached result
 * @param $url [in] a reuqest url
 * @param $cache_path [in] a expected cache path
 * @return [out] a json format result
 */
function DownloadRawdata($url, $cache_path)
{
	$json = FALSE;
	$need_refresh = TRUE;

	//Whether cache file already exist or not
	if (file_exists($cache_path)) {
		$lastupdated = filemtime($cache_path);
		if (86400 >= (time() - $lastupdated)) {
			$json = json_decode(@file_get_contents($cache_path));
			if (NULL !== $json) {
				$need_refresh = FALSE;
			}
		}
	}

	//If we need refresh cache file, grab rawdata from url website
	if ($need_refresh) {
		//create dir
		$path_parts = pathinfo($cache_path);
		if(!file_exists($path_parts['dirname'])) {
			mkdir($path_parts['dirname']);
		}

		//download
		$fh = fopen($cache_path, 'w');
		if (FALSE === $fh) {
			throw new Exception();
		}
		$response = HTTPGETDownload($url, $fh);
		fclose($fh);

		if (FALSE === $response) {
			@unlink($cache_path);
		} else {
			$json = json_decode(@file_get_contents($cache_path));
			if (NULL === $json) {
				$json = FALSE;
				@unlink($cache_path);
			}
		}
	}

	return $json;
}

/**
 * @brief get each kind of rawdata
 * @param $type [in] a kind of rawdata
 * @param $options [in] it may contain parameters for some kind
 *  			   of rawdata
 * @return [out] a json format result
 */
function GetRawdata($type, $options)
{
	$url = $cache_path = NULL;
	$cache_dir = GetPluginDataDirectory(PLUGINID);

	if (0 == strcmp($type, "search_movie")) {
		$query 		= urlencode($options['query']);
		$lang 		= $options['lang'];
		$url 		= GetApiUrl($lang) . "search?partner=" . PARTNER_CODE . "&format=json&filter=movie&q={$query}";
		$cache_path = "{$cache_dir}/movie_query/{$query}_{$lang}.json";
	} else if (0 == strcmp($type, "get_movie")) {
		$id			= $options['id'];
		$lang 		= $options['lang'];
		$url 		= GetApiUrl($lang) . "movie?partner=" . PARTNER_CODE . "&format=json&profile=large&code={$id}";
		$cache_path = "{$cache_dir}/movie_{$id}/{$lang}.json";
	} else if (0 == strcmp($type, "search_tvshow")) {
		$query 		= urlencode($options['query']);
		$lang 		= $options['lang'];
		$url 		= GetApiUrl($lang) . "search?partner=" . PARTNER_CODE . "&format=json&filter=tvseries&q={$query}";
		$cache_path = "{$cache_dir}/tvshow_query/{$query}_{$lang}.json";
	} else if (0 == strcmp($type, "get_tvshow")) {
		$id			= $options['id'];
		$lang 		= $options['lang'];
		$url 		= GetApiUrl($lang) . "tvseries?partner=" . PARTNER_CODE . "&format=json&profile=large&code={$id}";
		$cache_path = "{$cache_dir}/tvshow_{$id}/{$lang}.json";
	} else if (0 == strcmp($type, "get_season")) {
		$id			= $options['id'];
		$lang 		= $options['lang'];
		$url 		= GetApiUrl($lang) . "season?partner=" . PARTNER_CODE . "&format=json&profile=large&code={$id}";
		$cache_path = "{$cache_dir}/season_{$id}/{$lang}.json";
	} else if (0 == strcmp($type, "get_episode")) {
		$id			= $options['id'];
		$lang 		= $options['lang'];
		$url 		= GetApiUrl($lang) . "episode?partner=" . PARTNER_CODE . "&format=json&profile=large&code={$id}";
		$cache_path = "{$cache_dir}/episode_{$id}/{$lang}.json";
	}

	return DownloadRawdata($url, $cache_path);
}

function SeasonCompare($a, $b)
{
    if ($a['season'] == $b['season']) {
        return 0;
    }

    return ($a['season'] < $b['season']) ? -1 : 1;
}

function EpisodeCompare($a, $b)
{
    if ($a['episode'] == $b['episode']) {
        return 0;
    }
    return ($a['episode'] < $b['episode']) ? -1 : 1;
}

/**
 * @brief this is a auxiliary list that can help sort the
 *  	  episodes by season number and episode number.
 * @param $item [in] a episode item
 * @param $list [in, out] a json format. [['season' => 1,
 *  			'episode' => [<episode 1>, <episode 2>,
 *  			...]],[['season' => 2,'episode' => [<episode 1>,
 *  			<episode 2>,...]],...]
 */
function InsertItemToList($item, &$list)
{
	$season = $item['season'];
	if (!isset($list[$season])) {
		$list[$season] = array('season' => $season, 'episode' => array());
	}
	array_push($list[$season]['episode'], $item);
}

function SortList(&$list) {
	uasort($list, 'SeasonCompare');

	foreach($list as $key => $value) {
		uasort($list[$key]['episode'], 'EpisodeCompare');
	}

	$list = array_values($list);
}

function ParseCastMember(&$castMember, &$data)
{
	$data['actor'] 		= array();
	$data['director'] 	= array();
	$data['writer'] 	= array();

	foreach ($castMember as $item) {
		//actor (8001)
		if ($item->activity->code == 8001 && !empty($item->person->name)) {
			array_push($data['actor'], $item->person->name);
		//director (8002)
		} else if ($item->activity->code == 8002 && !empty($item->person->name)) {
			array_push($data['director'], $item->person->name);
		//writer (8004)
		} else if ($item->activity->code == 8004 && !empty($item->person->name)) {
			array_push($data['writer'], $item->person->name);
		}
	}

	if (15 < count($data['actor'])) {
		$data['actor'] = array_slice($data['actor'], 0, 15);
	}
	if (15 < count($data['director'])) {
		$data['director'] = array_slice($data['director'], 0, 15);
	}
	if (15 < count($data['writer'])) {
		$data['writer'] = array_slice($data['writer'], 0, 15);
	}
}

function m_GetMovieInfo($movie_data, $data)
{
	if (isset($movie_data->movie->title)) {
		$data['title'] = $movie_data->movie->title;
	} else {
		$data['title'] = $movie_data->movie->originalTitle;
	}
	if (isset($movie_data->movie->originalTitle)) {
		$data['original_title'] = $movie_data->movie->originalTitle;
	}

	//original available
	$productionYear = isset($movie_data->movie->release->releaseDate) ? (string)$movie_data->movie->productionYear : NULL;
	$releaseDate 	= isset($movie_data->movie->release->releaseDate) ? (string)$movie_data->movie->release->releaseDate : NULL;
	if (isset($productionYear) && isset($releaseDate)) {
		$item_year = (int)ParseYear($releaseDate);
		$data['original_available'] = ((int)$productionYear == $item_year) ? $releaseDate : $productionYear;
	} else if (isset($productionYear)) {
		$data['original_available'] = $productionYear;
	} else if (isset($releaseDate)) {
		$data['original_available'] = $releaseDate;
	}

	//summary
	if (isset($movie_data->movie->synopsis)) {
		$data['summary'] = strip_tags($movie_data->movie->synopsis);
	}

	//genre
	foreach ($movie_data->movie->genre as $item) {
		array_push($data['genre'], $item->{'$'});
    }

	//cast member
	if (isset($movie_data->movie->castMember)) {
		ParseCastMember($movie_data->movie->castMember, $data);
	}

	//extra
	$data['extra'][PLUGINID]['reference']['allocine'] = $movie_data->movie->code;
	if (!empty($movie_data->movie->statistics->userRating)) {
		//Normalize rating
		$rating = (float)$movie_data->movie->statistics->userRating;
		$rating = round($rating * 2, 1);
		$data['extra'][PLUGINID]['rating'] = array('allocine' => $rating);
	}
	if (isset($movie_data->movie->poster->href)) {
		 $data['extra'][PLUGINID]['poster'] = array($movie_data->movie->poster->href);
	}

	return $data;
}

/**
 * @brief get metadata for multiple movies
 * @param $query_data [in] a array contains multiple movie item
 * @param $lang [in] a language
 * @return [out] a result array
 */
function m_GetMetadata($query_data, $lang)
{
	global $DATA_TEMPLATE;

	//Foreach query result
	$result = array();
	foreach($query_data as $item) {
		//If languages are different, skip it
		if (0 != strcmp($item['lang'], $lang)) {
			continue;
		}

        //Copy template
		$data = $DATA_TEMPLATE;

		//Get movie
		$movie_data = GetRawdata("get_movie", array('id' => $item['id'], 'lang' => $item['lang']));
		if (!$movie_data) {
			continue;
		}
		$data = m_GetMovieInfo($movie_data, $data);

		//Append to result
		$result[] = $data;
	}

	return $result;
}

/**
 * @brief get a searching result array
 * @param $query [in] a query kwyword
 * @param $lang [in] a language
 * @param $limit [in] a limit number of result array
 * @return [out] a result array
 */
function m_QueryMovie($query, $year, $lang, $limit)
{
	$result = array();

	$search_data = GetRawdata('search_movie', array('query' => $query, 'lang' => $lang));
	if (!$search_data || $search_data->feed->totalResults == 0) {
		return $result;
	}

	//Get all items
	foreach($search_data->feed->movie as $item) {
		$data = array();
		$data['id'] 	= $item->code;
		$data['lang'] 	= $lang;

		$data['diff']	= 1000;
		if (isset($item->productionYear)) {
			$item_year = (int)$item->productionYear;
			$data['diff'] = abs($item_year - $year);
		} else if (isset($item->release->releaseDate)) {
			$item_year = ParseYear((string)$item->release->releaseDate);
			$data['diff'] = abs($item_year - $year);
		}

		if ($year && $data['diff'] > 2) {
			continue;
		}
		$result[] = $data;
	}

	//If no result
	if (!count($result)) {
		return $result;
	}

	//Get the first $limit items
	$result = array_slice($result, 0, $limit);

	return $result;
}

function m_Query($title, $year, $lang, $limit)
{
	$result = array();

	//This plugin doesn't support $title with $year as query keyword
	//So we just use $title as query keyword directly
	$query = $title;
	$result = m_QueryMovie($query, $year, $lang, $limit);

	return $result;
}

function t_ParseTVShowData(&$series_data, $data)
{
	if (isset($series_data->tvseries->title)) {
		$data['title'] = $series_data->tvseries->title;
	} else {
		$data['title'] = $series_data->tvseries->originalTitle;
	}
	if (isset($series_data->tvseries->originalTitle)) {
		$data['original_title'] = $series_data->tvseries->originalTitle;
	}
	if (isset($series_data->tvseries->originalBroadcast->dateStart)) {
		$data['original_available'] = $series_data->tvseries->originalBroadcast->dateStart;
	}
	if (isset($series_data->tvseries->synopsis)) {
		$data['summary'] = strip_tags($series_data->tvseries->synopsis);
	}

	//genre
	$data['genre'] = array();
	foreach ($series_data->tvseries->genre as $item) {
		array_push($data['genre'], $item->{'$'});
    }

	//cast member
	if (isset($series_data->tvseries->castMember)) {
		ParseCastMember($series_data->tvseries->castMember, $data);
	}

	//extra
	$data['extra'][PLUGINID]['reference']['allocine'] = $series_data->tvseries->code;
	if (!empty($series_data->tvseries->statistics->userRating)) {
		//Normalize rating
		$rating = (float)$series_data->tvseries->statistics->userRating;
		$rating = round($rating * 2, 1);
		$data['extra'][PLUGINID]['rating'] = array('allocine' => $rating);
	}
	if (isset($series_data->tvseries->poster->href)) {
		 $data['extra'][PLUGINID]['poster'] = array($series_data->tvseries->poster->href);
	}

	return $data;
}

function t_ParseEpisodeData(&$series_data, &$episode_data, $season, $episode, $data)
{
	$data['season']  = (int)$season;
	$data['episode'] = (int)$episode;

	if (isset($episode_data->episode->title)) {
		$data['tagline'] = $episode_data->episode->title;
	} else {
		$data['tagline'] = $episode_data->episode->originalTitle;
	}

	if (isset($episode_data->episode->originalTitle)) {
		$data['original_title'] = $episode_data->episode->originalTitle;
	}
	if (isset($episode_data->episode->synopsis)) {
		$data['summary'] = strip_tags($episode_data->episode->synopsis);
	}

	//genre
	$data['genre'] = array();
	foreach ($series_data->tvseries->genre as $item) {
		array_push($data['genre'], $item->{'$'});
    }

	//cast member
	if (isset($episode_data->episode->castMember)) {
		ParseCastMember($episode_data->episode->castMember, $data);
	}

	//extra
	$data['extra'][PLUGINID]['reference']['allocine'] = $episode_data->episode->code;
	if (!empty($episode_data->episode->statistics->userRating)) {
		//Normalize rating
		$rating = (float)$episode_data->episode->statistics->userRating;
		$rating = round($rating * 2, 1);
		$data['extra'][PLUGINID]['rating'] = array('allocine' => $rating);
	}

	if (isset($episode_data->episode->media)) {
		$data['extra'][PLUGINID]['poster'] = array();
		foreach($episode_data->episode->media as $item) {
			if ($item->type->{'$'} == 'Photo') {
				$data['extra'][PLUGINID]['poster'][] = $item->thumbnail->href;
			}
		}
	}

	return $data;
}

function t_ParseCommonEpisodeData(&$season_data)
{
	$item = array();
	$item['season'] = (int)$season_data->season->seasonNumber;
	$item['original_available'] = (string)$season_data->season->yearStart;

	if (isset($season_data->season->castMember)) {
		ParseCastMember($season_data->season->castMember, $item);
	}

	return $item;
}

function t_ParseSelfEpisodeData(&$episode_data, $item)
{
	$item['episode'] = (int)$episode_data->episodeNumberSeason;
	$item['title'] 	 = isset($episode_data->title) ? (string)$episode_data->title : (string)$episode_data->originalTitle;

	if (isset($episode_data->synopsis)) {
		$item['summary'] = strip_tags($episode_data->synopsis);
	}

	//extra
	$item['extra'][PLUGINID]['reference'] = array();
	$item['extra'][PLUGINID]['reference']['allocine'] = (string)$episode_data->code;

	return $item;
}
function t_GetTVShowInfo(&$series_data, &$seasons_data, $lang, $data)
{
	//Fill tvshow information
	$data = t_ParseTVShowData($series_data, $data);

	//Fill all episode information
	$list = array();
	foreach ($seasons_data as $season_data) {
		$item = t_ParseCommonEpisodeData($season_data);

		if (isset($season_data->season->episode)) {
			foreach ($season_data->season->episode as $episode_data) {
				$item = t_ParseSelfEpisodeData($episode_data, $item);
				InsertItemToList($item, $list);
			}
		}
	}
	SortList($list);
	$data['extra'][PLUGINID]['list'] = $list;

	return $data;
}

function t_GetEpisodeInfo(&$series_data, &$season_data, $season, $episode, $lang, $data)
{
	$episode_id = FALSE;

	//Find episode id
	if ($season_data) {
		foreach ($season_data->season->episode as $item) {
			if ($item->episodeNumberSeason == $episode) {
				$episode_id = (string)$item->code;
				break;
			}
		}
	}

	//Fill tvshow information
	$data['title'] = isset($series_data->tvseries->title) ? (string)$series_data->tvseries->title : (string)$series_data->tvseries->originalTitle;
	$data['extra'] = array(PLUGINID => array());
	$data['extra'][PLUGINID]['tvshow'] = t_ParseTVShowData($series_data, array());

	//Fill episode information
	if ($episode_id) {
		$episode_data = GetRawdata("get_episode", array('id' => $episode_id, 'lang' => $lang));
		if ($episode_data) {
			$data = t_ParseEpisodeData($series_data, $episode_data, $season, $episode, $data);
		}
	}

	return $data;
}

function t_GetMetadata($query_data, $season, $episode, $lang, $type)
{
	global $DATA_TEMPLATE;

	//Foreach query result
	$result = array();
	foreach($query_data as $item) {
		//If languages are different, skip it
		if (0 != strcmp($item['lang'], $lang)) {
			continue;
		}

        //Copy template
		$data = $DATA_TEMPLATE;

		//Get series
		$series_data = GetRawdata("get_tvshow", array('id' => $item['id'], 'lang' => $item['lang']));
		if (!$series_data) {
			continue;
		}
		if (0 == (int)$series_data->tvseries->seasonCount) {
			continue;
		}

		//Get season id
		$season_ids = array();
		foreach ($series_data->tvseries->season as $season_item) {
			$season_ids[$season_item->seasonNumber] = $season_item->code;
		}

		switch ($type) {
			case 'tvshow':
				//Get each season data
				$seasons_data = array();
				foreach ($season_ids as $season_id) {
					$season_data = GetRawdata("get_season", array('id' => $season_id, 'lang' => $item['lang']));
					if ($season_data) {
						$seasons_data[] = $season_data;
					}
				}

				$data = t_GetTVShowInfo($series_data, $seasons_data, $item['lang'], $data);

				break;

			case 'tvshow_episode':
				//Get season id
				$season  = (NULL === $season)  ? 1 : $season;
				$episode = (NULL === $episode) ? 1 : $episode;
				$season_id = isset($season_ids[$season]) ? $season_ids[$season] : NULL ;

				//Get season data
				$season_data = NULL;
				if ($season_id) {
					$season_data = GetRawdata("get_season", array('id' => $season_id, 'lang' => $item['lang']));
					if (!$season_data) {
						continue;
					}
				}

				$data = t_GetEpisodeInfo($series_data, $season_data, $season, $episode, $item['lang'], $data);
				break;
		}

		//Append to result
		$result[] = $data;
	}

	return $result;
}

function t_Query($query, $year, $lang, $limit)
{
	$result = array();

	$search_data = GetRawdata('search_tvshow', array('query' => $query, 'lang' => $lang));
	if (!$search_data || $search_data->feed->totalResults == 0) {
		return $result;
	}

	//Get all items
	foreach($search_data->feed->tvseries as $item) {
		$data = array();
		$data['id'] 	= $item->code;
		$data['lang'] 	= $lang;

		$data['diff']	= 1000;
		if (isset($item->yearStart)) {
			$item_year = (string)$item->yearStart;
			$data['diff'] = abs($item_year - $year);
		}

		if ($year && $data['diff'] > 2) {
			continue;
		}
		$result[] = $data;
	}

	//If no result
	if (!count($result)) {
		return $result;
	}

	//Get the first $limit items
	$result = array_slice($result, 0, $limit);

	return $result;
}

function Process($input, $lang, $type, $limit, $search_properties, $allowguess)
{
	$title 		= $input['title'];
	$year 		= ParseYear($input['original_available']);
	$lang 		= ConvertToAPILang($lang);
	$season  	= $input['season'];
	$episode 	= $input['episode'];
	if (!$lang) {
		return array();
	}

	//year
	if (isset($input['extra']) && count($input['extra']) > 0) {
		$pluginid = array_shift($input['extra']);
		if (!empty($pluginid['tvshow']['original_available'])) {
			$year = ParseYear($pluginid['tvshow']['original_available']);
		}
	}

	if ("movie" == $type) {
		//Search
		$query_data = array();
		$titles = GetGuessingList($title, $allowguess);
		foreach ($titles as $checkTitle) {
			if (empty($checkTitle)) {
				continue;
			}
			$query_data = m_Query($checkTitle, $year, $lang, $limit);
			if (0 < count($query_data)) {
				break;
			}
		}

		//Get metadata
		return m_GetMetadata($query_data, $lang);
	} else {
		//Search
		$query_data = array();
		$titles = GetGuessingList($title, $allowguess);
		foreach ($titles as $checkTitle) {
			if (empty($checkTitle)) {
				continue;
			}
			$query_data = t_Query($checkTitle, $year, $lang, $limit);
			if (0 < count($query_data)) {
				break;
			}
		}

		//Get metadata
		return t_GetMetadata($query_data, $season, $episode, $lang, $type);
	}

	exit(1);
}

PluginRun('Process');
?>
