#!/usr/bin/php
<?php

define('PLUGINID', 'com.synology.Freebase');
define('API_URL', 'https://www.googleapis.com/freebase/v1/');
define('DEPRECATED_API_URL', 'http://api.freebase.com/api/');
define('TRANS_URL', 'http://www.freebase.com/api/trans/raw/guid/');

$DEFAULT_TYPE = 'movie';
$DEFAULT_LANG = 'enu';

$SUPPORTED_TYPE = array('movie');
$SUPPORTED_PROPERTIES = array('title');

require_once(dirname(__FILE__) . '/../search.inc.php');

function ConvertToAPILang($lang)
{
	static $map = array(
		'enu' => 'en'
	);

	$ret = isset($map[$lang]) ? $map[$lang] : NULL;
	return $ret;
}

/**
 * @brief convert a id to a human readable id name
 * @param $id [in] a id
 * @return [out] a query string
 */
function GetReadableId($id)
{
	return preg_replace('/[\/]/', '_', $id);
}

/**
 * @brief get a mql query string for movie information.
 *  	  Sometimes, if we set actors parameter, Freebase will
 *  	  get null result. If we omit actors parameters, we will
 *  	  get correct info.
 * @param $mid [in] a unique id for freebase
 * @param $with_actors [in] do we need to query with actor items
 * @return [out] a query string
 */
function GetMovieMqlString($mid, $with_actors)
{
	$query = array(
		"mid"                               => "{$mid}",
		"type"                              => "/film/film",
        "name"                              => NULL,
		"/film/film/tagline"                => array(),
		"/film/film/initial_release_date"   => NULL,
		"/film/film/written_by"             => array(),
		"/film/film/directed_by"            => array(),
		"/film/film/genre"                  => array()
    );

	if ($with_actors) {
		$query["/film/film/starring"] = array(
			array("actor" => NULL)
		);
	}

	return json_encode($query);
}

/**
 * @brief get a mql query string for movie article and
 *  	  poster.
 * @param $mid [in] a unique id for freebase
 * @return [out] a query string
 */
function GetExtraMqlString($mid)
{
	$query = array(
		"mid"                   	=> "{$mid}",
		"type"						=> "/film/film",
		"/common/topic/article"		=> array(
			"guid" 		=> NULL,
			"id"		=> NULL,
			"limit"		=> 1,
			"optional"	=> TRUE
		),
		"/common/topic/image"		=> array(
			"guid" 		=> NULL,
			"id"		=> NULL,
			"limit"		=> 1,
			"optional"	=> TRUE
		)
    );

	return json_encode($query);
}

/**
 * @brief download rawdata from website. If we already cache the
 *  	  result, just return cached result
 * @param $url [in] a reuqest url
 * @param $cache_path [in] a expected cache path
 * @return [out] a json format result
 */
function DownloadRawdata($url, $cache_path, $json)
{
	$result = FALSE;
	$need_refresh = TRUE;

	//Whether cache file already exist or not
	if (file_exists($cache_path)) {
		$lastupdated = filemtime($cache_path);
		if (86400 >= (time() - $lastupdated)) {
			$result = file_get_contents($cache_path);
			if ($json) {
				$result = json_decode($result);
			}
			if (NULL !== $result) {
				$need_refresh = FALSE;
			}
		}
	}

	//If we need refresh cache file, grab rawdata from url website
	if ($need_refresh) {
		//create dir
		$path_parts = pathinfo($cache_path);
		if(!file_exists($path_parts['dirname'])) {
			mkdir($path_parts['dirname']);
		}

		//download
		$fh = fopen($cache_path, 'w');
		if (FALSE === $fh) {
			throw new Exception();
		}
		$response = HTTPGETDownload($url, $fh);
		fclose($fh);

		if (FALSE === $response) {
			@unlink($cache_path);
		} else {
			$result = file_get_contents($cache_path);
			if ($json) {
				$result = json_decode($result);
			}
			//Sometimes, this website will return a valid json object,
			//but json->result is NULL value. It means no results.
			if (NULL === $result ||
				($json && NULL === $result->result)) {
				$result = FALSE;
				@unlink($cache_path);
			}
		}
	}

	return $result;
}

/**
 * @brief get each kind of rawdata
 * @param $type [in] a kind of rawdata
 * @param $options [in] it may contain parameters for some kind
 *  			   of rawdata
 * @return [out] a json format result
 */
function GetRawdata($type, $options)
{
	$url = $cache_path = NULL;
	$json = TRUE;

	if (0 == strcmp($type, "search")) {
		$query 		= urlencode($options['query']);
		$url 		= API_URL . "search?query={$query}&type='/film/film'&indent=true";
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/query/{$query}.json";
	} else if (0 == strcmp($type, "movie")) {
		$mid		= $options['mid'];
		$rid		= GetReadableId($mid);
		$url 		= API_URL . "mqlread?query=" . urlencode(GetMovieMqlString($mid, TRUE));
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$rid}/movie.json";
	} else if (0 == strcmp($type, "movie_without_actors")) {
		$mid		= $options['mid'];
		$rid		= GetReadableId($mid);
		$url 		= API_URL . "mqlread?query=" . urlencode(GetMovieMqlString($mid, FALSE));
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$rid}/movie.json";
	} else if (0 == strcmp($type, "query_extra")) {
		$mid		= $options['mid'];
		$rid		= GetReadableId($mid);
		$url 		= API_URL . "mqlread?query=" . urlencode(GetExtraMqlString($mid));
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$rid}/query_extra.json";
	} else if (0 == strcmp($type, "article")) {
		$mid		= $options['mid'];
		$guid		= $options['guid'];
		$rid		= GetReadableId($mid);
		$json		= FALSE;
		$url 		= TRANS_URL . $guid;
		$cache_path = GetPluginDataDirectory(PLUGINID) . "/{$rid}/article.txt";
	}

	return DownloadRawdata($url, $cache_path, $json);
}

function GetMovieInfo($movie_data, $data)
{
    $data['title'] 				= $movie_data->result->name;
    $data['tagline'] 			= @$movie_data->result->{'/film/film/tagline'}[0];
    $data['original_available'] = $movie_data->result->{'/film/film/initial_release_date'};

	$data['writer'] 			= array_values(array_unique($movie_data->result->{'/film/film/written_by'}));
	$data['director'] 			= array_values(array_unique($movie_data->result->{'/film/film/directed_by'}));
	$data['genre'] 				= array_values(array_unique($movie_data->result->{'/film/film/genre'}));

	if(isset($movie_data->result->{'/film/film/starring'})) {
		foreach ($movie_data->result->{'/film/film/starring'} as $item) {
			if (!in_array($item->actor, $data['actor']) && !empty($item->actor)) {
				array_push($data['actor'], $item->actor);
			}
		}
	}

	//extra
	$data['extra'] = array();
	$data['extra'][PLUGINID] = array('reference' => array());
	$data['extra'][PLUGINID]['reference']['freebase'] = $movie_data->result->mid;

    return $data;
}

function GetExtraInfo($query_extra_data, $data)
{
	if (!isset($query_extra_data->result)) {
		return $data;
	}

	$mid = $query_extra_data->result->mid;

	//Article
	if (isset($query_extra_data->result->{'/common/topic/article'})) {
		$guid = $query_extra_data->result->{'/common/topic/article'}->guid;
		if ($guid) {
			$guid = substr($guid, 1); //remove first '#' character
			$article_data = GetRawdata("article", array('mid' => $mid, 'guid' => $guid));
			if (!empty($article_data)) {
				$data['summary'] = strip_tags($article_data);
			}
		}
	}

	//Poster
	if (isset($query_extra_data->result->{'/common/topic/image'})) {
		$guid = $query_extra_data->result->{'/common/topic/image'}->guid;
		if ($guid) {
			$guid = substr($guid, 1); //remove first '#' character
			$data['extra'][PLUGINID]['poster'] = array(TRANS_URL . $guid);
		}
	}

	return $data;
}

/**
 * @brief get metadata for multiple movies
 * @param $query_data [in] a array contains multiple movie item
 * @param $lang [in] a language
 * @return [out] a result array
 */
function GetMetadata($query_data, $lang)
{
	global $DATA_TEMPLATE;

	//Foreach query result
	$result = array();
	foreach($query_data as $item) {
		//If languages are different, skip it
		if (0 != strcmp($item['lang'], $lang)) {
			continue;
		}

        //Copy template
		$data = $DATA_TEMPLATE;

		//Get movie
		$movie_data = GetRawdata("movie", array('mid' => $item['mid']));
		if (!$movie_data) {
			$movie_data = GetRawdata("movie_without_actors", array('mid' => $item['mid']));
			if (!$movie_data) {
				continue;
			}
		}
		$data = GetMovieInfo($movie_data, $data);

		//Get extra
		$query_extra_data = GetRawdata("query_extra", array('mid' => $item['mid']));
		if ($query_extra_data) {
			$data = GetExtraInfo($query_extra_data, $data);
		}

		//Append to result
		$result[] = $data;
	}

	return $result;
}

/**
 * @brief get a searching result array
 * @param $query [in] a query kwyword
 * @param $lang [in] a language
 * @param $limit [in] a limit number of result array
 * @return [out] a result array
 */
function QueryMovie($query, $lang, $limit)
{
	$result = array();

	$search_data = GetRawdata('search', array('query' => $query));
	if (!$search_data) {
		return $result;
	}

	foreach($search_data->result as $item) {
		if (!isset($item->notable) || !isset($item->notable->id) || '/film/film' !== $item->notable->id) {
			continue;
		}

		$data = array();
		$data['mid'] 	= $item->mid;;
		$data['lang'] 	= 'en';

		$result[] = $data;
		if (0 === --$limit) {
			break;
		}
	}

	return $result;
}

function Query($title, $year, $lang, $limit)
{
	$result = array();

	//Freebase seems not accept year as keyword.
	//If you add year, the result get worse.
	/*
	if ($year) {
		$query = $title . ' ' . $year;
		$result = QueryMovie($query, $lang, $limit);
	}
	*/

	//If no result, use title as keyword to search again
	if (!count($result)) {
		$query = $title;
		$result = QueryMovie($query, $lang, $limit);
	}

	return $result;
}

function Process($input, $lang, $type, $limit, $search_properties)
{
	$title 	= $input['title'];
	$year 	= ParseYear($input['original_available']);
	$lang 	= ConvertToAPILang($lang);
	if (!$lang) {
		return array();
	}

	//Search
	$query_data = array();
	$titles = GetGuessingList($title);
	foreach ($titles as $checkTitle) {
		if (empty($checkTitle)) {
			continue;
		}
		$query_data = Query($checkTitle, $year, $lang, $limit);
		if (0 < count($query_data)) {
			break;
		}
	}

	//Get metadata
	return GetMetadata($query_data, $lang);
}

PluginRun('Process');

?>
