<?php
/* Copyright (c) 2000-2012 Synology Inc. All rights reserved. */
require_once('WebAPI.inc.php');
require_once('WebAPIUtil.php');

abstract class WebAPI {
	private $apiDesc;
	private $resp;
	private $error;
	private $headerSent = false;
	
	protected $api;
	protected $method;
	protected $version;
	
	function __construct($apiDescFilePath) {
		if (!file_exists($apiDescFilePath)) {
			WebAPIUtil::Log("api desc not found. path=$apiDescFilePath", basename(__FILE__), __LINE__);
			$this->apiDesc = NULL;
			goto End;
		}

		$jsonString = file_get_contents($apiDescFilePath);
		$this->apiDesc = json_decode($jsonString, true);
	End:
		return;
	}
	
	private function CheckRequestParam($api, $method, $version) {
		$ret = false;
		$apiDesc = $this->apiDesc[$api];
		$minVersion = 0;
		$maxVersion = 0;

		if (NULL === $apiDesc) {
			$this->SetError(WEBAPI_ERR_NO_SUCH_API);
			goto End;
		}

		if (!is_int($version) && !ctype_digit($version)) {
			$this->SetError(WEBAPI_ERR_BAD_REQUEST);
			goto End;
		}

		if (!isset($apiDesc[SZK_API_MIN_VERSION]) || !isset($apiDesc[SZK_API_MAX_VERSION])) {
			WebAPIUtil::Log("Bad API desc. api=$api", basename(__FILE__), __LINE__);
			$this->SetError(WEBAPI_ERR_NO_SUCH_API);
			goto End;
		}

		$minVersion = $apiDesc[SZK_API_MIN_VERSION];
		$maxVersion = $apiDesc[SZK_API_MAX_VERSION];

		if ($version < $minVersion || $maxVersion < $version) {
			$this->SetError(WEBAPI_ERR_NOT_SUPPORTED_VERSION);
			goto End;
		}

		if (!in_array($method, $apiDesc[SZK_API_METHODS][$version])) {
			$this->SetError(WEBAPI_ERR_NO_SUCH_METHOD);
			goto End;
		}

		$this->api = $api;
		$this->method = $method;
		$this->version = $version;

		$ret = true;
	End:
		return $ret;
	}

	protected function OutputSingleSpace() {
		// this function is for connection abort detection
		// PHP will not detect that the user has aborted the connection until an attempt is made to send information to the client.
		// http://www.php.net/manual/en/function.ignore-user-abort.php
		if (!$this->headerSent) {
			$this->headerSent = true;
			header('Content-type: text/plain; charset=utf-8');
		}
		echo ' ';
		ob_flush();
		flush();
	}
	
	private function OutputResponse() {
		$resp = array();
		
		if (WEBAPI_ERR_NONE != $this->error) {
			$resp[SZK_RESP_SUCCESS] = false;
			$resp[SZK_RESP_ERROR][SZK_RESP_ERROR_CODE] = $this->error;
			goto End;
		}

		$resp[SZK_RESP_SUCCESS] = true;
		if (!is_null($this->resp)) {
			$resp[SZK_RESP_DATA] = $this->resp;
		}
	End:
		if (!$this->headerSent) {
			$this->headerSent = true;
			header('Content-type: text/plain; charset=utf-8');
		}
		echo(json_encode($resp));
		//echo json_encode($resp, JSON_UNESCAPED_UNICODE);

		// transform from unicode back to utf8                   
		//echo preg_replace("#\\\u([0-9a-f]{4}+)#ie", "iconv('UCS-2', 'UTF-8', pack('H4', '\\1'))", json_encode($resp));
	}

	protected function CheckPermission() {
    	return true;
    }

	abstract protected function Process();
	
	protected function SetResponse($resp) {
		$this->resp = $resp;
		$this->error = WEBAPI_ERR_NONE;
	}
	
	protected function SetError($error) {
		$this->resp = NULL;
		$this->error = $error;
	}
   
	public function Run() {
		if (!isset($_REQUEST[SZK_PARAM_API]) || !isset($_REQUEST[SZK_PARAM_METHOD]) || !isset($_REQUEST[SZK_PARAM_VERSION])) {
			$this->SetError(WEBAPI_ERR_BAD_REQUEST);
			goto End;
		}
		
		$api = $_REQUEST[SZK_PARAM_API];
		$method = $_REQUEST[SZK_PARAM_METHOD];
		$version = $_REQUEST[SZK_PARAM_VERSION];

		if (!$this->CheckRequestParam($api, $method, $version)) {
			goto End;
		}

		if (!$this->CheckPermission()) {
			goto End;
		}

		$this->Process();

	End:
		$this->OutputResponse();
		return;
	}
}
?>
