<?php

define('SYNO_PHOTO_EXIV2_PROG', '/usr/syno/bin/exiv2');
define('SYNO_PHOTO_DSM_USER_PROG', '/usr/syno/bin/synophoto_dsm_user');
define('SYNO_PHOTO_LEGACY_KEY_PARAM', 'legacy_key');
define('SYNO_PHOTO_MERGED_KEY', 'ms_people_tag_merged');

/* get old merged key in config file */
$options = getopt("", array(SYNO_PHOTO_LEGACY_KEY_PARAM.':'));
$oldMergedKey = ('yes' === $options[SYNO_PHOTO_LEGACY_KEY_PARAM]);

/* list dsm users */
$dsmUserList = getDSMUserList();
$usernameList = array_merge(array(''), $dsmUserList);

foreach ($usernameList as $username) {
	$user = $username;
	$dbh = getDatabaseHandler();
	$escapeStr = getEscapeString();

	//no database, skip
	if (!$dbh) {
		continue;
	}

	if ($oldMergedKey) {
		setMergedKeyToDB();
		continue;
	}

	$newMergedKey = getMergedKeyFromDB();
	if ($newMergedKey) {
		continue;
	}

	$photoList = getPhotoListFromDB();
	$metadataTagCount = 0;

	foreach ($photoList as $photo) {
		//read tags from db
		$dbList = readTagListFromDB($photo);

		//read tags from metadata
		$metadataList = readTagListFromMetadata($photo['path'], $metadataTagCount);

		//rename db tag if the metadata tag with the same name exist
		renameDuplicateTagInDBList($dbList, $metadataList);

		//write tags
		writeTagListToMetadata($photo, $dbList, $metadataTagCount);
	}

	setMergedKeyToDB();

	if (strlen($username)) {
		/* Personal Photo Station */
		if(file_exists("/var/services/homes/$username/photo/.SYNOPPSDB")) {
			@exec("/usr/syno/bin/synoindex -R /var/services/homes/$username/photo");
		}
	} else {
		/* Photo Station */
		@exec("/usr/syno/bin/synoindex -R photo");
	}
}

function getDSMUserList()
{
	@exec(SYNO_PHOTO_DSM_USER_PROG." --enum", $dsmUserCount);
	@exec(SYNO_PHOTO_DSM_USER_PROG." --enum 0 ".intval($dsmUserCount[0]), $dsmUserList);
	$list = array();

	foreach($dsmUserList as &$dsmUser) {
		$dsmUser = explode(',', $dsmUser);
		if (count($dsmUser) != 6) {
			continue;
		}
		$list[] = $dsmUser[1];
	}
	return $list;
}

function getDatabaseHandler()
{
	global $user;

	$dbh = null;

	if (strlen($user)){
		$user_photo_db = "/var/services/homes/$user/photo/.SYNOPPSDB";

		if(!file_exists($user_photo_db)){
			return $dbh;
		}

		try {
			$dbh = new PDO("sqlite:{$user_photo_db}");
			$dbh->exec('PRAGMA case_sensitive_like=1');
			$dbh->exec('PRAGMA foreign_keys=ON');
		} catch (PDOException $e) {
			die($e->getMessage());
		}
	} else {
		try {
			$dbh = new PDO('pgsql:dbname=photo', 'admin', '', array(PDO::ATTR_PERSISTENT => false, PDO::ATTR_EMULATE_PREPARES => true));
		} catch (PDOException $e) {
			die($e->getMessage());
		}
	}

	return $dbh;
}

function getEscapeString()
{
	global $user;

	return strlen($user) ? "ESCAPE '\'" : "";
}

function setMergedKeyToDB()
{
	global $dbh, $escapeStr;

	$sqlCond = "SELECT * FROM photo_config WHERE module_name = 'photo' AND config_key = '".SYNO_PHOTO_MERGED_KEY."'";
	$sqlParam = array();

	$dbResult = $dbh->prepare($sqlCond);
	$dbResult->execute($sqlParam);

	if (false !== ($row = $dbResult->fetch())) {
		$id = $row['config_id'];
		$sqlCond = "UPDATE photo_config SET config_value = 'on' WHERE config_id = ".$row['config_id'];
	} else {
		$sqlCond = "INSERT INTO photo_config (module_name, config_key, config_value) VALUES ('photo', '".SYNO_PHOTO_MERGED_KEY."', 'on')";
	}

	$dbResult = $dbh->prepare($sqlCond);
	$dbResult->execute($sqlParam);
}

function getMergedKeyFromDB()
{
	global $dbh, $escapeStr;

	$sqlCond = "SELECT * FROM photo_config WHERE module_name = 'photo' AND config_key = '".SYNO_PHOTO_MERGED_KEY."'";
	$sqlParam = array();

	$dbResult = $dbh->prepare($sqlCond);
	$dbResult->execute($sqlParam);

	$ret = false;
	if (false !== ($row = $dbResult->fetch())) {
		if ('on' === $row['config_value']) {
			$ret = true;
		}
	}

	return $ret;
}

function getPhotoListFromDB()
{
	global $dbh, $escapeStr;

	$photoList = array();

	$sqlCond = 'SELECT id, path, resolutionx, resolutiony FROM photo_image WHERE id IN (SELECT DISTINCT image_id FROM photo_image_label)';
	$sqlParam = array();

	$dbResult = $dbh->prepare($sqlCond);
	$dbResult->execute($sqlParam);

	while (false !== ($row = $dbResult->fetch())) {
		if (!inAcceptableExtList($row['path'])) {
			continue;
		}

		$photoList[] = array(
			'path' => $row['path'],
			'resolutionx' => $row['resolutionx'],
			'resolutiony' => $row['resolutiony']
		);
	}

	return $photoList;
}

function inAcceptableExtList($path)
{
	$acceptableExtList = array('jpg', 'jpeg', 'jpe', 'tif', 'tiff', 'png', 'bmp');

	foreach ($acceptableExtList as $acceptableExt) {
		if (0 === substr_compare(strtolower($path), $acceptableExt, strlen($path) - strlen($acceptableExt), strlen($acceptableExt))) {
			return true;
		}
	}

	return false;
}

function readTagListFromDB($photo)
{
	global $dbh, $escapeStr;

	$tagList = array();

	$sqlCond =
		'SELECT photo_label.id, photo_label.name, photo_image_label.info FROM photo_image_label '.
		'INNER JOIN photo_label ON photo_image_label.label_id = photo_label.id '.
		'WHERE photo_label.category = 0 AND photo_image_label.status = \'t\' AND photo_image_label.image_id = '.$photo['id'];
	$sqlParam = array();

	$dbResult = $dbh->prepare($sqlCond);
	$dbResult->execute($sqlParam);

	while (false !== ($row = $dbResult->fetch())) {
		$tagList[$row['name']] = array(
			'newName' => $row['name'],
			'infoFloat' => convertDBInfoToFloat($photo['resolutionx'], $photo['resolutiony'], json_decode($row['info'], true)),
			'isDuplicate' => false
		);
	}

	return $tagList;
}

function readTagListFromMetadata($path, &$metadataTagCount)
{
	global $user;

	$tmpTagList = array();
	$tagList = array();
	$peopleTagPrefix = 'Xmp.MP.RegionInfo/MPRI:Regions[';
	$tagIndex = 1;

	$filePath = $path;
	if ('' !== $user) {
		$filePath = "/var/services/homes/$user/photo/".$filePath;
	}

	$cmd = SYNO_PHOTO_EXIV2_PROG.' -Pkt '.escapeshellarg($filePath);
	@exec($cmd, $cmdResult);

	foreach ($cmdResult as $tag) {
		if ($peopleTagPrefix !== substr($tag, 0, strlen($peopleTagPrefix))) {
			continue;
		}

		$targetTagPrefix = $peopleTagPrefix.$tagIndex.']';
		$nextTagPrefix = $peopleTagPrefix.($tagIndex + 1).']';

		if ($nextTagPrefix === substr($tag, 0, strlen($nextTagPrefix))) {
			//already arrive next people tag
			$tagIndex++;
			continue;
		}

		$subTag = explode(' ', $tag);
		$key = $subTag[0];
		$value = trim(substr($tag, strlen($key), strlen($tag) - strlen($key)));

		if ($key === $targetTagPrefix.'/MPReg:PersonDisplayName') {
			$tmpTagList[$tagIndex]['PersonDisplayName'] = $value;
			$nameList[$value] = $value;
		} else if ($key === $targetTagPrefix.'/MPReg:Rectangle') {
			$tmpTagList[$tagIndex]['Rectangle'] = $value;
		}
	}

	$metadataTagCount = 0;
	foreach($tmpTagList as $tag) {
		if (!$tag['Rectangle']) {
			continue;
		}

		$metadataTagCount++;

		if (!$tag['PersonDisplayName']) {
			continue;
		}

		$tagList[$tag['PersonDisplayName']] = array(
			'Rectangle' => $tag['Rectangle']
		);
	}

	return $tagList;
}

function renameDuplicateTagInDBList(&$dbList, $metadataList)
{
	foreach ($dbList as $name => &$tag) {
		//check for duplicate
		if (isset($metadataList[$name]) && isInToleranceRange($tag['infoFloat'], $metadataList[$name]['Rectangle'])) {
			$tag['isDuplicate'] = true;
			continue;
		}

		$suffix = 1;
		$newName = $name;

		//trying to rename
		while (true) {
			if (isset($metadataList[$newName])) {
				$newName = $name.'_'.($suffix++);
				continue;
			}

			$arrayKeys = array_keys($dbList, $newName);
			$arrayKeysCount = count($arrayKeys);

			if (!$arrayKeysCount || (1 == $arrayKeysCount && $key === $arrayKeys[0])) {
				//no same tag name in current dbNameList || only one same tag name and it is this tag name itself
				break;
			}

			$newName = $name.'_'.($suffix++);
		}

		$tag['newName'] = $newName;
	}
}

function writeTagListToMetadata($photo, $dbList, $metadataTagCount)
{
	$count = $metadataTagCount;

	//insert new tags from DB into metadata
	foreach ($dbList as $name => $tag) {
		if (!strlen($name) || $tag['isDuplicate']) {
			continue;
		}

		insertTagIntoMetadata($photo, ++$count, $tag['newName'], convertDBInfoToString($tag['infoFloat']));
	}
}

function insertTagIntoMetadata($photo, $index, $newName, $rectangle)
{
	global $user;

	$nameSpaceCmd = '-M"reg MP http://ns.microsoft.com/photo/1.2/" '.
		'-M"reg MPRI http://ns.microsoft.com/photo/1.2/t/RegionInfo#" '.
		'-M"reg MPReg http://ns.microsoft.com/photo/1.2/t/Region#"';

	$filePath = $photo['path'];
	if ('' !== $user) {
		$filePath = "/var/services/homes/$user/photo/".$filePath;
	}

	if (1 === $index) {
		$cmd = sprintf('%s %s %s %s', SYNO_PHOTO_EXIV2_PROG, $nameSpaceCmd,
					   '-M"set Xmp.MP.RegionInfo/MPRI:Regions XmpText type=Bag"',
					   escapeshellarg($filePath));
		@exec($cmd);
	}

	$cmd = sprintf('%s %s %s %s %s', SYNO_PHOTO_EXIV2_PROG, $nameSpaceCmd,
				   '-M"set Xmp.MP.RegionInfo/MPRI:Regions['.$index.']/MPReg:Rectangle '.$rectangle.'"',
				   '-M"set Xmp.MP.RegionInfo/MPRI:Regions['.$index.']/MPReg:PersonDisplayName '.str_replace('"', '\"', $newName).'"',
				   escapeshellarg($filePath));
	@exec($cmd);
}

function convertDBInfoToFloat($photox, $photoy, $info)
{
	$precision = 6;
	$thumbSize = 800;

	$ratio = floatval($photoy) / floatval($photox);

	if ($ratio > 1) {//higher image
		$x = round(floatval($info['x']) / ($thumbSize / $ratio), $precision);
		$y = round(floatval($info['y']) / $thumbSize, $precision);
		$width = round(floatval($info['width']) / ($thumbSize / $ratio), $precision);
		$height = round(floatval($info['height']) / $thumbSize, $precision);
	} else {//wider image
		$x = round(floatval($info['x']) / $thumbSize, $precision);
		$y = round(floatval($info['y']) / ($thumbSize * $ratio), $precision);
		$width = round(floatval($info['width']) / $thumbSize, $precision);
		$height = round(floatval($info['height']) / ($thumbSize * $ratio), $precision);
	}

	//prevent 0 for width & height
	$width = $width > 0 ? $width : 0.000001;
	$height = $height > 0 ? $height : 0.000001;

	return array(
		'x' => $x,
		'y' => $y,
		'width' => $width,
		'height' => $height
	);
}

function convertDBInfoToString($infoFloat)
{
	return "{$infoFloat['x']}, {$infoFloat['y']}, {$infoFloat['width']}, {$infoFloat['height']}";
}

function isInToleranceRange($infoFloat, $rectangle)
{
	$dbTolerance = 0.01;

	$recInfo = explode(',', $rectangle);
	foreach($recInfo as &$item) {
		$item = floatval(trim($item));
	}

	$arr = array(
		$dbTolerance,
		abs($infoFloat['x'] - $recInfo[0]),
		abs($infoFloat['y'] - $recInfo[1]),
		abs($infoFloat['width'] - $recInfo[2]),
		abs($infoFloat['height'] - $recInfo[3])
	);

	return $dbTolerance === max($arr);
}

?>
