<?php

/*
 +-----------------------------------------------------------------------+
 | program/include/rcube_user.inc                                        |
 |                                                                       |
 | This file is part of the Roundcube Webmail client                     |
 | Copyright (C) 2005-2010, Roundcube Dev. - Switzerland                 |
 | Licensed under the GNU GPL                                            |
 |                                                                       |
 | PURPOSE:                                                              |
 |   This class represents a system user linked and provides access      |
 |   to the related database records.                                    |
 |                                                                       |
 +-----------------------------------------------------------------------+
 | Author: Thomas Bruederli <roundcube@gmail.com>                        |
 +-----------------------------------------------------------------------+

 $Id: rcube_user.php,v 1.10 2011-04-11 08:23:47 chihcheng Exp $

*/


/**
 * Class representing a system user
 *
 * @package    Core
 * @author     Thomas Bruederli <roundcube@gmail.com>
 */
class rcube_user
{
    public $ID = null;
    public $data = null;
    public $language = null;

    private $db = null;

    /**
     * Object constructor
     *
     * @param object DB Database connection
     */
    function __construct($id = null, $sql_arr = null)
    {
        $this->db = rcmail::get_instance()->get_dbh();
    
        if ($id && !$sql_arr) {
            $sql_result = $this->db->query(
                "SELECT * FROM ".get_table_name('users')." U, users_type UT WHERE U.user_id = UT.user_id AND U.user_id = ?", $id);
            $sql_arr = $this->db->fetch_assoc($sql_result);
        }

        if (!empty($sql_arr)) {
            $this->ID       = $sql_arr['user_id'];
            $this->data     = $sql_arr;
            $this->language = $sql_arr['language'];
        }
    }


    /**
     * Build a user name string (as e-mail address)
     *
     * @param string Username part (empty or 'local' or 'domain')
     * @return string Full user name or its part
     */
    function get_username($part = null)
    {
        if ($this->data['username']) {
            list($local, $domain) = explode('@', $this->data['username']);

            // at least we should always have the local part
            if ($part == 'local') {
                return $local;
            }
            // if no domain was provided...
            if (empty($domain)) {
                $rcmail = rcmail::get_instance();
                $domain = $rcmail->config->mail_domain($this->data['mail_host']);
            }

            if ($part == 'domain') {
                return $domain;
            }

            if (!empty($domain))
                return $local . '@' . $domain;
            else
                return $local;
        }

        return false;
    }


    /**
     * Get the preferences saved for this user
     *
     * @return array Hash array with prefs
     */
    function get_prefs()
    {
        if (!empty($this->language))
            $prefs = array('language' => $this->language);

        if ($this->ID && $this->data['preferences'])
            $prefs += (array)unserialize($this->data['preferences']);

        return $prefs;
    }


    /**
     * Write the given user prefs to the user's record
     *
     * @param array User prefs to save
     * @return boolean True on success, False on failure
     */
    function save_prefs($a_user_prefs)
    {
        if (!$this->ID)
            return false;
      
        $config = rcmail::get_instance()->config;
        $old_prefs = (array)$this->get_prefs();

        // merge (partial) prefs array with existing settings
        $save_prefs = $a_user_prefs + $old_prefs;
        unset($save_prefs['language']);
    
        // don't save prefs with default values if they haven't been changed yet
        foreach ($a_user_prefs as $key => $value) {
            if (!isset($old_prefs[$key]) && ($value == $config->get($key)))
                unset($save_prefs[$key]);
        }

        $save_prefs = serialize($save_prefs);

        $this->db->query(
            "UPDATE ".get_table_name('users').
            " SET preferences = ?".
                ", language = ?".
            " WHERE user_id = ?",
            $save_prefs,
            $_SESSION['language'],
            $this->ID);

        $this->language = $_SESSION['language'];

        if ($this->db->affected_rows()) {
            $config->set_user_prefs($a_user_prefs);
            $this->data['preferences'] = $save_prefs;
            return true;
        }

        return false;
    }


    /**
     * Return a list of all users established in roundcube mail
     *
     * @return array List of users
     */
    function list_users()
    {
        $result = array();

        $sql_result = $this->db->query(
            "SELECT * FROM ".get_table_name('users'));
        while ($sql_arr = $this->db->fetch_assoc($sql_result)) {
            $result[] = $sql_arr;
        }
    
        return $result;
    }

    /**
     * Get smtp of this user
     *
     * @param int  smtp ID. If empty, the admin default is returned
     * @return array Hash array with all cols of the smtp record
     */
    function get_smtp($sid = null)
    {
        $result = $this->list_smtp($sid ? sprintf('AND smtp_id = %d', $sid) : 'AND ifdefault = 1');
        return $result[0];
    }


    /**
     * Return a list of all smtp linked with this user
     *
     * @return array List of smtp
     */
    function list_smtp($sql_add = '')
    {
        $result = array();
        if ($user_id)
          $this->ID = $user_id;
        $sql_result = $this->db->query(
            "SELECT * FROM ".get_table_name('smtp').
            " WHERE del <> 1 AND user_id = ?".
            ($sql_add ? " ".$sql_add : "").
            " ORDER BY smtp_id ASC",
            $this->ID);
    
        while ($sql_arr = $this->db->fetch_assoc($sql_result)) {
            $result[] = $sql_arr;
        }
    
        return $result;
    }



    /**
     * Update a specific smtp record
     *
     * @param int    smtp ID
     * @param array  Hash array with col->value pairs to save
     * @return boolean True if saved successfully, false if nothing changed
     */
    function update_smtp($sid, $data)
    {
        if (!$this->ID)
            return false;

        $query_cols = $query_params = array();
    
        foreach ((array)$data as $col => $value) {
            $query_cols[]   = $this->db->quoteIdentifier($col) . ' = ?';
            $query_params[] = $value;
        }
        $query_params[] = $sid;
        $query_params[] = $this->ID;

        $sql = "UPDATE ".get_table_name('smtp').
            " SET changed = ".$this->db->now().", ".join(', ', $query_cols).
            " WHERE smtp_id = ?".
                " AND user_id = ?".
                " AND del <> 1";

        call_user_func_array(array($this->db, 'query'),
            array_merge(array($sql), $query_params));
    
        return $this->db->affected_rows();
    }
  
  
    /**
     * Create a new smtp record linked with this user
     *
     * @param array  Hash array with col->value pairs to save
     * @return int  The inserted identity ID or false on error
     */
  
    function insert_smtp($data)
    {
        if (!$this->ID)
            return false;

        unset($data['user_id']);

        $insert_cols = $insert_values = array();
        foreach ((array)$data as $col => $value) {
            $insert_cols[]   = $this->db->quoteIdentifier($col);
            $insert_values[] = $value;
        }
        $insert_cols[]   = 'user_id';
        $insert_values[] = $this->ID;

        $sql = "INSERT INTO ".get_table_name('smtp').
            " (changed, ".join(', ', $insert_cols).")".
            " VALUES (".$this->db->now().", ".join(', ', array_pad(array(), sizeof($insert_values), '?')).")";

        call_user_func_array(array($this->db, 'query'),
            array_merge(array($sql), $insert_values));

        return $this->db->insert_id('smtp_ids');
    }
  
    /**
     * Mark the given smtp as deleted
     *
     * @param int smtp ID
     * @return boolean True if deleted successfully, false if nothing changed
     */
  
    function delete_smtp($sid)
    {
        if (!$this->ID)
            return false;

        $sql_result = $this->db->query(
            "SELECT count(*) AS smtp_count FROM ".get_table_name('smtp').
            " WHERE user_id = ? AND del <> 1",
            $this->ID);

        $sql_arr = $this->db->fetch_assoc($sql_result);

        // we'll not delete last smtp
        if ($sql_arr['smtp_count'] <= 1)
            return false;
    
        $this->db->query(
            "UPDATE ".get_table_name('smtp').
            " SET del = 1, changed = ".$this->db->now().
            " WHERE user_id = ?".
                " AND smtp_id = ?",
            $this->ID,
            $sid);

        return $this->db->affected_rows();
    }
  
    /**
     * Make this smtp the default one for this user
     *
     * @param int The smtp ID
     */
    function set_default_smtp($sid)
    {
        if ($this->ID && $sid) {
            $this->db->query(
                "UPDATE ".get_table_name('smtp').
                " SET ".$this->db->quoteIdentifier('ifdefault')." = '0'".
                " WHERE user_id = ?".
                    " AND smtp_id <> ?".
                    " AND del <> 1",
                $this->ID,
                $sid);
        }
    }


    function filter_popinfo($data)
    {
       $popinfo_arr = array('ext', 'extusername', 'extpd', 'popserver', 'popport',
                        'ifssl', 'select_folder', 'remove_mail', 'select_smtp');
       $popinf = array();
       if (!is_array($data)) {
            raise_error(array(
                'code' => 500,
                'type' => 'php',
                'line' => __LINE__,
                'file' => __FILE__,
                'message' => "Failed to create new user"), true, false);
            return;
       }
       if (!isset($data['ext'])) { 
         return array($data, $popinfo);
       }   
       foreach($data as $key => $value) {
	      if (in_array($key,$popinfo_arr)) {
            $popinfo[$key] = $data[$key];
            unset($data[$key]);
          }
       }
       return array($data, $popinfo);
    }


    function insert_popinfo($popinfo)
    {
        if (!isset($popinfo['ext'])) {
           return;
        }

        $insert_cols = $insert_values = array();
        foreach ((array)$popinfo as $col => $value) {
            $insert_cols[]   = $this->db->quoteIdentifier($col);
            $insert_values[] = $value;
        }

        $sql = "INSERT INTO ".get_table_name('popinfo').
            " (changed, ".join(', ', $insert_cols).")".
            " VALUES (".$this->db->now().", ".join(', ', array_pad(array(), sizeof($insert_values), '?')).")";

        call_user_func_array(array($this->db, 'query'),
            array_merge(array($sql), $insert_values));
    	return;       
    }

    function list_popinfo($result)
    {

        if (!$result) {
            return $result; 
        }
        $result_arr = array();
        foreach($result as $elem) {
            $sql_result = $this->db->query(
                "SELECT * FROM ".get_table_name('popinfo').
                " WHERE identity_id = ?" ,$elem['identity_id']);
            $sql_arr = $this->db->fetch_assoc($sql_result);
            $result_arr[] = is_array($sql_arr)?array_merge($elem, $sql_arr):$elem;
         }

        return $result_arr;
    }

    function update_popinfo($popinfo,$iid)
    {
        if (!isset($popinfo['ext'])) {
           return;
        }
        $query_cols = $query_params = array();
    
        foreach ((array)$popinfo as $col => $value) {
            $query_cols[]   = $this->db->quoteIdentifier($col) . ' = ?';
            $query_params[] = $value;
        }
        $query_params[] = $iid;

        $sql = "UPDATE ".get_table_name('popinfo').
            " SET changed = ".$this->db->now().", ".join(', ', $query_cols).
            " WHERE identity_id = ?";

        call_user_func_array(array($this->db, 'query'),
            array_merge(array($sql), $query_params));

        return;
    }

    /**
     * Get identity of this user by email
     *
     * @param char* E-mail. 
     * @return array Hash array with all cols of the identity record
     */
    function get_identity_by_email($email)
    {
        $result = $this->list_identities($email ? sprintf("AND email = '%s'", $email) : '');
        return $result[0];
    }

    /**
     * Get default identity of this user
     *
     * @param int  Identity ID. If empty, the default identity is returned
     * @return array Hash array with all cols of the identity record
     */
    function get_identity($id = null)
    {
        $result = $this->list_identities($id ? sprintf('AND identity_id = %d', $id) : '');
        return $result[0];
    }


    /**
     * Return a list of all identities linked with this user
     *
     * @return array List of identities
     */
    function list_identities($sql_add = '')
    {
        $result = array();

        $sql_result = $this->db->query(
            "SELECT * FROM ".get_table_name('identities').
            " WHERE del <> 1 AND user_id = ?".
            ($sql_add ? " ".$sql_add : "").
            " ORDER BY ".$this->db->quoteIdentifier('standard')." DESC, name ASC, identity_id ASC",
            $this->ID);
        while ($sql_arr = $this->db->fetch_assoc($sql_result)) {
            $result[] = $sql_arr;
        }
    
		$result = $this->list_popinfo($result);
        return $result;
    }


    /**
     * Update a specific identity record
     *
     * @param int    Identity ID
     * @param array  Hash array with col->value pairs to save
     * @return boolean True if saved successfully, false if nothing changed
     */
    function update_identity($iid, $data)
    {
		list($data,$popinfo) = $this->filter_popinfo($data);
        $this->update_popinfo($popinfo,$iid);
        if (!$this->ID)
            return false;

        $query_cols = $query_params = array();
    
        foreach ((array)$data as $col => $value) {
            $query_cols[]   = $this->db->quoteIdentifier($col) . ' = ?';
            $query_params[] = $value;
        }
        $query_params[] = $iid;
        $query_params[] = $this->ID;

        $sql = "UPDATE ".get_table_name('identities').
            " SET changed = ".$this->db->now().", ".join(', ', $query_cols).
            " WHERE identity_id = ?".
                " AND user_id = ?".
                " AND del <> 1";

        call_user_func_array(array($this->db, 'query'),
            array_merge(array($sql), $query_params));
    
        return $this->db->affected_rows();
    }
  
  
    /**
     * Create a new identity record linked with this user
     *
     * @param array  Hash array with col->value pairs to save
     * @return int  The inserted identity ID or false on error
     */
    function insert_identity($data)
    {
		list($data,$popinfo) = $this->filter_popinfo($data);

        if (!$this->ID)
            return false;

        unset($data['user_id']);

        $insert_cols = $insert_values = array();
        foreach ((array)$data as $col => $value) {
            $insert_cols[]   = $this->db->quoteIdentifier($col);
            $insert_values[] = $value;
        }
        $insert_cols[]   = 'user_id';
        $insert_values[] = $this->ID;

        $sql = "INSERT INTO ".get_table_name('identities').
            " (changed, ".join(', ', $insert_cols).")".
            " VALUES (".$this->db->now().", ".join(', ', array_pad(array(), sizeof($insert_values), '?')).")";

        call_user_func_array(array($this->db, 'query'),
            array_merge(array($sql), $insert_values));

		$iid = $this->db->insert_id('identities');
        $popinfo['identity_id'] = $iid;
        $this->insert_popinfo($popinfo);

        return $iid;
    }
  
  
    /**
     * Mark the given identity as deleted
     *
     * @param int  Identity ID
     * @return boolean True if deleted successfully, false if nothing changed
     */
    function delete_identity($iid)
    {
        if (!$this->ID)
            return false;

        $sql_result = $this->db->query(
            "SELECT count(*) AS ident_count FROM ".get_table_name('identities').
            " WHERE user_id = ? AND del <> 1",
            $this->ID);

        $sql_arr = $this->db->fetch_assoc($sql_result);

        // we'll not delete last identity
        if ($sql_arr['ident_count'] <= 1)
            return false;
    
        $this->db->query(
            "UPDATE ".get_table_name('identities').
            " SET del = 1, changed = ".$this->db->now().
            " WHERE user_id = ?".
                " AND identity_id = ?",
            $this->ID,
            $iid);

        return $this->db->affected_rows();
    }
  
  
    /**
     * Make this identity the default one for this user
     *
     * @param int The identity ID
     */
    function set_default($iid)
    {
        if ($this->ID && $iid) {
            $this->db->query(
                "UPDATE ".get_table_name('identities').
                " SET ".$this->db->quoteIdentifier('standard')." = '0'".
                " WHERE user_id = ?".
                    " AND identity_id <> ?".
                    " AND del <> 1",
                $this->ID,
                $iid);
        }
    }
   private function _touch_identity()
   {
     $user_email = sprintf('%s@%s', $this->data['username'],
                           rcmail::get_instance()->config->mail_domain(null));
     $iid = -1;
     $identities = $this->list_identities();
     foreach($identities as $ident)
     {
       if ($ident['email'] == $user_email)
       {
         $iid = $ident['identity_id'];
         break;
       }
     }
     if ($iid < 0)
     {
       $ident = array('name'=>$this->data['username'], 'email'=>$user_email, 'standard'=>1);
       $iid = $this->insert_identity($ident);
       $this->set_default($iid);
     }
   }
  
  
    /**
     * Update user's last_login timestamp
     */
    function touch()
    {
        if ($this->ID) {
            $this->db->query(
                "UPDATE ".get_table_name('users').
                " SET last_login = ".$this->db->now().
                " WHERE user_id = ?",
                $this->ID);
                $this->_touch_identity();
        }
    }
  
  
    /**
     * Clear the saved object state
     */
    function reset()
    {
        $this->ID = null;
        $this->data = null;
    }
	
    /**
     * Write contents to files
     */
		
    function writefile($filepath,$content)
    {
        $fp = fopen($filepath, 'w');
        fwrite($fp, $content);
        fclose($fp);
        return true;
    }

	function getRealUserName($username)
	{
		$cmd = RCMAIL_FETCH_EXEC . ' ' . escapeshellarg($username);
		@exec($cmd.' 4 ' , $uid, $retval);

		if (0 == $retval && isset($uid[0])) {
			$info_arr = posix_getpwuid($uid[0]);
		} else {
			return $username;
		}

		if (isset($info_arr['name'])) {
			return $info_arr['name'];
		} else {
			return $username;
		}
	}
	function getUserDirectory($username)
	{
		$cmd = RCMAIL_FETCH_EXEC . ' ' . escapeshellarg($username);
		@exec($cmd.' 5 ' , $Dir, $retval);

		if (0 == $retval && isset($Dir[0])) {
			return $Dir[0];
		} else {
			return $username;
		}
	}
	static function getMailServerConf()
	{
		$cmd = RCMAIL_FETCH_EXEC . ' ' . escapeshellarg('nobody');
		@exec($cmd.' 6 ' , $Conf, $retval);
		$result = array();

		if (0 == $retval && isset($Conf[0])) {
			$result['account_type'] = $Conf[0];
			$result['domain_name'] = $Conf[1];
			return $result;
		} else {
			$result['account_type'] = 'local';
			$result['domain_name'] = '';
			return $result;
		}
	}

    /**
     * 
     *
     * @param string IMAP user name
     * @return object rcube_user New user instance
     */
    function write_fetchmailrc($data,$_iid=null)
    {
        $err = false;
        $rcmail = rcmail::get_instance();
	$username = $this->get_full_username();
	$DsmUserName = $this->getRealUserName($username);
        $DsmUserDirectory = $this->getUserDirectory($username);
        $fetchfile = RCMAIL_EXT_DIR . '/' . $username . '_fetch';
        $fetchtemp = RCMAIL_EXT_DIR . '/fetchmailrc';
        $iid = ($_iid)?$_iid:get_input_value('_iid', RCUBE_INPUT_POST);
        $old_data = $this->get_identity($iid);
  
        $content = '';
        if (file_exists($fetchfile)) {
          // "update"
          if (!$content = file_get_contents($fetchfile)) {
            return $err;
          }
          
          $pollblock = '/#### "'.$old_data['email'].'".*"'.$old_data['email'].'"\s/s';
            
          if (!$content = preg_replace($pollblock,'',$content)) {
            return $err;
          }
        }
        else {
          // "Add"
          if (!($content = file_get_contents($fetchtemp))) {
            return $err;
          }
          if (!$content = preg_replace("/_currentuserdirectory/",$DsmUserDirectory,$content)) {
            return $err;
          }
        }
        // fixed fetch period
        $period = $rcmail->config->get('extmailperiod',5);
        settype($period,"integer");
		$period = $period * 60 ; // min to sec.
        if (!$content = preg_replace("/set daemon \d+/", "set daemon ".$period, $content)) {
          return $err;
        }

        if($data) {
          $userdomain = preg_replace('/@/','.',$data['email']);
          $procfile = RCMAIL_EXT_DIR . '/' . $username . '.proc.' . $userdomain;
          $content .= '#### "'.$data['email']."\"\n";                                                      
	      $content .= 'poll "'. $data['popserver']. '" with protocol POP3 ';
          if ($data['remove_mail']!=1)
            $content .= 'uidl ';
          $content .= 'and port '. $data['popport'].":\n";
	      $content .= "\tuser \"".$data['extusername']."\" pass \"".$data['extpd']."\" is \"".$DsmUserName."\" here\n";
	      if ($data['ifssl']==1) {
	        $content .=  "\toptions ssl ";
	      }
		  $content .=	"\n";
          if ($data['remove_mail']==1)
            $content .=  "\tno keep\n";
          else
            $content .=  "\tkeep\n";
          $content .= "mda \"" .RCMAIL_PROC_EXEC ." -m \\'".$procfile."\\'\"\n";
          $content .= "#### \"".$data['email']."\"\n";
        }

        return $this->writefile($fetchfile,$content);
    }
    function proc_data_limit($days)
    {
      $mon = array("Jan","Feb","Mar","Apr","May","Jue","Jul","Aug","Sep","Oct","Nov","Dec");
      $content = '';
      if ($days < 0)
          return $content;
      $time_lower_bound = array();
      $time_lower_bound = getdate(time() - ($days * 24 * 60 * 60));


      $content = "\n:0\n* ^Date.*(1970"; 
      for ($y = 1971 ; $y < $time_lower_bound['year'] ; $y++)
         $content .= '|' . $y;
      $content .= ")\n/dev/null\n";

      $m = 0;
      if ($time_lower_bound['mon'] > 1) {
          $content .= "\n:0\n* ^Date.*(Jan"; 
          for ($m = 1 ; $m < $time_lower_bound['mon'] - 1 ; $m++)
             $content .= '|' . $mon[$m];
          $content .= ") ".$y."\n/dev/null\n";
      }

      $content .= "\n:0\n* ^Date.*(01"; 
      for ($d = 2 ; $d < $time_lower_bound['mday'] ; $d++)
         $content .= ($d < 10 )? ('|0'.$d) : ('|'.$d);
      $content .= ") ".$mon[$m]." ".$y."\n/dev/null\n";

      return $content;

    }
    function write_procmailrc($data,$days)
    {
        $err = false;
	$username = $this->get_full_username();
        $DsmUserDirectory = $this->getUserDirectory($username);
        $userdomain = preg_replace('/@/','.',$data['email']);
        $procfile = RCMAIL_EXT_DIR . '/' . $username . '.proc.' . $userdomain;
        $proctemp = RCMAIL_EXT_DIR . '/procmailrc';

        $content = '';
        if (!($content .= file_get_contents($proctemp))) {
           return $err;
        }
        if (!$content = preg_replace("/_currentuserdirectory/",$DsmUserDirectory,$content)) {
          return $err;
        }

        $mailpath = '';
        $mailpath = ($data['select_folder'] == 'INBOX')? '.Maildir/':'.Maildir/.'.preg_replace('/ /','\ ',$data['select_folder']).'/';
        $content .= $this->proc_data_limit($days);                                                      
        $content .= "\n:0\n".$mailpath."\n";                                                     
        return $this->writefile($procfile,$content);
        

    }

    public function writerc($data, $days = null , $_iid = null)
    {
        $rcmail = rcmail::get_instance();
        $err = true;
	$username = $this->get_full_username();
        $iid = ($_iid)?$_iid:get_input_value('_iid', RCUBE_INPUT_POST);
        $old_data = $this->get_identity($iid);
        $old_userdomain = preg_replace('/@/','.',$old_data['email']);
        $cmd = RCMAIL_FETCH_EXEC . ' ' . $username;

        //pre-modify
		system ( $cmd.' 1 '.escapeshellarg($old_userdomain));

        //modify
        if (!$this->write_fetchmailrc($data,$_iid)){
          $err = false;
        }	
        if ($data)
          $err = $this->write_procmailrc($data,$days); 
        

        // post-modify
        $userdomain = $data?preg_replace('/@/','.',$data['email']):$old_userdomain;
        system ($cmd.' 2 '.escapeshellarg($userdomain));


        // delete: modify idfile and restart daemon
        if (!$data)
          system ($cmd.' -3 '.escapeshellarg($old_data['extusername'].'@'.$old_data['popserver']));
        // add/modify: awakening daemon
        else
          system($cmd.' -4');

        return $err;
    }
    function get_full_username()
    {
	$username = $_SESSION["username"];
	if('ldap' == $_SESSION["account_type"] ) {
		$username = $username . "@" . $_SESSION["domain_name"];
	} else if ('win' == $_SESSION["account_type"] ) {
		$username = $_SESSION["domain_name"] . "\\" . $username;
	}

	return $username;
    }
    /**
     * Find a user is admin group member
     *
     * @param 
     * @return 
     *          1: @a szUser is a member of the admin group
     *          0: @a szUser is not a member of the admin group
     *         -1: error
     *         
     */

    public function Is_AdminGroup()
    {
		$blAdmin = false;
		$username = $this->get_full_username();
	
		$cmd = RCMAIL_FETCH_EXEC . ' ' . $username . ' 0';

        $buffer = system($cmd,$blAdmin);
		return $blAdmin;
    }

    /**
     * Connect to pop server and do a Certificate Authority (CA) test
     *
     * @param string pop server hostname
     * @param string pop server port
     * @return true  when CA are not recognized form serve;
     *         false when CA success
     */
    function ca_failure_test($popserver,$port)
    {
		$connecto = $popserver.':'.$port;
        $descriptorspec = array(                                                                                                                 
           0 => array("pipe", "r"),  // stdin is a pipe that the child will read from                                                            
           1 => array("pipe", "w"),  // stdout is a pipe that the child will write to                                                            
           2 => array("file", "/var/packages/MailStation/target/roundcubemail/logs/error", "a") // stderr is a file to write to                  
        );                                                                                                                                       
        $cmd = '/usr/syno/bin/openssl s_client -verify -purpose -connect ';
        $cmd .= $connecto . ' -CApath ' . RCMAIL_EXT_DIR . '/.cert';  
        $cwd = NULL;                                                                                                                             
        $env = NULL;                                                                                                                             
                                                                                                                                         
        $process = @proc_open($cmd, $descriptorspec, $pipes, $cwd, $env);                                                                         
                                                                                                                                        
        if (is_resource($process)) {
          @fwrite($pipes[0], 'quit');

          @fclose($pipes[0]);

          $result = stream_get_contents($pipes[1]);

          @fclose($pipes[1]);

          @fclose($pipes[2]);

        }
        @proc_close($process);
        if ( 0 < preg_match("/Verify return code: ([\d]+) \(/",$result,$matches) && 0 == intval($matches[1])) {
          return false;
        }
        return true;
     }

    /**
     * Connect to pop server test
     *
     * @param string pop server hostname
     * @param string pop server port
     * @return false when connect failure form server
     */
    function popserver_connect_test($popserver,$port)
    {
      $errno = 0;
      $fp = @fsockopen($popserver, $port, $errno, $errstr, 10);
      if (!$fp) {
          $errno = ($errno==0)?500:$errno;
      }
      fclose($fp);
      return $errno;
    }

    /**
     * Find a user record matching the given name and host
     *
     * @param string IMAP user name
     * @param string IMAP host name
     * @return object rcube_user New user instance
     */
    static function query($user, $host)
    {
        $dbh = rcmail::get_instance()->get_dbh();
		$MailServerConf = (array)rcube_user::getMailServerConf();
		if(!isset($MailServerConf)){
			return false;
		}
    
        // query for matching user name
        $query = "SELECT * FROM ".get_table_name('users').
			" U, users_type UT WHERE U.user_id = UT.user_id AND mail_host = ? AND %s = ? AND account_type = ? AND domain_name = ?";
		$query .= " ORDER BY U.user_id ASC";
        $sql_result = $dbh->query(sprintf($query, 'username'), $host, $user
				,$MailServerConf['account_type'], $MailServerConf['domain_name']);
    
        // query for matching alias
        if (!($sql_arr = $dbh->fetch_assoc($sql_result))) {
            $sql_result = $dbh->query(sprintf($query, 'alias'), $host, $user
					,$MailServerConf['account_type'], $MailServerConf['domain_name']);
            $sql_arr = $dbh->fetch_assoc($sql_result);
        }
    
        // user already registered -> overwrite username
        if ($sql_arr)
            return new rcube_user($sql_arr['user_id'], $sql_arr);
        else
            return false;
    }
  
  
    /**
     * Create a new user record and return a rcube_user instance
     *
     * @param string IMAP user name
     * @param string IMAP host
     * @return object rcube_user New user instance
     */
    static function create($user, $host)
    {
        $user_name  = '';
        $user_email = '';
        $rcmail = rcmail::get_instance();
        $user_lc = mb_strtolower($user);

        // try to resolve user in virtuser table and file
        if ($email_list = self::user2email($user, false, true)) {
            $user_email = is_array($email_list[0]) ? $email_list[0]['email'] : $email_list[0];
        }

        $data = $rcmail->plugins->exec_hook('user_create',
	        array('user'=>$user, 'user_name'=>$user_name, 'user_email'=>$user_email));

        // plugin aborted this operation
        if ($data['abort'])
            return false;

        $user_name  = $data['user_name'];
        $user_email = $data['user_email'];

        $dbh = $rcmail->get_dbh();

		$MailServerConf = (array)rcube_user::getMailServerConf();
		if(!isset($MailServerConf)){
            raise_error(array(
                'code' => 500,
                'type' => 'php',
                'line' => __LINE__,
                'file' => __FILE__,
                'message' => "Failed to create new user, can't read MailServerConf"), true, false);
			return false;
		}

		//created user on users and users_type table
		$dbh->query(
				"INSERT INTO ".get_table_name('users').
				" (created, last_login, username, mail_host, alias, language)".
				" VALUES (".$dbh->now().", ".$dbh->now().", ?, ?, ?, ?)",
				strip_newlines($user_lc),
				strip_newlines($host),
				strip_newlines($data['alias'] ? $data['alias'] : $user_email),
				strip_newlines($data['language'] ? $data['language'] : $_SESSION['language']));
		$dbh->query(
				"INSERT INTO users_type".
				" (user_id, account_type, domain_name)".
				" VALUES (?, ?, ?)",
				strip_newlines($dbh->insert_id('users')),
				strip_newlines($MailServerConf['account_type']),
				strip_newlines($MailServerConf['domain_name']));

        if ($user_id = $dbh->insert_id('users')) {
            // create rcube_user instance to make plugin hooks work
            $user_instance = new rcube_user($user_id);
            $rcmail->user  = $user_instance;

            $mail_domain = $rcmail->config->mail_domain($host);

            if ($user_email == '') {
                $user_email = strpos($user, '@') ? $user : sprintf('%s@%s', $user, $mail_domain);
            }
            if ($user_name == '') {
                $user_name = $user != $user_email ? $user : '';
            }

            if (empty($email_list))
                $email_list[] = strip_newlines($user_email);
            // identities_level check
            else if (count($email_list) > 1 && $rcmail->config->get('identities_level', 0) > 1)
                $email_list = array($email_list[0]);

            // create new identities records
            $standard = 1;
            foreach ($email_list as $row) {
	            $record = array();

                if (is_array($row)) {
	                $record = $row;
                }
                else {
                    $record['email'] = $row;
                }

	            if (empty($record['name']))
	                $record['name'] = $user_name;
                $record['name'] = strip_newlines($record['name']);
                $record['user_id'] = $user_id;
                $record['standard'] = $standard;

                $plugin = $rcmail->plugins->exec_hook('identity_create',
	                array('login' => true, 'record' => $record));
          
                if (!$plugin['abort'] && $plugin['record']['email']) {
                    $rcmail->user->insert_identity($plugin['record']);
                }
                $standard = 0;
            }
            // create new smtp records
	        $record = array();
            $record['user_id'] = $user_id;
            $record['smtpdesc'] = $rcmail->config->get('smtp_server', 'localhost');
            $record['smtpserver'] = $rcmail->config->get('smtp_server', 'localhost');
            $record['smtpport'] = intval($rcmail->config->get('smtp_port', 25));
            $record['smtpuser'] = $rcmail->config->get('smtp_user');
            $record['smtppass'] = $rcmail->config->get('smtp_pass');
            $record['ifdefault'] = '1';
            $rcmail->user->insert_smtp($record);
        }
        else {
            raise_error(array(
                'code' => 500,
                'type' => 'php',
                'line' => __LINE__,
                'file' => __FILE__,
                'message' => "Failed to create new user"), true, false);
        }
    
        return $user_id ? $user_instance : false;
    }
  
  
    /**
     * Create a new smtp form user which lose smtp data.
     *
     * @param string user id
     * @return smtp id or false
     */
    static function create_default_smtp($user_id)
    {
        $rcmail = rcmail::get_instance();

        // create new smtp records
	    $record = array();
        $record['user_id'] = $user_id;
        $record['smtpdesc'] = $rcmail->config->get('smtp_server', 'localhost');
        $record['smtpserver'] = $rcmail->config->get('smtp_server', 'localhost');
        $record['smtpport'] = intval($rcmail->config->get('smtp_port', 25));
        $record['smtpuser'] = $rcmail->config->get('smtp_user');
        $record['smtppass'] = $rcmail->config->get('smtp_pass');
        $record['ifdefault'] = '1';
        $rcmail->user->ID = $user_id;
        return $rcmail->user->insert_smtp($record);

    }


    /**
     * Resolve username using a virtuser plugins
     *
     * @param string E-mail address to resolve
     * @return string Resolved IMAP username
     */
    static function email2user($email)
    {
        $rcmail = rcmail::get_instance();
        $plugin = $rcmail->plugins->exec_hook('email2user',
            array('email' => $email, 'user' => NULL));

        return $plugin['user'];
    }


    /**
     * Resolve e-mail address from virtuser plugins
     *
     * @param string User name
     * @param boolean If true returns first found entry
     * @param boolean If true returns email as array (email and name for identity)
     * @return mixed Resolved e-mail address string or array of strings
     */
    static function user2email($user, $first=true, $extended=false)
    {
        $rcmail = rcmail::get_instance();
        $plugin = $rcmail->plugins->exec_hook('user2email',
            array('email' => NULL, 'user' => $user,
                'first' => $first, 'extended' => $extended));

        return empty($plugin['email']) ? NULL : $plugin['email'];
    }

}
