# Local status cache repository support
# Copyright (C) 2002-2017 John Goerzen & contributors
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import os

from offlineimap.folder.LocalStatus import LocalStatusFolder
from offlineimap.folder.LocalStatusSQLite import LocalStatusSQLiteFolder
from offlineimap.folder.LocalStatusSynoSQLite import LocalStatusSynoSQLiteFolder, LocalStatusSynoGsuiteSQLiteFolder, remove_file_dir
from offlineimap.repository.Base import BaseRepository
from offlineimap.error import OfflineImapError


class LocalStatusRepository(BaseRepository):
    def __init__(self, reposname, account):
        BaseRepository.__init__(self, reposname, account)

        # class and root for all backends.
        self.backends = {}
        self.backends['sqlite'] = {
            'class': LocalStatusSQLiteFolder,
            'root': os.path.join(account.getaccountmeta(), 'LocalStatus-sqlite')
        }
        self.backends['plain'] = {
            'class': LocalStatusFolder,
            'root': os.path.join(account.getaccountmeta(), 'LocalStatus')
        }

        #add by Synology. Add a status backend. It is mostly identical to LocalStatus-sqlite.
        #It touch a file when processing a folder, and delete the file after the process finishes.
        self.backends['syno_sqlite'] = {
            'class': LocalStatusSynoSQLiteFolder,
            'root': os.path.join(account.getaccountmeta(), 'LocalStatus-Syno-sqlite')
        }
        self.backends['syno_gsuite_sqlite'] = {
            'class': LocalStatusSynoGsuiteSQLiteFolder,
            'root': os.path.join(account.getaccountmeta(), 'LocalStatus-SynoGsuite-sqlite')
        }

        # Set class and root for the configured backend
        status_backend = self.account.getconf('status_backend', 'sqlite')
        self.setup_backend(status_backend)
        #if self.account.getconf('status_backend', None) is not None:
        #    raise OfflineImapError(
        #        "the 'status_backend' configuration option is not supported"
        #        " anymore; please, remove this configuration option.",
        #        OfflineImapError.ERROR.REPO
        #    )
        ## Set class and root for sqlite.
        #self.setup_backend('sqlite')

        if not os.path.exists(self.root):
            os.mkdir(self.root, 0o700)

        # self._folders is a dict of name:LocalStatusFolders().
        self._folders = {}

        self.used_status_backend = status_backend
        if self.used_status_backend in ['syno_sqlite', 'syno_gsuite_sqlite']:
            self.processing_root = os.path.join(account.getaccountmeta(), 'Curr-Processing')
            self.indicator_path = os.path.join(self.processing_root, '1pR4ii08rJPUKdK')
            try:
                remove_file_dir(self.processing_root)
                os.mkdir(self.processing_root, 0700)
            except OSError as e:
                raise OfflineImapError('Errors happened when trying to initialize folder, [{0}]. {1}'.format(self.processing_root, e) , OfflineImapError.ERROR.REPO)

    def create_procressing_user_indicator(self):
        #add by Synology. Add a file to indicate it is currently processing this user
        if self.used_status_backend == 'syno_sqlite':
            try:
                with open(self.indicator_path, 'a'):
                    #touch the file
                    pass
            except OSError as e:
                raise OfflineImapError('Errors happened when trying to creat processing indicator at [{0}]. {1}'.format(self.indicator_path, e) , OfflineImapError.ERROR.REPO)

    def clear_procressing_user_indicator(self):
        #add by Synology. Clear the file indicates it is currently processing this user
        if self.used_status_backend == 'syno_sqlite':
            try:
                if os.path.exists(self.indicator_path):
                    os.remove(self.indicator_path)
            except OSError as e:
                raise OfflineImapError('Errors happened when trying to clear processing indicator at [{0}]. {1}'.format(self.indicator_path, e) , OfflineImapError.ERROR.REPO)


    def _instanciatefolder(self, foldername):
        return self.LocalStatusFolderClass(foldername, self) # Instanciate.

    def setup_backend(self, backend):
        if backend in self.backends.keys():
            self._backend = backend
            self.root = self.backends[backend]['root']
            self.LocalStatusFolderClass = self.backends[backend]['class']

    def import_other_backend(self, folder):
        for bk, dic in self.backends.items():
            # Skip folder's own type.
            if dic['class'] == type(folder):
                continue

            repobk = LocalStatusRepository(self.name, self.account)
            repobk.setup_backend(bk)      # Fake the backend.
            folderbk = dic['class'](folder.name, repobk)

            # If backend contains data, import it to folder.
            if not folderbk.isnewfolder():
                self.ui._msg("Migrating LocalStatus cache from %s to %s "
                    "status folder for %s:%s"%
                    (bk, self._backend, self.name, folder.name))

                folderbk.cachemessagelist()
                folder.messagelist = folderbk.messagelist
                folder.saveall()
                break

    def getsep(self):
        return '.'

    def makefolder(self, foldername):
        """Create a LocalStatus Folder."""

        if self.account.dryrun:
            return # Bail out in dry-run mode.

        # Create an empty StatusFolder.
        folder = self._instanciatefolder(foldername)
        # First delete any existing data to make sure we won't consider obsolete
        # data. This might happen if the user removed the folder (maildir) and
        # it is re-created afterwards.
        folder.purge()
        folder.openfiles()
        folder.save()
        folder.closefiles()

        # Invalidate the cache.
        self.forgetfolders()

    def getfolder(self, foldername):
        """Return the Folder() object for a foldername.

        Caller must call closefiles() on the folder when done."""

        if foldername in self._folders:
            return self._folders[foldername]

        folder = self._instanciatefolder(foldername)

        # If folder is empty, try to import data from an other backend.
        if folder.isnewfolder():
            self.import_other_backend(folder)

        self._folders[foldername] = folder
        return folder

    def getfolders(self):
        """Returns a list of all cached folders.

        Does nothing for this backend. We mangle the folder file names
        (see getfolderfilename) so we can not derive folder names from
        the file names that we have available. TODO: need to store a
        list of folder names somehow?"""

        pass

    def forgetfolders(self):
        """Forgets the cached list of folders, if any.  Useful to run
        after a sync run."""

        self._folders = {}
