# Maildir repository support
# Copyright (C) 2002-2015 John Goerzen & contributors
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

from offlineimap import folder
from offlineimap.ui import getglobalui
from offlineimap.error import OfflineImapError
from offlineimap.repository.Base import BaseRepository
import time
import os
from stat import *

import glob

import sys
for egg in glob.glob(r'/var/packages/MailPlus-Server/target/usr/bin/gsuite/*.egg'):
    sys.path.append(egg)

from googleapiclient.discovery import build
from oauth2client.service_account import ServiceAccountCredentials
import googleapiclient.errors as errors

class GsuiteApi():
    def __init__(self, json_keyfile, target_user):
        SCOPES = ['https://mail.google.com']
        self._credentials = ServiceAccountCredentials.from_json_keyfile_name(json_keyfile, SCOPES)
        self.target_user = target_user
        #https://oauth2client.readthedocs.io/en/latest/source/oauth2client.service_account.html
        self._delegated_credentials = self._credentials.create_delegated(target_user)
        self._service = build('gmail', 'v1', credentials=self._delegated_credentials)

    def listlabels(self):
        # https://developers.google.com/gmail/api/v1/reference/users/labels/list
        return self._service.users().labels().list(userId='me').execute()

    def getlabels(self, labelId):
        # https://developers.google.com/gmail/api/v1/reference/users/labels/list
        return self._service.users().labels().get(userId='me', id=labelId).execute()

    def listmessages(self, labelId=None, query=None, pageToken=None):
        # https://developers.google.com/gmail/api/v1/reference/users/messages/list
        labelIds = [labelId] if labelId else None
        return self._service.users().messages().list(userId='me', labelIds=labelIds, q=query, pageToken=pageToken, maxResults=1000).execute()

    def getmessage(self, messageId):
        # https://developers.google.com/gmail/api/v1/reference/users/messages/get
        return self._service.users().messages().get(userId='me', id=messageId, format='raw').execute()

    def getservice(self):
        return self._service

SYNO_SPECIAL_FOLDERS = {
    'draft': {
        'id': 'DRAFT',
        'result_folder_name': 'Drafts'
    },
    'junk': {
        'id': 'SPAM',
        'result_folder_name': 'Junk'
    },
    'trash': {
        'id': 'TRASH',
        'result_folder_name': 'Trash'
    },
    'sent': {
        'id': 'SENT',
        'result_folder_name': 'Sent'
    }
}

class GsuiteRepository(BaseRepository):
    def __init__(self, reposname, account):
        """Initialize a MaildirRepository object.  Takes a path name
        to the directory holding all the Maildir directories."""

        BaseRepository.__init__(self, reposname, account)

        self.folders = None
        self.ui = getglobalui()
        json_keyfile = self.getconf('json_keyfile', None)
        self.skip_archive = self.getconfboolean('skip_archived_mail', False)
        try:
            self._api = GsuiteApi(json_keyfile, self.getuser())
        except:
            self.ui.gsuite_json_keyfile_error()
            raise

        self.should_do_syno_nametrans = self.getconfboolean('should_do_syno_nametrans', False)
        if self.should_do_syno_nametrans:
            self._special_folders = dict()
            self._all_visible_folder_names = set()

        self.should_skip_trash = self.getconfboolean('skiptrash', False)
        self.should_skip_junk = self.getconfboolean('skipjunk', False)

    def getsep(self):
        return '/'

    def getuser(self):
        user = self.getconf('remoteuser', None)
        return user

    def getfolder(self, foldername):
        """Return a Folder instance of this Gsuite service

        If necessary, scan and cache all foldernames to make sure that
        we only return existing folders and that 2 calls with the same
        name will return the same object."""
        folders = self.getfolders()
        for f in folders:
            if f.name == foldername:
                return f

        raise OfflineImapError("getfolder() asked for a nonexisting "
            "folder '%s'."% foldername, OfflineImapError.ERROR.FOLDER)

    def _memorize_special_folder_name(self, folder):
        """memorize the name of the special folder to help us do syno nametrans"""
        #add by Synology, translate the folder name considering folder flags, dot, and inbox
        for special_folder_key, special_folder_item in SYNO_SPECIAL_FOLDERS.items():
            if special_folder_item['id'] == folder.get_uidvalidity():
                self._special_folders[special_folder_key] = {
                        'name': folder.ffilter_name,
                        'length': len(folder.ffilter_name)
                }
                break

    def _do_syno_nametrans(self, folders):
        """How to translate the folder name by folder flags"""
        #add by Synology, translate the folder name considering folder id, dot, and inbox
        folder_separator = self.getsep()

        # If the result folder name also exists in the folder list, we do not do the conversion
        for special_folder_key, folder_name_item in self._special_folders.items():
            if SYNO_SPECIAL_FOLDERS[special_folder_key]['result_folder_name'] in self._all_visible_folder_names:
                self._special_folders.pop(special_folder_key)

        all_visible_folder_names = set()
        for folder in folders:
            # convert special folder to the result folder name
            for special_folder_key, folder_name_item in self._special_folders.items():
                if folder.ffilter_name == folder_name_item['name']:
                    folder.visiblename = SYNO_SPECIAL_FOLDERS[special_folder_key]['result_folder_name']
                    break
                elif folder.ffilter_name.startswith(folder_name_item['name'] + folder_separator):
                    folder.visiblename = SYNO_SPECIAL_FOLDERS[special_folder_key]['result_folder_name'] + folder.ffilter_name[folder_name_item['length']:]
                    break
            # convert the inbox and its subfolder to uppercase
            if 'inbox' == folder.ffilter_name.lower():
                folder.visiblename = 'INBOX'
            if folder.ffilter_name.lower().startswith('inbox' + folder_separator):
                folder.visiblename = 'INBOX' + folder.ffilter_name[5:]
            folder.visiblename = folder.visiblename.replace('.', '_')
            all_visible_folder_names.add(folder.visiblename)
        # handle the consecutive /
        for folder in folders:
            visiblename = folder.visiblename
            newname = visiblename
            idx = visiblename.rfind('/', 0, len(visiblename))
            while idx != -1:
                subname = visiblename[:idx]
                if subname not in all_visible_folder_names:
                    newname = subname + '_' + newname[idx+1:]
                elif idx + 1 < len(visiblename) and newname[idx+1] == '/':
                    newname = subname + '_' + newname[idx+1:]
                idx = visiblename.rfind('/', 0, idx)
            if newname.endswith('/'):
                newname = newname[:-1] + '_'
            folder.visiblename = newname

        # handle two folder is nametransed to the same name
        all_visiblename = set()
        for folder in folders:
            if folder.visiblename not in all_visiblename:
                all_visiblename.add(folder.visiblename)
            else:
                ori_visiblename = folder.visiblename
                new_name = ''
                i = 0
                while i < 1000000:
                    new_name = '{0}_{1}'.format(folder.visiblename, i)
                    if new_name not in all_visiblename:
                        break
                    i += 1
                folder.visiblename = new_name
                all_visiblename.add(folder.visiblename)
                for another_folder in folders:
                    #use ffilter_name to check hierarchy in original folder name
                    #use visiblename to check hierarchy in new folder name
                    if another_folder.ffilter_name.startswith(folder.ffilter_name + folder_separator) and \
                        another_folder.visiblename.startswith(ori_visiblename + folder_separator):
                        ori_another_visiblename = another_folder.visiblename
                        another_folder.visiblename = folder.visiblename + folder_separator + another_folder.visiblename[(len(ori_visiblename) + 1):]
                        if ori_another_visiblename in all_visiblename:
                            all_visiblename.remove(ori_another_visiblename)
                            all_visiblename.add(another_folder.visiblename)
        self._all_visible_folder_names = all_visiblename

    def getfolders(self):
        if self.folders is not None:
            return self.folders

        skipsyslabel = set(['CHAT', 'IMPORTANT', 'STARRED', 'UNREAD'])
        if self.should_skip_trash:
            skipsyslabel.add('TRASH')
        if self.should_skip_junk:
            skipsyslabel.add('SPAM')

        retry_left = 3
        while retry_left > 0:
            try:
                labels = self._api.listlabels().get('labels', [])
            except Exception as e:
                if retry_left > 1:
                    retry_left -= 1
                    self.ui.warn('{0} Retrying to list labels in 2 seconds, {1} times tried.'.format(e, 3 - retry_left))
                    time.sleep(2)
                else:
                    raise
            else:
                retry_left = 0
                break

        folders = []
        for label in labels:
            folder = self.getfoldertype()(self._api, label, self)
            folder.sync_this = folder.decide_should_sync_folder()
            if label['type'] == 'system':
                if label['id'].startswith('CATEGORY_') or label['id'] in skipsyslabel:
                    folder.sync_this = False
            folders.append(folder)
            if True == self.should_do_syno_nametrans:
                self._memorize_special_folder_name(folder)
                self._all_visible_folder_names.add(folder.visiblename)

        if self.should_do_syno_nametrans:
            self._do_syno_nametrans(folders)

        if not self.skip_archive:
            no_label = {
                'type': 'system',
                'labelListVisibility': 'labelHide',
                'messageListVisibility': 'hide',
                'id': '',
                'name': 'Archived'
            }
            folder = self.getfoldertype()(self._api, no_label, self)
            folders.append(folder)

        self.folders = folders
        return self.folders

    def getfoldertype(self):
        return folder.Gsuite.GsuiteFolder

    def forgetfolders(self):
        """Forgets the cached list of folders, if any.  Useful to run
        after a sync run."""

        self.folders = None

    def testremoteserver(self, istestconnectivity):
        if istestconnectivity:
            return
        self._api.listlabels()
