#!/bin/sh

. /var/packages/MailPlus-Server/target/scripts/daemon/util.sh

DKIM_SIGN_PID="/var/run/mailplus_server/opendkim_sign.pid"
DKIM_SIGN_CONF="${MAIL_SERVER_PKG}/target/etc/opendkim_sign.conf"
DKIM_SIGN_BIN="${MAIL_SERVER_PKG}/target/sbin/opendkim -x ${DKIM_SIGN_CONF}"
DKIM_SIGN_LOCK="/var/lock/mailplus_server/opendkim_sign.lock"
DKIM_SIGN_SOCK="/var/spool/@MailPlus-Server/opendkim_sign.sock"

checkStarted()
{
	if [ -f ${DKIM_SIGN_PID} ]; then
		PID=`cat ${DKIM_SIGN_PID}`
		PROCESS=`ps --pid ${PID} -o pid,args | grep opendkim | grep sign`
	fi

	if [ ! -z "${PROCESS}" ]; then
		return 1
	fi
	return 0
}

getpid()
{
	local pid="$(ps -C opendkim -o pid,args | grep "${DKIM_SIGN_CONF}" | awk '{print $1}')"
	echo "${pid}"
}

bin_status()
{
	if [ -f "${DKIM_SIGN_PID}" ]; then
		PID=`cat ${DKIM_SIGN_PID}`
		# Add -a to let grep search through binary content
		PROCESS=`cat "/proc/${PID}/cmdline" | grep -a "opendkim" | grep -a "sign"`

		if [ ! -z "${PROCESS}" ]; then
			return ${SERVICE_RUNNING}
		fi
		return ${SERVICE_DEAD_FPID}
	else
		if [ 0 != `ps -C opendkim -o pid,args | grep "${DKIM_SIGN_CONF}" | wc -l` ] ; then
			return ${SERVICE_RUNNING}
		fi
		return ${SERVICE_NOT_RUNNING}
	fi > /dev/null
}

conf_status()
{
	# Use jq to select key with prefix "enable_dkim_sign-" and value is "yes", and then count length
	local opendkim_sign_enable_number="$(${MAILPLUS_SERVER_BACKEND_BINARY} --getTree "spam" |
		jq 'with_entries(select((.key | startswith("enable_dkim_sign-")) and (.value == "yes"))) | length')"
	if [ "0" != "$opendkim_sign_enable_number" ]; then
		return ${RUNKEY_ENABLE}
	else
		return ${RUNKEY_DISABLE}
	fi
}

status()
{
	conf_status
	local opendkim_enable=$?
	bin_status
	local opendkim_status=$?

	if [ "${SERVICE_NOT_RUNNING}" -eq ${opendkim_status} ] && [ "${RUNKEY_DISABLE}" -ne "${opendkim_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	elif [ "${SERVICE_RUNNING}" -eq ${opendkim_status} ] && [ "${RUNKEY_ENABLE}" -ne "${opendkim_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	fi
	return ${opendkim_status}
}

runDaemon()
{
	rm -f "${DKIM_SIGN_SOCK}"
	su MailPlus-Server -s /bin/sh -c "${DKIM_SIGN_BIN}"

	local daemon_status=""
	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		status
		daemon_status=$?
		debug_log "daemon_status opendkim (sign): ${daemon_status}"
		if [ ${daemon_status} -eq "${SERVICE_RUNNING}" ]; then
			#binary is ready
			return 0
		elif [ ${daemon_status} -eq "${SERVICE_DEAD_FPID}" ]; then
			#[FIXME] Maybe we can do something to recover
			return 1
		else
			sleep "${SLEEP_TIME}"
		fi
	done
	return 1
}

start()
{
	(
	if flock -x 8; then
		checkStarted
		if [ 1 -eq $? ] ; then
			echo "opendkim (sign) is already running"
			restart_without_lock
		else
			runDaemon
		fi

		flock -u 8
		rm ${DKIM_SIGN_LOCK}
	fi
	)8> ${DKIM_SIGN_LOCK}
}

checkStopped()
{
	local stopped="no"
	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		if [ 0 == `ps -C opendkim -o pid,args | grep "${DKIM_SIGN_CONF}" | wc -l` ] ; then
			# opendkim (sign) is terminated
			stopped="yes"
			break
		fi
		sleep "${SLEEP_TIME}"
	done

	if [ "yes" != "${stopped}" ]; then
		echo "Force kill opendkim (sign)"
		kill -9 "$(getpid)" &> /dev/null
	fi
}

stopDaemon()
{
	kill `cat ${DKIM_SIGN_PID}` &> /dev/null
	checkStopped
}

stop()
{
	(
	if flock -x 8; then
		stopDaemon

		flock -u 8
		rm ${DKIM_SIGN_LOCK}
	fi
	)8> ${DKIM_SIGN_LOCK}
}

restart_without_lock()
{
	stopDaemon
	sleep 1
	runDaemon
}

restart()
{
	(
	if flock -x 8; then
		restart_without_lock

		flock -u 8
		rm ${DKIM_SIGN_LOCK}
	fi
	)8> ${DKIM_SIGN_LOCK}
}

reload()
{
	local pid="$(getpid)"

	if [ -n "${pid}" ]; then
		kill -s USR1 "${pid}"
	else
		err_log "opendkim (sign) is not alive"
	fi
}

composePID()
{
	local pid=$(cat ${DKIM_SIGN_PID} | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
	echo "${pid}"
}

composeScript()
{
	echo "/var/packages/MailPlus-Server/target/scripts/daemon/opendkim_sign.sh"
}


case "$1" in
	start)
		startOperation "mailserver_service_opendkim_sign"
		;;
	restart)
		restartOperation "mailserver_service_opendkim_sign"
		;;
	stop)
		stopOperation "mailserver_service_opendkim_sign"
		;;
	reload)
		reloadOperation "mailserver_service_opendkim_sign"
		;;
	status)
		status
		;;
	*)
		echo "Usage: $0 {start|stop|restart|status}" >&2
		exit 1
		;;
esac
