#!/bin/sh
MAILPLUS_SERVER_PATH="/var/packages/MailPlus-Server"
PRIVATE_LOCATION="${MAILPLUS_SERVER_PATH}/target"

VOLUME=$(/usr/bin/readlink ${PRIVATE_LOCATION} | /usr/bin/cut -d '/' -f2)

MAILPLUS_SERVER_MONITOR_PID="/var/run/mailplus_server/mailserver_syno_monitor.pid"
MAILPLUS_SERVER_BACKEND_BINARY="${PRIVATE_LOCATION}/bin/syno_mailserver_backend"

MEMCACHED_BINARY="/var/packages/MailPlus-Server/target/usr/bin/memcached"
MEMCACHED_PORT=8502
MEMCACHED_PID_FILE="/var/run/mailplus_server/mailserver_config_memcached.pid"

MAILPLUS_SERVER_PATH="/var/packages/MailPlus-Server"
PRIVATE_LOCATION="${MAILPLUS_SERVER_PATH}/target"
VOLUME=$(/usr/bin/readlink ${PRIVATE_LOCATION} | /usr/bin/cut -d '/' -f2)

CLUSTER_SCRIPT_LOG="/${VOLUME}/@maillog/clusterScript.log"

if [ ! -f "${CLUSTER_SCRIPT_LOG}" ]; then
	/bin/touch "${CLUSTER_SCRIPT_LOG}"
	/bin/chown MailPlus-Server:MailPlus-Server "${CLUSTER_SCRIPT_LOG}"
	/bin/chmod 644 "${CLUSTER_SCRIPT_LOG}"
fi
exec 2>>"${CLUSTER_SCRIPT_LOG}"

. /var/packages/MailPlus-Server/target/backend_hook/hookUtils.conf

## Helper functions
debugInfo()
{
	echo "$(date) - DEBUG: " "$@" >> "${CLUSTER_SCRIPT_LOG}"
}
errInfo()
{
	echo "$(date) - " "$@" >> "${CLUSTER_SCRIPT_LOG}"
}
checkFileExist()
{
	local filePath=$1
	local timeout=$2
	local fileExist=0

	for i in $(seq 1 1 "${timeout}"); do
		if [ -e "${filePath}" ];then
			fileExist=1
			break
		fi
		sleep 1
	done

	if [ ${fileExist} -ne 1 ]; then
		errInfo "File ${filePath} is not exist"
		return 1
	fi

	return 0
}
checkFileRemoved()
{
	local filePath=$1
	local timeout=$2
	local fileExist=1

	for i in $(seq 1 1 "${timeout}"); do
		if [ ! -e "${filePath}" ];then
			fileExist=0
			break
		fi
		sleep 1
	done

	if [ ${fileExist} -ne 0 ]; then
		errInfo "File ${filePath} still exist after timeout ${timeout}"
		return 1
	fi

	return 0
}
checkPidAlive()
{
	local pidPath=$1
	local timeout=$2
	local pidList=
	local pid=""

	if ! checkFileExist "${pidPath}" "${timeout}"; then
		return 1
	fi

	pidList="$(cat "${pidPath}")"
	for pid in ${pidList}; do
		if [ ! -e /proc/"${pid}" ]; then
			errInfo "pid:${pid} is not alive"
			return 1
		fi
	done

	return 0
}
waitPidDead()
{
	local pidPath=$1
	local timeout=$2
	local pidList=

	if checkFileRemoved "${pidPath}" "${timeout}"; then
		return 0
	fi

	## Force to kill pid
	pidList="$(cat "${pidPath}")"
	for pid in ${pidList}; do
		if [ -e /proc/"${pid}" ]; then
			kill -9 "${pid}"
			errInfo "Force to kill ${pid}"
		fi
	done

	return 1
}

## Check daemon
checkWatcherAlive()
{
	checkPidAlive "${MAILPLUS_SERVER_MONITOR_PID}" 60
}
waitWatcherStopped()
{
	waitPidDead "${MAILPLUS_SERVER_MONITOR_PID}" 5
	rm -f "${MAILPLUS_SERVER_MONITOR_PID}"
}
checkBackendAlive()
{
	local i=
	local timeout=$1

	for i in $(seq 1 1 "${timeout}"); do
		if initctl status pkg-MailPlus-Server_backend | grep "running"; then
			return 0
		fi
		sleep 1
	done

	return 1
}
# TODO waitBackendStopped
waitMemcachedStopped()
{
	waitPidDead "${MEMCACHED_PID_FILE}" 5
	rm -f "${MEMCACHED_PID_FILE}"
}

## Memchached
MemCachedStart()
{
	debugInfo "Start memcached..."
	## Allow 5m file sync which do base64 become about 6.6M so increase memcached size to 7M
	initctl start pkg-MailPlus-Server_kv_memcached \
		MEMCACHED_BINARY=${MEMCACHED_BINARY} \
		MEMCACHED_PORT=${MEMCACHED_PORT} \
		MEMCACHED_PID_FILE=${MEMCACHED_PID_FILE}

	return 0
}
MemCachedStop()
{
	debugInfo "Stop memcached..."

	initctl stop pkg-MailPlus-Server_kv_memcached

	waitMemcachedStopped
	return 0
}


## Watcher
WatcherStart()
{
	debugInfo "Watcher Start..."

	initctl start pkg-MailPlus-Server_watcher \
		WATCHER_OPTIONS="$@"

	if ! checkWatcherAlive; then
		return 1
	fi

	return 0
}
WatcherStop()
{
	debugInfo "Watcher Stop..."

	## Stop syno monitor
	initctl stop pkg-MailPlus-Server_watcher

	waitWatcherStopped
	return 0
}


## Monitor
MonitorStart()
{
	debugInfo "Monitor Start..."

	if ! MemCachedStart; then
		errInfo "Failed to start ${MEMCACHED_BINARY}"
	fi

	if ! WatcherStart; then
		errInfo "Failed to WatcherStart"
		return 1
	fi

	return 0
}
MonitorStop()
{
	debugInfo "Monitor Stop..."
	local timeout=5

	if ! WatcherStop; then
		errInfo "Failed to WatcherStop"
	fi

	MemCachedStop

	return 0
}
monitorRestart()
{
	MonitorStop
	sleep 1
	MonitorStart
}


## Backend daemon
BackendStatus()
{
	${BACKEND_BINARY} "backend-command" --command "is_redis_cluster_health"
}
BackendStart()
{
	debugInfo "Backend start..."

	if checkBackendAlive 1; then
		debugInfo "Backend already running..."
		return 0
	fi

	## check if config exist
	if ! ${MAILPLUS_SERVER_BACKEND_BINARY} --isClusterInit; then
		debugInfo "The cluster has not inited"
		return 2
	fi

	## Start daemon
	initctl start pkg-MailPlus-Server_backend

	if ! checkBackendAlive 30; then
		errInfo "Backend server is not alive"
		return 1
	fi

	return 0
}
BackendStop()
{
	debugInfo "Backend stop..."

	initctl stop pkg-MailPlus-Server_backend
	return 0
}

## Cluster
ClusterStart()
{
	debugInfo "Cluster start..."
	local timeout=30
	local ret=
	local i=

	BackendStart
	ret=$?
	if [ 1 -eq ${ret} ]; then
		errInfo "Failed to BackendStart"
		return 1
    elif [ 2 -eq ${ret} ]; then
		return 2
	fi

	if ! MonitorStart; then
		errInfo "Failed to MonitorStart"
		return 1
	fi

	debugInfo "Cluster start success"

	return 0
}
ClusterStop()
{
	debugInfo "Cluster stop..."
	local err=0

	if ! MonitorStop; then
		errInfo "Failed to MonitorStop"
		err=1
	fi
	if ! BackendStop; then
		errInfo "Failed to BackendStop"
		err=1
	fi

	local balancerIF=$(${MAILPLUS_SERVER_BACKEND_BINARY} --getBalancerIF)
	ifconfig "${balancerIF}" down

	return ${err}
}
ClusterRestart()
{
	ClusterStop
	sleep 1
	ClusterStart
}
ClusterStatus()
{
	BackendStatus
}

case "$1" in
	start)
		ClusterStart
		ret=$?
		if [ 1 -eq ${ret} ]; then
			errInfo "cluster start fail so stop cluster"
			ClusterStop > /dev/null 2>&1
			exit 1
		fi
		;;
	stop)
		ClusterStop
		;;
	restart)
		ClusterRestart
		;;
	status)
		ClusterStatus
		;;
	backend_start)
		BackendStart
		;;
	backend_stop)
		BackendStop
		;;
	backend_restart)
		BackendStop
		sleep 1
		BackendStart
		;;
	monitor_start)
		MonitorStart
		;;
	monitor_stop)
		MonitorStop
		;;
	monitor_restart)
		monitorRestart
		;;
	watch_start)
		WatcherStart "$2"
		;;
	watch_stop)
		WatcherStop
		;;
	watch_restart)
		WatcherStop
		sleep 1
		WatcherStart "$2"
		;;
	generate_certificate)
		GenerateInternalCert
		;;
	create_mailplus_share_env)
		PrepareMailPlusShareEnv "$2"
		;;
	*)
		echo "Usage: $0 {start|stop|restart|status|backend_start|backend_stop|backend_restart|monitor_start|monitor_stop|monitor_restart|watch_start|watch_stop}" >&2
		exit 1
		;;
esac
