#!/usr/bin/python
import sys
import os
import re
from multiprocessing import Pool, Queue

sys.path.append("/var/packages/MailPlus-Server/target/backend_hook/")
from _Utils import *
from _Define import *

PARREL_SYNC_USER=5
MAX_RETRY_TIMES=3
SYNC_USER_LOCK_TIMEOUT=30
SYNOUSER_BINARY = "synouser"
LdapReg = re.compile(r'.*\[(\w*)\@.*\].*')
DomainReg = re.compile(r'.*\[.*\\+(\w*)\].*')
syncQueue = Queue()

def getUserList(domainName):
    userList = []

    if not os.path.islink(MAILDIR_PREFIX_PATH):
        return userList

    mailDirPath = os.path.join(MAILDIR_PREFIX_PATH, domainName)
    for d in os.listdir(mailDirPath):
        p = os.path.join(mailDirPath, d)
        if not os.path.isdir(p):
            continue
        for uid in os.listdir(p):
            mailDir = "%s/%s/Maildir/cur" % (p, uid)
            if os.path.isdir(mailDir):
                ## get user short name from uid
                cmd = '%s --getuid %s | grep "User Name"' % (SYNOUSER_BINARY, uid)
                out, _ = executeCommand(cmd, True)
                if '@' in out:
                    r = LdapReg.match(out)
                    if r.group(1):
                        userList.append(r.group(1))
                elif '\\' in out:
                    r = DomainReg.match(out)
                    if r.group(1):
                        userList.append(r.group(1))
                else:
                    SYSLOG(LOG_ERR, "The output [%s] is not domain user" % str(out))
    return userList

def syncUser(syncObj):
    if not syncObj["username"]:
        return
    proto = syncObj["proto"]
    cmd = "/var/packages/MailPlus-Server/target/bin/doveadm sync -l %d -u %s %s:%s:%d" % \
            (SYNC_USER_LOCK_TIMEOUT, syncObj["username"], proto, syncObj["serverIP"], syncObj["serverPort"])
    userList, ret = executeCommand(cmd, False)
    if not ret:
        SYSLOG(LOG_ERR, "Failed to [%s]" % cmd)
        syncQueue.put(syncObj)
        return

def main():
    import optparse
    import time
    import traceback

    SYSLOG(LOG_ERR, "Start mailSync...")

    parser = optparse.OptionParser(usage=main.__doc__)
    parser.add_option('-d', '--domainName', dest='domainName',
            type=str, help=('domain type ldap|domain'))
    parser.add_option('-s', '--serverIP', dest='serverIP',
            type=str, help=('Dsync server IP'))
    parser.add_option('-p', '--port', dest='port',
            type=int, help=('Dsync server port'))
    options, args = parser.parse_args()

    if not options.domainName or not options.port:
        parser.print_help()
        exit(1)

    eAPI = BackendAPI()
    hostID = eAPI.getHostID()
    serverIDList = eAPI.mailconfGet("mailer").split(" ")
    for i in serverIDList:
        if i == hostID:
            continue
        eAPI.setServSyncTag("mailserver_service_dovecot", i, "dsync", False)
        eAPI.setServSyncTag("mailserver_service_postfix", i, "dsync", False)

    ## Become daemon mode
    pid = os.fork()
    if pid == 0:
        exit(0)
    pid = os.fork()
    if pid > 0:
        exit(0)

    proto = "tcp"
    enableDsyncSSL = eAPI.mailconfGet("dsync_ssl_enable")
    if "no" != enableDsyncSSL:
        proto = "tcps"

    try:
        ## Get user list
        syncList = []
        retry = 0
        userList = getUserList(options.domainName)
        if not userList:
            return 1
        for _ in userList:
            if not _:
                continue

            syncObj = {
                "username": _,
                "serverIP": options.serverIP,
                "serverPort": options.port,
                "proto": proto
            }
            syncList.append(syncObj)

        pool = Pool(PARREL_SYNC_USER)
        while len(syncList) > 0 and retry < MAX_RETRY_TIMES:
            ## Sync user
            pool.map(syncUser, syncList)

            ## There are no failed sync user => Success
            if syncQueue.empty():
                SYSLOG(LOG_ERR, "mailSync Done...")
                return 0

            retry += 1
            time.sleep(10)
            syncList = []
            while not syncQueue.empty():
                syncList.append(syncQueue.get())
    except Exception as e:
        trace = traceback.format_exc()
        tracelog(trace)
        SYSLOG(LOG_ERR, "MailSync exception: %s" % e)
    finally:
        for i in serverIDList:
            if i == hostID:
                continue
            eAPI.setServSyncTag("mailserver_service_dovecot", i, "dsync", True)
            eAPI.setServSyncTag("mailserver_service_postfix", i, "dsync", True)

    ## There are some user failed to sync
    userList = [_["username"] for _ in syncList if _["username"]]
    SYSLOG(LOG_ERR, "Failed to sync user: {0}".format(userList))
    return 1

if __name__ == '__main__':
    ret = main()
    exit(ret)
