# Maildir folder support
# Copyright (C) 2002-2016 John Goerzen & contributors
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import time
import os
from sys import exc_info
import offlineimap.accounts
from offlineimap import imaputil

from offlineimap import OfflineImapError, emailutil
from .Base import BaseFolder

import glob

import sys
for egg in glob.glob(r'/var/packages/MailPlus-Server/target/usr/bin/gsuite/*.egg'):
    sys.path.append(egg)

from googleapiclient.discovery import build
from googleapiclient.http import BatchHttpRequest
from base64 import urlsafe_b64decode
import googleapiclient.errors as errors

class GsuiteFolder(BaseFolder):
    def __init__(self, api, label, repository):
        self.sep = repository.getsep() # needs to be set before super().__init__
        # convert label name to mutf7 first
        name = imaputil.utf8_IMAP(label['name'].encode('utf8')) if label['name'] != repository.getsep() else '_'
        super(GsuiteFolder, self).__init__(name, repository)
        self.root = None
        self._label = label
        self._api = api

    def get_saveduidvalidity(self):
        """Return the previously cached UIDVALIDITY value

        :returns: UIDVALIDITY as (long) number or None, if None had been
            saved yet."""

        if hasattr(self, '_base_saved_uidvalidity'):
            return self._base_saved_uidvalidity
        uidfilename = self._getuidfilename()
        if not os.path.exists(uidfilename):
            self._base_saved_uidvalidity = None
        else:
            file = open(uidfilename, "rt")
            # Modified by Synology, do not conver uidvalidity to int
            self._base_saved_uidvalidity = file.readline().strip()
            file.close()
        return self._base_saved_uidvalidity

    # Interface from BaseFolder
    def save_uidvalidity(self):
        """Save the UIDVALIDITY value of the folder to the cache

        This function is not threadsafe, so don't attempt to call it
        from concurrent threads."""

        newval = self.get_uidvalidity()
        uidfilename = self._getuidfilename()

        # Modified by Synology, do not conver uidvalidity to int
        with open(uidfilename + ".tmp", "wt") as uidfile:
            uidfile.write("%s\n"% newval)
        os.rename(uidfilename + ".tmp", uidfilename)
        self._base_saved_uidvalidity = newval

    def get_uidvalidity(self):
        """Retrieve the current connections UIDVALIDITY value

        Maildirs have no notion of uidvalidity, so we just return a magic
        token."""
        # use label id as uidvalidity
        return self._label['id']

    def storesmessages(self):
        """Should be true for any backend that actually saves message bodies.
        (Almost all of them).  False for the LocalStatus backend.  Saves
        us from having to slurp up messages just for localstatus purposes."""

        return False

    def getenddate(self):
        """ Retrieve the value of the configuration option enddate """

        # copy from folder/IMAP
        from datetime import datetime, timedelta
        datestr = self.repository.getconf('enddate', None)
        try:
            if not datestr:
                return None
            date_time = datetime.strptime(datestr, "%Y-%m-%d")
            date_time = date_time + timedelta(days=1)
            date = date_time.timetuple()
            if date[0] < 1900:
                raise OfflineImapError("startdate led to year %d. "
                    "Abort syncing."% date[0],
                    OfflineImapError.ERROR.MESSAGE)
            return date
        except ValueError:
            raise OfflineImapError("invalid endtdate value %s",
                OfflineImapError.ERROR.MESSAGE)

    def genquery(self, labelId, min_date, max_date):
        conditions = []

        # find message without label.
        if len(labelId) == 0:
            conditions.append('has:nouserlabels -in:sent -in:chat -in:draft -in:inbox')

        # only migrate mail to draft and not other label
        if labelId != 'DRAFT':
            conditions.append('-in:draft')

        # date condition.
        if min_date != None:
            # Find out what the oldest message is that we should look at.
            conditions.append('after:%d/%d/%d'% (
                min_date[0], min_date[1], min_date[2]))
        if max_date != None:
            # Find out what the newest message is that we should look at.
            conditions.append('before:%d/%d/%d'% (
                max_date[0], max_date[1], max_date[2]))
        # maxsize condition.
        maxsize = self.getmaxsize()
        if maxsize != None:
            conditions.append('smaller:%d'% maxsize)

        return '%s'% ' '.join(conditions)

    def msglist_item_initializer(self, id, flags, time):
        return {'uid': id, 'flags': flags, 'time': time}

    # Interface from BaseFolder
    def cachemessagelist(self, min_date=None, min_uid=None, max_date=None):
        # we only cache mesaage list and do not get the message flags and times for now.
        # We get the flags and time of the message when we want to copy message or
        # we get the flags when we try to sync flags
        self.ui.loadmessagelist(self.repository, self)
        self.dropmessagelistcache()
        labelId = self.get_uidvalidity()
        self._query = self.genquery(labelId, min_date, max_date)
        nextPageToken = None
        while True:
            retry_left = 3
            while retry_left > 0:
                try:
                    response = self._api.listmessages(labelId, self._query, nextPageToken)
                    messages = response.get('messages', [])
                    for msg in messages:
                        uid = msg['id']
                        self.messagelist[uid] = self.msglist_item_initializer(uid, set(), 0)
                except Exception as e:
                    if retry_left > 1:
                        retry_left -= 1
                        self.ui.warn('{0} Retrying to cachemessagelist for folder [{1}] in 2 seconds, {2} times tried, error: '.format(e, self.getname(), 3 - retry_left))
                        time.sleep(2)
                    else:
                        raise
                else:
                    nextPageToken = response.get('nextPageToken', None)
                    retry_left = 0
                    break
            if not nextPageToken or offlineimap.accounts.Account.abort_NOW_signal.is_set():
                break

        self.ui.messagelistloaded(self.repository, self, self.getmessagecount())

    #Added by Synology. Rearrange logs.
    def get_decoded_name(self):
        """The deocded nametrans-transposed name of the folder's name."""

        return imaputil.decode_mailbox_name(self.getfullname())

    # Interface from BaseFolder
    def getmessage(self, uid):
        """Return the content of the message. Also update messagelist after fetch message"""

        if 'content' in self.messagelist[uid]:
            # We may get message content in _BaseFolder__syncmessagesto_copy
            # If we have the message content, just get it and return.
            content = self.messagelist[uid]['content']
            del self.messagelist[uid]['content']
            return content

        retry_left = 3
        while retry_left > 0:
            try:
                response = self._api.getmessage(uid)
            except Exception as e:
                if retry_left > 1:
                    retry_left -= 1
                    self.ui.warn('{0} Retrying to getmessage {1} for folder [{2}] in 2 seconds, {3} times tried'.format(e, uid, self.getname(), 3 - retry_left))
                    time.sleep(2)
                else:
                    raise
            else:
                retry_left = 0
                break

        raw = response.get('raw', '')
        message = urlsafe_b64decode(raw.encode('ASCII'))
        labelIds = response.get('labelIds', [])
        flags = set()
        if 'UNREAD' not in labelIds:
            flags.add('S')
        if 'DRAFT' in labelIds:
            flags.add('D')
        if 'STARRED' in labelIds:
            flags.add('F')
        self.messagelist[uid]['flags'] = flags
        self.messagelist[uid]['time'] = int(internaldate) / 1000

        return message

    # overwrite __syncmessagesto_copy in BaseFolder
    def _BaseFolder__syncmessagesto_copy(self, dstfolder, statusfolder):
        """Pass1: Copy locally existing messages not on the other side.

        This will copy messages to dstfolder that exist locally but are
        not in the statusfolder yet. The strategy is:

        1) Look for messages present in self but not in statusfolder.
        2) invoke copymessageto() on those which:
           - If dstfolder doesn't have it yet, add them to dstfolder.
           - Update statusfolder.

        This function checks and protects us from action in dryrun mode."""

        def getmsgcontentcallback(idx, response, exception):
            if exception:
                raise
            uid = response['id']
            raw = response.get('raw', '')
            message = urlsafe_b64decode(raw.encode('ASCII'))
            internaldate = response.get('internalDate', 0)
            labelIds = response.get('labelIds', [])
            flags = set()
            if 'UNREAD' not in labelIds:
                flags.add('S')
            if 'DRAFT' in labelIds:
                flags.add('D')
            if 'STARRED' in labelIds:
                flags.add('F')
            self.messagelist[uid]['flags'] = flags
            self.messagelist[uid]['time'] = int(internaldate) / 1000
            self.messagelist[uid]['content'] = message

        def retrygetmsg(batch):
            retry_left = 3
            while retry_left > 0:
                try:
                    batch.execute()
                except Exception as e:
                    if retry_left > 1:
                        retry_left -= 1
                        self.ui.warn('{0} Retrying to get message in folder [{1}] in 2 seconds, {2} times tried'.format(e, self.getname(), 3 - retry_left))
                        time.sleep(2)
                    else:
                        raise
                else:
                    retry_left = 0
                    break

        def batchsavestatus():
            statusfolder.savemessagesbulk(self.statusmsgs)
            self.statusmsgs = []

        # We have no new mail yet.
        self.have_newmail = False

        messagelist = self.getmessagelist()

        copylist = [uid for uid in self.getmessageuidlist()
            if not statusfolder.uidexists(uid)]
        num_to_copy = len(copylist)

        # Honor 'copy_ignore_eval' configuration option.
        if self.copy_ignoreUIDs is not None:
            for uid in self.copy_ignoreUIDs:
                if uid in copylist:
                    copylist.remove(uid)
                    self.ui.ignorecopyingmessage(uid, self, dstfolder)

        if num_to_copy > 0 and self.repository.account.dryrun:
            self.ui.info("[DRYRUN] Copy {} messages from {}[{}] to {}".format(
                num_to_copy, self, self.repository, dstfolder.repository)
            )
            return

        service = self._api.getservice()
        batch = BatchHttpRequest()
        batchuidlist = []
        batchsize = 50
        self.statusmsgs = []
        with self:
            for num, uid in enumerate(copylist):
                # Bail out on CTRL-C or SIGTERM.
                if offlineimap.accounts.Account.abort_NOW_signal.is_set():
                    batchuidlist = []
                    break

                if uid == 0:
                    self.ui.warn("Assertion that UID != 0 failed; ignoring message.")
                    continue

                # get message with gmail batch api
                batchuidlist.append(uid)
                batch.add(service.users().messages().get(userId='me', id=uid, format='raw', fields='id,raw,internalDate,labelIds'), getmsgcontentcallback)

                if len(batchuidlist) % batchsize == 0:
                    retrygetmsg(batch)
                    startidx = num - batchsize + 2
                    for idx, _uid in enumerate(batchuidlist):
                        # Bail out on CTRL-C or SIGTERM.
                        if offlineimap.accounts.Account.abort_NOW_signal.is_set():
                            batchsavestatus()
                            break
                        self.ui.copyingmessage(_uid, startidx + idx, num_to_copy, self, dstfolder)
                        self.copymessageto(_uid, dstfolder, statusfolder)
                    batch = BatchHttpRequest()
                    batchuidlist = []
                    batchsavestatus()


            if batchuidlist:
                retrygetmsg(batch)
                startidx = num_to_copy - (num_to_copy % batchsize) + 1 if num == num_to_copy - 1 else num
                for idx, _uid in enumerate(batchuidlist):
                    if offlineimap.accounts.Account.abort_NOW_signal.is_set():
                        batchsavestatus()
                        break
                    self.ui.copyingmessage(_uid, startidx + idx, num_to_copy, self, dstfolder)
                    self.copymessageto(_uid, dstfolder, statusfolder)
                batchsavestatus()

        # Execute new mail hook if we have new mail.
        if self.have_newmail:
            if self.newmail_hook != None:
                self.newmail_hook()


    def copymessageto(self, uid, dstfolder, statusfolder):
        """Copies a message from self to dst if needed, updating the status

        Note that this function does not check against dryrun settings,
        so you need to ensure that it is never called in a
        dryrun mode.

        :param uid: uid of the message to be copied.
        :param dstfolder: A BaseFolder-derived instance
        :param statusfolder: A LocalStatusFolder instance
        :param register: whether we should register a new thread."
        :returns: Nothing on success, or raises an Exception."""

        # Sometimes, it could be the case that if a sync takes awhile,
        # a message might be deleted from the maildir before it can be
        # synced to the status cache.  This is only a problem with
        # self.getmessage().  So, don't call self.getmessage unless
        # really needed.

        retry_left = 3
        while retry_left > 0:
            try:
                message = None
                # If any of the destinations actually stores the message body,
                # load it up.
                if dstfolder.storesmessages():
                    message = self.getmessage(uid)
                flags = self.getmessageflags(uid)
                rtime = self.getmessagetime(uid)

                # Succeeded? -> IMAP actually assigned a UID. If newid
                # remained negative, no server was willing to assign us an
                # UID. If newid is 0, saving succeeded, but we could not
                # retrieve the new UID. Ignore message in this case.
                new_uid = dstfolder.savemessage(uid, message, flags, rtime)
                if new_uid > 0:
                    if new_uid != uid:
                        # Got new UID, change the local uid to match the new one.
                        self.change_message_uid(uid, new_uid)
                        statusfolder.deletemessage(uid)
                        # Got new UID, change the local uid.
                    # Save uploaded status in the statusfolder.
                    self.statusmsgs.append((new_uid, flags, rtime))
                    # Check whether the mail has been seen.
                    if 'S' not in flags:
                        self.have_newmail = True
                elif new_uid == 0:
                    # Message was stored to dstfolder, but we can't find it's UID
                    # This means we can't link current message to the one created
                    # in IMAP. So we just delete local message and on next run
                    # we'll sync it back
                    # XXX This could cause infinite loop on syncing between two
                    # IMAP servers ...
                    self.deletemessage(uid)
                else:
                    raise OfflineImapError("Trying to save msg (uid %d) on folder "
                        "%s returned invalid uid %d"% (uid, dstfolder.getvisiblename(),
                        new_uid), OfflineImapError.ERROR.MESSAGE)
            except (KeyboardInterrupt): # Bubble up CTRL-C.
                raise
            except OfflineImapError as e:
                #Added by Synology. Retry to handle the imap server restarting
                if retry_left > 1:
                    retry_left -= 1
                    self.ui.warn('{0} Retrying to copy message with uid [{1}] in folder [{2}] in 2 seconds, {3} times tried.'.format(e, uid, self.getname(), 3 - retry_left))
                    time.sleep(2)
                else:
                    retry_left = 0
                    #Added by Synology. Send notification mail to users once the sync finishes
                    self.repository.account.sync_success = False
                    if e.severity > OfflineImapError.ERROR.MESSAGE:
                        raise # Bubble severe errors up.
                    statusfolder.add_message_error_num_by_one()
                    self.ui.error(e, exc_info()[2])
            except Exception as e:
                raise
                #Added by Synology. Retry to handle the imap server restarting
                if retry_left > 1:
                    retry_left -= 1
                    self.ui.warn('{0} Retrying to copy message with uid [{1}] in folder [{2}] in 2 seconds, {3} times tried.'.format(e, uid, self.getname(), 3 - retry_left))
                    time.sleep(2)
                else:
                    self.ui.error(e, exc_info()[2],
                      msg = "Copying message %s [acc: %s]"% (uid, self.accountname))
                    raise  # Raise on unknown errors, so we can fix those.
            else:
                retry_left = 0
                break


    # overwrite __syncmessagesto_flags in BaseFolder
    def _BaseFolder__syncmessagesto_flags(self, dstfolder, statusfolder):
        """Pass 3: Flag synchronization.

        Compare flag mismatches in self with those in statusfolder. If
        msg has a valid UID and exists on dstfolder (has not e.g. been
        deleted there), sync the flag change to both dstfolder and
        statusfolder.

        This function checks and protects us from action in ryrun mode.
        """

        # For each flag, we store a list of uids to which it should be
        # added.  Then, we can call addmessagesflags() to apply them in
        # bulk, rather than one call per message.
        addflaglist = {}
        delflaglist = {}
        changeflagdict = {}

        def getflaggedmessage(query, flag):
            labelId = self.get_uidvalidity()
            nextPageToken = None
            newquery = self._query + ' ' + query if self._query else query
            while True:
                retry_left = 3
                while retry_left > 0:
                    try:
                        response = self._api.listmessages(labelId, newquery, nextPageToken)
                        messages = response.get('messages', [])
                        for msg in messages:
                            uid = msg['id']
                            if self.uidexists(uid):
                                self.messagelist[uid]['flags'].add(flag)
                    except Exception as e:
                        if retry_left > 1:
                            retry_left -= 1
                            self.ui.warn('{0} Retrying to get flag {1} for folder [{2}] in 2 seconds, {3} times tried, error: '.format(e, flag, self.getname(), 3 - retry_left))
                            time.sleep(2)
                        else:
                            raise
                    else:
                        nextPageToken = response.get('nextPageToken', None)
                        retry_left = 0
                        break
                if not nextPageToken or offlineimap.accounts.Account.abort_NOW_signal.is_set():
                    break

        getflaggedmessage('-in:UNREAD', 'S')
        if self.get_uidvalidity() == 'DRAFT':
            getflaggedmessage('in:DRAFT', 'D')
        getflaggedmessage('in:STARRED', 'F')

        for uid in self.getmessageuidlist():
            # Ignore messages with negative UIDs missed by pass 1 and
            # don't do anything if the message has been deleted remotely
            if not dstfolder.uidexists(uid):
                continue

            if statusfolder.uidexists(uid):
                statusflags = statusfolder.getmessageflags(uid)
            else:
                statusflags = set()
            flags = self.messagelist[uid]['flags']

            addflags = flags - statusflags
            delflags = statusflags - flags

            for flag in addflags:
                if not flag in addflaglist:
                    addflaglist[flag] = []
                addflaglist[flag].append(uid)

            for flag in delflags:
                if not flag in delflaglist:
                    delflaglist[flag] = []
                delflaglist[flag].append(uid)

            if addflags or delflags:
                changeflagdict[uid] = flags

        for flag, uids in addflaglist.items():
            self.ui.addingflags(uids, flag, dstfolder)
            if self.repository.account.dryrun:
                continue # Don't actually add in a dryrun.
            dstfolder.addmessagesflags(uids, set(flag))

        for flag, uids in delflaglist.items():
            self.ui.deletingflags(uids, flag, dstfolder)
            if self.repository.account.dryrun:
                continue # Don't actually remove in a dryrun.
            dstfolder.deletemessagesflags(uids, set(flag))

        if changeflagdict:
            statusfolder.savemessagesflagsbulk(changeflagdict)

    # Interface from BaseFolder
    def getmessagetime(self, uid):
        return self.messagelist[uid]['time']

    # Interface from BaseFolder
    def getmessageflags(self, uid):
        return self.messagelist[uid]['flags']
