#!/usr/bin/env python
# -*- encoding=utf8 -*-
from __future__ import print_function
import os, sys

task_config = {}
user_list = {}
server_type = [
    'COMMON_IMAP',
    'EXCHANGE_SERVER',
    'GMAIL',
    'HOTMAIL',
    'YAHOO',
    'GSUITE',
    'OFFICE365'
]

def eprint(*args, **kwargs):
    print(*args, file=sys.stderr, **kwargs)

class User:
     def __init__(self, username, password):
         self.username = username
         self.password = password

def genExtensionFile(file_path):
    import re
    keyword_full_match_list = []
    keyword_regexp_list = []
    exception_full_match_list = []
    exception_regexp_list = []

    if 'filter_setting' not in task_config:
        with open(file_path, 'w') as out_file:
            out_file.write('#!/usr/bin/env python\n')
            out_file.write('def name_filter_function(folder_name):\n')
            out_file.write('    return true')
            out_file.close()
        return

    def get_lists(list_name, full_match_list, regexp_list):
        if list_name in task_config['filter_setting']:
            config_list = task_config['filter_setting'][list_name]
            for item in config_list:
                result = re.match('^\/(.+)\/$', item)
                if None != result:
                    pattern = result.group(1)
                    pattern = re.sub('\\\\', '\\\\\\\\', pattern)
                    pattern = re.sub('\'', '\\\'', pattern)
                    if 0 < len(pattern):
                        if '^' != pattern[0]:
                            pattern = '^' + pattern
                        if '$' != pattern[-1]:
                            pattern = pattern + '$'
                    regexp_list.append(pattern)
                else:
                    full_match_list.append(item.encode('utf-8'))

    def gen_regexp_pattern(regexp_list):
        pattern_str = '(^$)'
        if 0 < len(regexp_list):
            for idx, item in enumerate(regexp_list):
                pattern_str = u'{0:s}|({1:s})'.format(pattern_str, item)

        pattern_str = u'u\'{0:s}\''.format(pattern_str)
        return pattern_str

    get_lists('keyword_list', keyword_full_match_list, keyword_regexp_list)
    keyword_regexp_pattern = gen_regexp_pattern(keyword_regexp_list)

    get_lists('exception_list', exception_full_match_list, exception_regexp_list)
    exception_regexp_pattern = gen_regexp_pattern(exception_regexp_list)

    decision_expression = '    return True\n'
    if task_config['filter_setting']['is_blacklist']:
        decision_expression = '    return (folder_name in exception_full_match_list or None != re.search(exception_regexp_pattern, folder_name)) or (folder_name not in keyword_full_match_list and None == re.search(keyword_regexp_pattern, folder_name))\n'
    else:
        decision_expression = '    return (folder_name not in exception_full_match_list and None == re.search(exception_regexp_pattern, folder_name)) and (folder_name in keyword_full_match_list or None != re.search(keyword_regexp_pattern, folder_name))\n'

    with open(file_path, 'w') as out_file:
        out_file.write('#!/usr/bin/env python\n')
        out_file.write('# -*- coding: utf-8 -*-\n')
        out_file.write('import re\n')
        out_file.write('MUTF7_SHIFT_RE = re.compile(r"&[^-]*-|\+")\n')
        out_file.write('def decode_mailbox_name(name):\n')
        out_file.write('    def demodify(m):\n')
        out_file.write('        s = m.group()\n')
        out_file.write('        if s == "+":\n')
        out_file.write('            return "+-"\n')
        out_file.write('        return "+" + s[1:-1].replace(",", "/") + "-"\n')
        out_file.write('    ret = MUTF7_SHIFT_RE.sub(demodify, name)\n')
        out_file.write('    try:\n')
        out_file.write('        return ret.decode("utf-7").encode("utf-8")\n')
        out_file.write('    except UnicodeEncodeError:\n')
        out_file.write('        return name\n')
        out_file.write('\n')
        out_file.write('keyword_full_match_list = {0:s}\n'.format(str(keyword_full_match_list)))
        out_file.write(u'keyword_regexp_pattern = {0:s}\n'.format(keyword_regexp_pattern).encode('utf8'))
        out_file.write('exception_full_match_list = {0:s}\n'.format(str(exception_full_match_list)))
        out_file.write(u'exception_regexp_pattern = {0:s}\n'.format(exception_regexp_pattern).encode('utf8'))
        out_file.write('def name_filter_function(folder_name):\n')
        out_file.write('    folder_name = decode_mailbox_name(folder_name)\n')
        out_file.write(decision_expression)
        out_file.close()

def createMetadataDir(offlineimap_metadata_dir):
    try:
        if not os.path.exists(offlineimap_metadata_dir):
            os.mkdir(offlineimap_metadata_dir, 0o744)
    except OSError as e:
        eprint('Internal error. ' + str(e))
        exit(-1)

def getVarNames(user_info):
    account_name = user_info['src_acnt'].encode('utf8') + '->' + user_info['dest_acnt'].encode('utf8')
    src_repo_name = 'remote-' + user_info['src_acnt'].encode('utf8')
    dest_repo_name = 'local-' + user_info['dest_acnt'].encode('utf8')

    return account_name, src_repo_name, dest_repo_name

def fillRemoteRepoSetting(cf, repo_name, user, is_conn_test, task_name):
    global server_type
    import datetime

    repository = 'Repository ' + repo_name
    cf.add_section(repository)

    _type = server_type[task_config['server_setting']['type']]
    is_imap_type = 'GSUITE' != _type
    if is_imap_type:
        cf.set(repository, 'type', 'SYNOIMAP')
        cf.set(repository, 'remotehost', task_config['server_setting']['addr'].encode('idna'))
        cf.set(repository, 'remoteport', task_config['server_setting']['port'])
        cf.set(repository, 'auth_mechanisms', 'CRAM-MD5, PLAIN, LOGIN')

        cf.set(repository, 'ssl', task_config['server_setting']['use_ssl'])
        if task_config['server_setting']['use_ssl']:
            cf.set(repository, 'sslcacertfile','/etc/ssl/certs/ca-certificates.crt')
        else:
            cf.set(repository, 'starttls', 'no')
        if not is_conn_test:
            if True == task_config['server_setting']['enable_imap_path_prefix'] and task_config['server_setting']['imap_path_prefix']:
                namespace = task_config['server_setting']['imap_path_prefix']
                if 0 < len(namespace):
                    cf.set(repository, 'reference', namespace)
            if 'EXCHANGE_SERVER' == _type and True == task_config['server_setting']['enable_master_user']:
                cf.set(repository, 'remote_identity', user.username)
                cf.set(repository, 'remoteuser', task_config['server_setting']['master_acnt'])
                cf.set(repository, 'remotepass', task_config['server_setting']['master_passwd'])
            else:
                cf.set(repository, 'remoteuser', user.username)
                cf.set(repository, 'remotepass', user.password)
    else:
        cf.set(repository, 'type', 'Gsuite')
        json_keyfile_path = os.path.join('/var/packages/MailPlus-Server/target/var/migration_data', task_name, 'json_keyfile')
        cf.set(repository, 'json_keyfile', json_keyfile_path)
        cf.set(repository, 'remoteuser', user.username)
        if 'skip_archived_mail' in  task_config['filter_setting']:
            cf.set(repository, 'skip_archived_mail', task_config['filter_setting']['skip_archived_mail'])

    if not is_conn_test:
        if task_config['filter_setting']['enable_start_time']:
            start_time = datetime.date.fromtimestamp(task_config['filter_setting']['start_time'])
            cf.set(repository, 'startdate', start_time.isoformat())
        if task_config['filter_setting']['enable_end_time']:
            end_time = datetime.date.fromtimestamp(task_config['filter_setting']['end_time'])
            cf.set(repository, 'enddate', end_time.isoformat())
        if task_config['filter_setting']['ignore_trash']:
            cf.set(repository, 'skiptrash', task_config['filter_setting']['ignore_trash'])
        if task_config['filter_setting']['ignore_junk']:
            cf.set(repository, 'skipjunk', task_config['filter_setting']['ignore_junk'])
        if task_config['filter_setting']['enable_folder_name_filter']:
            cf.set(repository, 'folderfilter', 'name_filter_function')
        cf.set(repository, 'readonly', True)
        cf.set(repository, 'should_do_syno_nametrans', True)
    else:
        cf.set(repository, 'remoteuser', user.username)
        cf.set(repository, 'remotepass', user.password)
    cf.set(repository, 'singlethreadperfolder', True)

def fillLocalRepoSetting(cf, repo_name, user, is_conn_test, local_master_acnt = '', local_master_password = ''):
    global server_type

    repository = 'Repository ' + repo_name
    cf.add_section(repository)
    is_imap_type = 'GSUITE' != server_type[task_config['server_setting']['type']]
    if is_imap_type:
        cf.set(repository, 'type', 'SYNOIMAP')
    else:
        cf.set(repository, 'type', 'SYNOGsuiteIMAP')

    cf.set(repository, 'remotehost', '127.0.0.1')
    cf.set(repository, 'remoteport', '143')
    cf.set(repository, 'ssl', False)
    cf.set(repository, 'subscribe_on_creation', True)
    cf.set(repository, 'remoteuser', '{0:s}*{1:s}'.format(user.username, local_master_acnt))#used to login
    cf.set(repository, 'remotepass', local_master_password)
    cf.set(repository, 'auth_mechanisms', 'PLAIN, LOGIN')
    cf.set(repository, 'singlethreadperfolder', True)

def genConfForOfflineImap(task_name, is_test_conn = False, local_master_acnt = '', local_master_password = ''):
    import ConfigParser
    global user_list, server_type

    all_accounts = []
    cf = ConfigParser.ConfigParser()
    is_imap_type = 'GSUITE' != server_type[task_config['server_setting']['type']]

    cf.add_section('general')
    cf.set('general', 'syno_decrypt_password', True)
    if not is_test_conn:
        cf.set('general', 'profilename', task_name)
        offlineimap_metadata_dir = os.path.join('/var/packages/MailPlus-Server/target/var/migration_data', task_name, 'offlineimap_data')
        createMetadataDir(offlineimap_metadata_dir)
        cf.set('general', 'metadata', offlineimap_metadata_dir)
        offlineimap_config = os.path.join('/var/packages/MailPlus-Server/target/var/migration_data', task_name, 'offlineimap.conf')
        offlineimap_extention_script = os.path.join('/var/packages/MailPlus-Server/target/var/migration_data', task_name, 'extension.py')
        genExtensionFile(offlineimap_extention_script)
        cf.set('general', 'pythonfile', offlineimap_extention_script)
        cf.set('general', 'use_min_uid_when_set_date', False)
        cf.set('general', 'run_syno_post_sync_hook', True)
    else:
        offlineimap_metadata_dir = os.path.join('/tmp/mailplus_server/', task_name, 'offlineimap_data')
        createMetadataDir(offlineimap_metadata_dir)
        cf.set('general', 'metadata', offlineimap_metadata_dir)
        cf.set('general', 'socktimeout', 20)
        offlineimap_config = os.path.join('/tmp/mailplus_server/', task_name, 'offlineimap.conf')
        user_list = [{'src_acnt': 'DRevshYXaFkLhGTWXaJO', 'dest_acnt': 'DRevshYXaFkLhGTWXaJO', 'src_passwd': 'fake'}]

    for user_info in user_list:
        src_user = User(user_info['src_acnt'].encode('utf8'), user_info['src_passwd'])
        dest_user = User(user_info['dest_acnt'].encode('utf8'), 'fixme to be use in masteruser')
        account_name, src_repo_name, dest_repo_name = getVarNames(user_info)
        all_accounts.append(account_name)
        account = 'Account ' + account_name
        cf.add_section(account)
        cf.set(account, 'remoterepository', src_repo_name)
        cf.set(account, 'localrepository', dest_repo_name)
        if is_imap_type:
            cf.set(account, 'status_backend', 'syno_sqlite')
        else:
            cf.set(account, 'status_backend', 'syno_gsuite_sqlite')
        if (not is_test_conn) and task_config['filter_setting']['enable_size_limit']:
            cf.set(account, 'maxsize', int(task_config['filter_setting']['size_limit']) * 1024)

        try:
            fillRemoteRepoSetting(cf, src_repo_name, src_user, is_test_conn, task_name)
            fillLocalRepoSetting(cf, dest_repo_name, dest_user, is_test_conn, local_master_acnt, local_master_password)
        except BaseException as e:
            eprint("encounter error when process user {0} / {1}".format(src_user.username, dest_user.username))
            raise
            exit(-1)

    cf.set('general', 'accounts', ', '.join(all_accounts))
    cf.write(open(offlineimap_config, 'wb'))

def readJsonFile(task_name, file_name):
    import json

    json_file_path = os.path.join('/var/packages/MailPlus-Server/target/var/migration_data', task_name, file_name)

    json_data = None

    if not os.path.isfile(json_file_path):
        eprint('file [{0:s}] does not exist'.format(json_file_path))
        exit(-1)

    with open(json_file_path, 'r') as in_file:
        try:
            json_data = json.load(in_file)
        except ValueError:
            eprint('cannot parse file [{0:s}]'.format(json_file_path))
            exit(-1)

    return json_data


def main():
    import argparse, csv, json

    global task_config, user_list
    #must specify the path of the csv file
    parser = argparse.ArgumentParser(usage='%(prog)s [-c <-a addr -p port [-s])>] taskName')
    parser.add_argument('-c', '--gen-conn-test-conf', action='store_true', dest='is_conn_test' , default=False, help='generate the config for connection test')
    parser.add_argument('-a', '--addr', type=str, default=None, help='addr setting of connection test')
    parser.add_argument('-p', '--port', type=int, default=None, help='port setting of connection test')
    parser.add_argument('-s', '--use_ssl', action='store_true', dest='use_ssl', default=False, help='ssl setting of connection test')
    parser.add_argument('args', nargs=argparse.REMAINDER)
    args = parser.parse_args()
    if 1 != len(args.args):
        eprint('must specify task name')
        exit(-1)

    task_name = args.args[0]
    local_master_acnt = ''
    local_master_password = ''

    if not args.is_conn_test:
        user_list = readJsonFile(task_name, 'user_list')
        task_config = readJsonFile(task_name, 'config')
        local_master_acnt = raw_input()
        local_master_password = raw_input()
    else:
        if args.addr is None or args.port is None:
            parser.print_usage()
            exit(-1)
        server_setting = {
            'addr': args.addr.decode('utf8'),
            'port': args.port,
            'use_ssl': args.use_ssl,
            'type': 1
        }
        task_config = {
            'server_setting': server_setting
        }
    genConfForOfflineImap(task_name, args.is_conn_test, local_master_acnt, local_master_password)

if __name__ == '__main__':
    main()
