#!/bin/sh

. /var/packages/MailPlus-Server/target/backend_hook/hookUtils.conf
. /var/packages/MailPlus-Server/target/scripts/daemon/util.sh

YES=0
NO=1

RUNNING_STRING="running"
NEED_UPDATE_STRING="need update"
NOT_RUNNING_STRING="not running"
NOT_INSTALLED_STRING="not installed"

SYNOAV_MCAFEE_PACKAGE_NAME="AntiVirus-McAfee"
SYNOAV_MCAFEE_PACKAGE_PATH="/var/packages/${SYNOAV_MCAFEE_PACKAGE_NAME}"

ANTI_VIRUS_ENGINE="synoav-mcafee"
SYNOAVSCAN_BIN="${SYNOAV_MCAFEE_PACKAGE_PATH}/target/bin/synoavscan"
SYNOAVDSCAN_BIN="${SYNOAV_MCAFEE_PACKAGE_PATH}/target/bin/synoavdscan"
SYNOAVDCONTROL_BIN="${SYNOAV_MCAFEE_PACKAGE_PATH}/target/bin/synoavdcontrol"
SECURITY_CHECK="${MAIL_SERVER_PKG}/target/scripts/SecurityCheck.sh"
SYNOAVD_STOP_HOOK_PATH="${MAIL_SERVER_PKG}/target/hook/MailPlusServer-SynoAV-McAfeeStop.sh"

ANTIVIRUS_STATUS="/tmp/MailPlusServer.antivirus.status"

_synoav_mcafee_package_status()
{
	local package_status="$(synopkg status ${SYNOAV_MCAFEE_PACKAGE_NAME})"

	case "${package_status}" in
		*"started"*)
			return ${SERVICE_RUNNING}
			;;
		*"stopped"*)
			return ${SERVICE_NOT_RUNNING}
			;;
		*"No such package"*)
			return ${SERVICE_NOT_INSTALLED}
			;;
		*)
			return ${SERVICE_UNKNOWN}
			;;
	esac
}

_synoav_mcafee_has_daemon()
{
	if [ -x "${SYNOAVDSCAN_BIN}" ] && [ -x "${SYNOAVDCONTROL_BIN}" ]; then
		return ${YES}
	else
		return ${NO}
	fi
}

bin_status()
{
	_synoav_mcafee_package_status
	local synoav_mcafee_package_status=$?

	if [ ${SERVICE_RUNNING} -ne ${synoav_mcafee_package_status} ]; then
		return ${synoav_mcafee_package_status}
	fi

	checkPidFileStatus "$(daemon_name)" "$(daemon_pid)"
	local synoavd_status=$?
	if [ "${SERVICE_RUNNING}" != ${synoavd_status} ]; then
		return ${SERVICE_NOT_INSTALLED}
	fi

	checkProcessRun "synoavscan --update"
	local synoavscan_update_status=$?
	if [ "${SERVICE_RUNNING}" != ${synoavscan_update_status} ] && [ "${SERVICE_NOT_RUNNING}" != ${synoavscan_update_status} ] ; then
		return ${synoavscan_update_status}
	fi

	checkProcessRun "synoavdscan update"
	local synoavdscan_update_status=$?
	if [ "${SERVICE_RUNNING}" != ${synoavdscan_update_status} ] && [ "${SERVICE_NOT_RUNNING}" != ${synoavdscan_update_status} ] ; then
		return ${synoavdscan_update_status}
	fi

	if [ "${SERVICE_RUNNING}" == ${synoavscan_update_status} ] ; then
		return "${SERVICE_UPDATING}"
	fi

	if [ "${SERVICE_RUNNING}" == ${synoavdscan_update_status} ] ; then
		return "${SERVICE_UPDATING}"
	fi

	if [ "${SERVICE_RUNNING}" == ${synoavd_status} ] ; then
		return "${SERVICE_RUNNING}"
	fi

	if [ "${SERVICE_NOT_RUNNING}" == ${synoavd_status} ]; then
		return "${SERVICE_NOT_RUNNING}"
	fi

	return "${SERVICE_UNKNOWN}"
}

conf_status()
{
	checkConfKey "anti_virus_enable"
	local anti_virus_enable=$?
	local anti_virus_engine="$(${MAILPLUS_SERVER_BACKEND_BINARY} --getConfKeyVal "anti_virus_engine")"

	if [ "${RUNKEY_ENABLE}" == ${anti_virus_enable} ] && [ "${ANTI_VIRUS_ENGINE}" == ${anti_virus_engine} ] ; then
		return "${RUNKEY_ENABLE}"
	else
		return "${RUNKEY_DISABLE}"
	fi
}

engine_status()
{
	_synoav_mcafee_package_status
	local synoav_mcafee_package_status=$?
	_synoav_mcafee_has_daemon
	local synoav_mcafee_has_daemon_result=$?

	if [ ${SERVICE_RUNNING} -ne ${synoav_mcafee_package_status} ]; then
		return ${synoav_mcafee_package_status}
	fi
	if [ ${NO} -eq ${synoav_mcafee_has_daemon_result} ]; then
		return ${SERVICE_NOT_INSTALLED}
	fi

	local engine_status_result="$(${SYNOAVDSCAN_BIN} engine_status)"

	case "${engine_status_result}" in
		*initialized*)
			return ${SERVICE_RUNNING}
			;;
		*terminated*)
			return ${SERVICE_NOT_RUNNING}
			;;
		*)
			return ${SERVICE_UNKNOWN}
			;;
	esac
}

status()
{
	engine_status
	local synoav_mcafee_engine_status=$?

	if [ ${SERVICE_NOT_INSTALLED} -eq ${synoav_mcafee_engine_status} ] || [ ${SERVICE_UNKNOWN} -eq ${synoav_mcafee_engine_status} ]; then
		return ${synoav_mcafee_engine_status}
	fi

	conf_status
	local synoav_mcafee_enable=$?

	if [ "${SERVICE_NOT_RUNNING}" -eq ${synoav_mcafee_engine_status} ] && [ "${RUNKEY_DISABLE}" -ne "${synoav_mcafee_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	elif [ "${SERVICE_RUNNING}" -eq ${synoav_mcafee_engine_status} ] && [ "${RUNKEY_ENABLE}" -ne "${synoav_mcafee_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	fi

	return ${synoav_mcafee_engine_status}
}

fallback_to_clamav()
{
	${MAILPLUS_SERVER_BACKEND_BINARY} --setConfKeyVal "anti_virus_engine" "clamav"
}

start()
{
	local synoav_mcafee_status="$(synoav_mcafee_status)"

	if [ "${RUNNING_STRING}" != "${synoav_mcafee_status}" ]; then
		err_log "synoav_mcafee_status is ${synoav_mcafee_status}, fallback to clamav"
		fallback_to_clamav
		return 1
	fi

	# Add stop hook before starting engine to ensure that if AntiVirus-McAfee
	# is stopped when starting engine, MailPlus Server will fallback to ClamAV.
	${SYNOAVDCONTROL_BIN} add_stop_script ${SYNOAVD_STOP_HOOK_PATH}
	${SYNOAVDSCAN_BIN} init_engine
	rm ${ANTIVIRUS_STATUS}

	local daemon_status=""
	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		engine_status
		daemon_status=$?
		debug_log "daemon_status synoav_mcafee: ${daemon_status}"
		if [ ${daemon_status} -eq "${SERVICE_RUNNING}" ] || [ ${daemon_status} -eq "${SERVICE_UPDATING}" ] || [ ${daemon_status} -eq "${SERVICE_STARTING}" ]; then
			#binary is ready

			warn_log "SynoAV-McAfee is started"
			return ${START_SUCCESSFUL}
		elif [ ${daemon_status} -eq "${SERVICE_DEAD_FPID}" ]; then
			#[FIXME] Maybe we can do something to recover
			return ${START_FAILED}
		else
			sleep "${SLEEP_TIME}"
		fi
	done
	return ${START_FAILED}
}

stop()
{
	engine_status
	local synoav_mcafee_engine_status=$?

	# SynoAV-McAfee is already disabled, no need to stop
	if [ ${SERVICE_NOT_INSTALLED} -eq ${synoav_mcafee_engine_status} ] || [ ${SERVICE_UNKNOWN} -eq ${synoav_mcafee_engine_status} ]; then
		return 0
	fi

	${SYNOAVDSCAN_BIN} terminate_engine
	# Remove stop hook after starting engine to follow the same order in start().
	${SYNOAVDCONTROL_BIN} remove_stop_script ${SYNOAVD_STOP_HOOK_PATH}

	# Because of suspicious memory leak when terminating McAfee engine, we restart the daemon
	${SYNOAVDSCAN_BIN} restart
	warn_log "SynoAV-McAfee is stopped"
}

restart()
{
	stop
	sleep 1
	start
}


composePID()
{
	#it is very different becuase it will do updating/starting/start
	local pid=""

	pid=`ps -I | grep "synoavscan --update" | grep -v "grep" | awk -F" " 'NR == 1 {print $1}'`
	if [ x"" != x"${pid}" ]; then
		echo "${pid}"
		return
	fi

	if [ -f "$(daemon_pid)" ]; then
		pid=$(cat "$(daemon_pid)" | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
		echo "${pid}"
		return
	fi
	err_log "Not found any for synoav_mcafee"
	echo ""
}

composeScript()
{
	echo "/var/packages/MailPlus-Server/target/scripts/daemon/synoav_mcafee.sh"
}

update()
{
	if isLoadBalancer; then
		nowTime=$(date "+%Y/%m/%d %H:%M:%S")
		${MAILPLUS_SERVER_BACKEND_BINARY} --setConfKeyVal "anti_virus_last_update_time" "${nowTime}"
	fi
}

update_local_only()
{
	${SYNOAVSCAN_BIN} --update

	# restart daemon
	${SYNOAVDSCAN_BIN} restart
	${SYNOAVDSCAN_BIN} init_engine

	# reset antivirus status
	rm ${ANTIVIRUS_STATUS}

	# TODO may need to use synoav engine data
	# update anti-spam status for dashboard
	${SECURITY_CHECK} "anti_virus"
}

reload()
{
	# SynoAV do not have setting, no need to reload
	#${SYNOAVDSCAN_BIN} reload
	return
}

check_bin_and_exec()
{
	if [ -e "$1" ]; then
		"$1" "$2"
	else
		echo "Error: McAfee not exist"
		return 1
	fi
}

update_date()
{
	check_bin_and_exec ${SYNOAVDSCAN_BIN} update_date
}

virus_info()
{
	check_bin_and_exec ${SYNOAVDSCAN_BIN} virus_info
}

engine_version()
{
	check_bin_and_exec ${SYNOAVDSCAN_BIN} engine_version
}

release_date()
{
	check_bin_and_exec ${SYNOAVDSCAN_BIN} release_date
}

check_license()
{
	check_bin_and_exec ${SYNOAVDSCAN_BIN} check_license
}

check_update()
{
	check_bin_and_exec ${SYNOAVDSCAN_BIN} check_update
}

daemon_pid()
{
	check_bin_and_exec ${SYNOAVDCONTROL_BIN} daemon_pid
}

daemon_name()
{
	check_bin_and_exec ${SYNOAVDCONTROL_BIN} daemon_name
}

synoav_mcafee_status()
{
	_synoav_mcafee_package_status
	local synoav_mcafee_package_status=$?
	_synoav_mcafee_has_daemon
	local synoav_mcafee_has_daemon_result=$?

	if [ ${SERVICE_RUNNING} -eq ${synoav_mcafee_package_status} ]; then
		if [ ${YES} -eq ${synoav_mcafee_has_daemon_result} ]; then
			echo ${RUNNING_STRING}
		else
			echo ${NEED_UPDATE_STRING}
		fi
	elif [ ${SERVICE_NOT_RUNNING} -eq ${synoav_mcafee_package_status} ]; then
		echo ${NOT_RUNNING_STRING}
	else
		echo ${NOT_INSTALLED_STRING}
	fi
}

case "$1" in
	start)
		startOperation "mailserver_service_synoav_mcafee"
		;;
	restart)
		restartOperation "mailserver_service_synoav_mcafee"
		;;
	stop)
		stopOperation "mailserver_service_synoav_mcafee"
		;;
	status)
		status
		;;
	update)
		update
		;;
	update_local_only)
		update_local_only
		;;
	reload)
		reload
		;;
	fallback_to_clamav)
		fallback_to_clamav
		;;
	update_date)
		update_date
		;;
	virus_info)
		virus_info
		;;
	engine_version)
		engine_version
		;;
	release_date)
		release_date
		;;
	check_license)
		check_license
		;;
	check_update)
		check_update
		;;
	daemon_pid)
		daemon_pid
		;;
	daemon_name)
		daemon_name
		;;
	synoav_mcafee_status)
		synoav_mcafee_status
		;;
	*)
		echo "Usage: $0 {start|stop|restart|status|update|reload|update_local_only|fallback_to_clamav|update_date|virus_info|engine_version|release_date|check_license|check_update|daemon_pid|daemon_name|synoav_mcafee_status}" >&2
		exit 1
		;;
esac
