#!/bin/bash
. /var/packages/MailPlus-Server/target/backend_hook/hookUtils.conf
. /var/packages/MailPlus-Server/target/scripts/DsmVersionUtil.sh

MAILPLUS_SERVER_IP_CHANGE_LOCK_FILE="/tmp/mailplus_server/ip_post_change.lock"
CLUSTER_LOCK_KEY="change_ip"

hookPreCheck()
{
	# do nothing when interface change on shutdown step
	if /usr/syno/bin/synobootseq --is-shutdown > /dev/null 2>&1 ; then
		exit 0
	fi
	# do nothing when booting, we'll run this in upstart when booting is done.
	if /usr/syno/bin/synobootseq --is-booting-up > /dev/null 2>&1 ; then
		exit 0
	fi
	# if package is not enabled skip ip change hook, because no change would apply to redis
	if ! isPackageEnable; then
		exit 1
	fi
	if [ "tun1000" = "${IFNAME}" ]; then
		exit 0
	fi
	if [ "${NEW_ADDRESS}" = "none" ];then
		warn_log "skip ip change because we think [${IFNAME}] [${ORIGIN_ADDRESS} == ${NEW_ADDRESS}] [${oldAddr} == ${newAddr}]"
		exit 0
	fi
}

processAddress()
{
	local result=$1
	local default=$2
	if [ -z "${result}" -o "${result}" = "none" ]; then
		result=${default}
	fi
	echo "${result}"
}

getProcessedAddress()
{
	newAddr=
	oldAddr=
	if checkIfWeCare; then
		#If address is none, convert to current node ip,
		#or it will break the cluster because we set the backend ip to none
		oldAddr=$(processAddress "${ORIGIN_ADDRESS}" "${hostIP}")
		newAddr=$(processAddress "${NEW_ADDRESS}" "${ORIGIN_ADDRESS}")
	else
		#If address is none, convert to empty string
		oldAddr=$(processAddress "${ORIGIN_ADDRESS}" "")
		newAddr=$(processAddress "${NEW_ADDRESS}" "")
	fi
}

updateNodeList()
{
	local nodeID="$(getHostID)"
	local nodeIPList="$(${MAILPLUS_SERVER_BACKEND_BINARY} --getHostIPList)"
	local key="node_ip_list-${nodeID}"

	info_log "[MailPlusServerIPChange] - start updateNodeList"

	if [ -z "${oldAddr}" ] || ! echo "${nodeIPList}" | grep -w "${oldAddr}" ; then
		if echo "${nodeIPList}" | grep -w "${newAddr}"; then
			nodeIPList=$(echo "${nodeIPList}" | sed -e "s/\<${oldAddr}\>//")
		else
			nodeIPList="${nodeIPList} ${newAddr}"
		fi
	else
		nodeIPList=$(echo "${nodeIPList}" | sed -e "s/\<${oldAddr}\>/${newAddr}/")
	fi

	#remove unnecessary spaces
	nodeIPList=$(echo "${nodeIPList}" | sed -e "s/\s\+/ /g" -e "s/^ *//g" -e "s/ *$//g")

	## Remove execute command flag
	rm -fr "${FIREWALL_REFRESH_FLAG}"

	if ! ${MAILPLUS_SERVER_BACKEND_BINARY} --setConfKeyVal "${key}" "${nodeIPList}"; then
		 err_log "[MailPlusServerIPChange] - Failed to update node ip list [${key}] to [${nodeIPList}] when ip change"
	fi

	if checkIfWeCare; then
		for i in $(seq 1 1 10); do
			if [ -e "${FIREWALL_REFRESH_FLAG}" ];then
				rm -fr "${FIREWALL_REFRESH_FLAG}"
				break
			fi
			info_log "[MailPlusServerIPChange] - Wait for firewall refresh..."
			sleep 1
		done
	fi
	info_log "[MailPlusServerIPChange] - done updateNodeList"
}

checkIfWeCare()
{
	## Check if interface/ip is mailplus server used interface/ip
	if [ -n "${hostIF}" ]; then
		if [ "${hostIF}" == "${IFNAME}" ]; then
			return 0
		fi
	else [ "${ORIGIN_ADDRESS}" == "${hostIP}" ];
		return 0
	fi
	return 1
}

decideSkipHook()
{
	local currHostIP=$(getHostIP)
	if [ "${hostIF}" = "${IFNAME}" -a "${currHostIP}" = "${newAddr}" ]; then
		warn_log "skip ip change because we think current host ip and new address is the same [${IFNAME}=>${currHostIP}]"
		return 0
	fi
	return 1
}
updateBackendPre()
{
	info_log "[MailPlusServerIPChange] - start updateBackendPre"
	if ! checkIfWeCare; then
		return 0
	fi

	# acquire lock for change ip
	for i in $(seq 1 1 180); do
		if acquireClusterLock "${CLUSTER_LOCK_KEY}" "180"; then
			break
		fi
		[ $((i % 10)) -eq 1 ] && info_log "[MailPlusServerIPChange] - Wait for cluster script lock..."
		sleep 1
	done
	if decideSkipHook; then
		warn_log "[MailPlusServerIPChange] - updateBackendPre skip ip change because we think current host ip and new address is the same [${IFNAME}=>${newAddr}]"
		releaseClusterLock "${CLUSTER_LOCK_KEY}"
		exit 0
	fi

	${MAILPLUS_SERVER_BACKEND_BINARY} --setServSync mailserver_service_dovecot "${hostID}"

	## Notify new master IP, NOTICE: here load balancer equal redis master
	local masterIP="$(${BACKEND_BINARY} backend-command --command get_redis_master_ip)"
	if [ "${nodeNum}" -gt 1 -a "127.0.0.1" == "${masterIP}" ] ; then
		# There are 2 nodes, and I am master node
		${BACKEND_BINARY} "backend-command" --command "set" --key "/tmp/new_master_ip" --value "${newAddr}" --ttl "60"
	fi
	info_log "[MailPlusServerIPChange] - done updateBackendPre"
}
updateBackendPost()
{
	info_log "[MailPlusServerIPChange] - start updateBackendPost [${newAddr}]"

	## update certificate have to before updateSelfMapping to avoid get old certificate
	if ! GenerateInternalCert "onlyCert"; then
		err_log "[MailPlusServerIPChange] - Failed to regen certificate according new IP"
	fi

	if decideSkipHook; then
		warn_log "[MailPlusServerIPChange] - updateBackendPost skip ip change because we think current host ip and new address is the same [${IFNAME}=>${newAddr}]"
		exit 0
	fi
	if ! checkIfWeCare; then
		info_log "[MailPlusServerIPChange] - quick done updateBackendPost"
		return 0
	fi

	if [ -n "${newAddr}" ]; then
		## update backend config and mapping table
		if ! ${MAILPLUS_SERVER_BACKEND_BINARY} --updateSelfMapping "${newAddr}"; then
			err_log "[MailPlusServerIPChange] - Failed to regen mailplus server id-ip mapping table"
		fi
	else
		 err_log "[MailPlusServerIPChange] - skip updateSelfMapping because ip is none"
	fi

	${MAILPLUS_SERVER_BACKEND_BINARY} --setServSyncDone mailserver_service_dovecot "${hostID}"

	releaseClusterLock "${CLUSTER_LOCK_KEY}"
	info_log "[MailPlusServerIPChange] - done updateBackendPost"
}

case $1 in
	--sdk-mod-ver)
	#Print SDK support version
	echo "1.0";
	;;
	--name)
	#Print package name
	echo "MailPlus-Server";
	;;
	--pkg-ver)
	#Print package version
	echo "1.0";
	;;
	--vendor)
	#Print package vendor
	echo "Synology";
	;;
	--pre)
		hostIP=$(getHostIP)
		hostIF=$(getHostIF)
		hostID=$(getHostID)
		nodeNum=$(${MAILPLUS_SERVER_BACKEND_BINARY} --getPeersNum)

		warn_log "[MailPlusServerIPChange] $1 - IFNAME [${IFNAME}], ORIGIN_ADDRESS [${ORIGIN_ADDRESS}], NEW_ADDRESS [${NEW_ADDRESS}]"
		getProcessedAddress
		hookPreCheck
		if [ "${nodeNum}" -ne 1 ]; then
			#when there are 2 nodes, do the job in --pre
			updateBackendPre
			updateNodeList
		fi
	;;
	--post)
		hostIP=$(getHostIP)
		hostIF=$(getHostIF)
		hostID=$(getHostID)
		nodeNum=$(${MAILPLUS_SERVER_BACKEND_BINARY} --getPeersNum)

		warn_log "[MailPlusServerIPChange] $1 - IFNAME [${IFNAME}], ORIGIN_ADDRESS [${ORIGIN_ADDRESS}], NEW_ADDRESS [${NEW_ADDRESS}]"
		getProcessedAddress
		hookPreCheck
		if [ "${nodeNum}" -eq 1 ]; then
			#when there is only one node, do the job in --post
			updateBackendPre
			updateNodeList
		fi
		(
		if flock -n -x 8; then
			## avoid exec many times when encounter series net interface change event
			sleep 1
			flock -u 8
			rm ${MAILPLUS_SERVER_IP_CHANGE_LOCK_FILE}

			updateBackendPost
		fi
		)8> ${MAILPLUS_SERVER_IP_CHANGE_LOCK_FILE} &
	;;
	*)
	echo "Usage: $0 --sdk-mod-ver|--name|--pkg-ver|--vendor|--pre|--post"
	;;
esac

