#!/bin/bash
. /var/packages/MailPlus-Server/target/backend_hook/hookUtils.conf
loadEnv

DIRECTOR_SCRIPT="/var/packages/${PACKAGE_NAME}/target/scripts/daemon/director.sh"
DOVECOT_CONF_LOCK="/var/lock/mailplus_server/syno_dovecot_conf_lock"

## Check identity
NODE_ID=$(getHostID)
mailer_servers="$(${MAILPLUS_SERVER_BACKEND_BINARY} --getConfKeyVal "mailer")"
if ! containString "${NODE_ID}" "${mailer_servers}"; then
	exit 0
fi

ftsEnable="$(${MAILPLUS_SERVER_BACKEND_BINARY} --getConfKeyVal "fts_enabled")"

getDovecotConfLock()
{
	local timeout=30
	## Which have to be the same as define SZF_DOVECOT_CONF_LOCK
	local lockFile="${DOVECOT_CONF_LOCK}"

	exec 100>${lockFile}
	for i in `seq 1 $timeout`; do
		flock -x -n 100
		if [ 0 -eq $? ]; then
			return 0
		fi
		sleep 1
	done
	err_log "Get lock fail \"${lockFile}\""
	return 1
}

freeDovecotConfLock()
{
	local lockFile="${DOVECOT_CONF_LOCK}"
	flock -u 100
}

getDecodeScriptPath()
{
	local VERSION_FILE=/etc.defaults/VERSION
	local major=$(/usr/syno/bin/synogetkeyvalue "${VERSION_FILE}" majorversion)
	local minor=$(/usr/syno/bin/synogetkeyvalue "${VERSION_FILE}" minorversion)
	if [[ "${major}" == "6" && "${minor}" == "0" ]];
	then
		echo "/var/packages/MailPlus-Server/target/libexec/dovecot/syno_decode2text_6_0.sh"
		return
	fi
	echo "/var/packages/MailPlus-Server/target/libexec/dovecot/syno_decode2text.sh"
}

changeDovecotMasterConf()
{
	local localMountIP=
	local storage_servers="$(${MAILPLUS_SERVER_BACKEND_BINARY} --getConfKeyVal "storager")"
	local enable_balancer_address="$(${MAILPLUS_SERVER_BACKEND_BINARY} --getConfKeyVal "enable_balancer_address")"
	local hostIP="$(getHostIP)"
	local listenIP=

	local doveadmPort="24245"
	local imapPort="$(imapPop3PortGet "imap")"
	local imapsPort="$(imapPop3PortGet "imaps")"
	local pop3Port="$(imapPop3PortGet "pop3")"
	local pop3sPort="$(imapPop3PortGet "pop3s")"
	## FIXME private ip range
	local privateIPRange="0.0.0.0"
	local imapEnable="$(${MAILPLUS_SERVER_BACKEND_BINARY} --getConfKeyVal "imap_enabled")"
	local enableDsyncSSL=$(${MAILPLUS_SERVER_BACKEND_BINARY} --getConfKeyVal "dsync_ssl_enable")
	local indexerWorkerNum="$(nproc)"
	local extraPlugins="$(synogetkeyvalue "/var/packages/MailPlus-Server/etc/extra_plugins" "mail_plugins")"
	local mailPlugins="${extraPlugins}"
	local attachDecodeScript=$(getDecodeScriptPath)

	if [ "${enable_balancer_address}" == "no" ]; then
		## Have to disable director or the port may hold by director
		${DIRECTOR_SCRIPT} stop
		listenIP="0.0.0.0"
	else
		listenIP="${hostIP},127.0.0.1"
	fi
	if [ "yes" != "${imapEnable}" ]; then
		imapNodeIP="127.0.0.1"
	else
		imapNodeIP="${listenIP}"
	fi

	if [ "yes" == "${ftsEnable}" ]; then
		mailPlugins="${mailPlugins} fts fts_lucene"
	fi

	if containString "${NODE_ID}" "${storage_servers}"; then
		localMountIP=${hostIP}
	else
		localMountIP=$(${MAILPLUS_SERVER_BACKEND_BINARY} --getConfKeyVal "storage_runtime_mapping-${NODE_ID}")
	fi

	if [ "${indexerWorkerNum}" -le "1" ]; then
		# at least two workers to avoid starvation
		indexerWorkerNum="2"
	fi

	## Get lock
	if ! getDovecotConfLock; then
		err_log "[Fatal] Failed to get dovecot conf lock, but still generate dovecot conf"
	fi

	cp -f "/${CONF_TEMPLATE_DIR}/10-master.template" "/var/packages/${PACKAGE_NAME}/target/etc/dovecot/conf.d/10-master.conf"
	sed -i -e "s|__NODE_IP__|${listenIP}|g" \
		-i -e "s|__IMAP_NODE_IP__|${imapNodeIP}|g" \
		-i -e "s|__IMAP_PORT__|${imapPort}|g" \
		-i -e "s|__IMAPS_PORT__|${imapsPort}|g" \
		-i -e "s|__POP3_PORT__|${pop3Port}|g" \
		-i -e "s|__POP3S_PORT__|${pop3sPort}|g" \
		-i -e "s|__DOVEADM_PORT__|${doveadmPort}|g" \
		-i -e "s|__NPROC__|${indexerWorkerNum}|g" \
		-i -e "s|__ATTACHMENT_DECODE_SCRIPT__|${attachDecodeScript}|g" "/var/packages/${PACKAGE_NAME}/target/etc/dovecot/conf.d/10-master.conf"

	if [ "${enableDsyncSSL}" != "no" ]; then
		## Create ca-chain
		GenerateInternalCaChain
		sed -i -e "s|__ENABLE_DSYNC_SSL__|yes|g" "/var/packages/${PACKAGE_NAME}/target/etc/dovecot/conf.d/10-master.conf"
	else
		sed -i -e "s|__ENABLE_DSYNC_SSL__|no|g" "/var/packages/${PACKAGE_NAME}/target/etc/dovecot/conf.d/10-master.conf"
	fi

	local hasReplica="no"
	local servIP=
	local servers=""
	for servID in ${storage_servers}; do
		servIP="$(${MAILPLUS_SERVER_BACKEND_BINARY} --idToIP "${servID}")"
		if [ "${servIP}" != "${localMountIP}" ]; then
			if [ "${enableDsyncSSL}" != "no" ]; then
				servers="${servers} tcps:${servIP}:${doveadmPort}"
			else
				servers="${servers} tcp:${servIP}:${doveadmPort}"
			fi
			hasReplica="yes"
		fi
	done
	sed -i -e "s|__MAIL_REPLICATION__|${servers}|g" "/var/packages/${PACKAGE_NAME}/target/etc/dovecot/conf.d/10-master.conf"

	cp -f "/${CONF_TEMPLATE_DIR}/10-mail.template" "/var/packages/${PACKAGE_NAME}/target/etc/dovecot/conf.d/10-mail.conf"
	if [ "${hasReplica}" == "no" ]; then
		mailPlugins="${mailPlugins} notify"
	else
		mailPlugins="${mailPlugins} notify replication"
	fi
	sed -i -e "s|__DOVECOT_PLUGINS__|${mailPlugins}|g" "/var/packages/${PACKAGE_NAME}/target/etc/dovecot/conf.d/10-mail.conf"

	freeDovecotConfLock
}

jiebaCallback()
{
	if isKeyChanged "fts_enabled" || isKeyChanged "jieba_enabled"; then
		jiebaEnable="$(${MAILPLUS_SERVER_BACKEND_BINARY} --getConfKeyVal "jieba_enabled")"
		if [ x"${jiebaEnable}" = x"yes" ] && [ x"${ftsEnable}" = x"yes" ]; then
			${CALLBACK_BINARY} "jieba" "start"
		else
			${CALLBACK_BINARY} "jieba" "stop"
		fi
	fi
}

if isTreeChanged "imap_pop3" || isKeyChanged "storage_runtime_mapping-${NODE_ID}" || isKeyChanged "balancer_master_node" || isKeyChanged "fts_enabled" || isKeyChanged "enable_balancer_address" || isTreeChanged "mapping_table" || isKeyChanged "storager" || containString "node_ip_list-" "${_MAIL_mailserver_trigger_events}"; then
	jiebaCallback

	## Convert syno config
	changeDovecotMasterConf	## Change 10-master.conf

	if ! ${SET_CONF_BINARY} "dovecot"; then
		err_log "Failed to set conf dovecot"
		exit 1
	fi

	## Actions
	if ! ${CALLBACK_BINARY} "dovecot" "${_MAIL_mailserver_trigger_events}"; then
		err_log "Failed to callback dovecot"
		exit 1
	fi
fi

exit 0
