# Noninteractive UI
# Copyright (C) 2002-2016 John Goerzen & contributors.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import logging
import threading
import traceback

from offlineimap.ui.UIBase import UIBase

class SynoFormatter(logging.Formatter):
    """Specific Formatter that adds thread information to the log output.
    prepend the account infomation in the log message"""

    def format(self, record):
        """Override format to add thread information."""

        #super() doesn't work in py2.6 as 'logging' uses old-style class
        log_str = logging.Formatter.format(self, record)

        t_name = record.threadName
        if t_name == 'MainThread':
            return '{0:s}:General:{1:s}'.format(record.name, log_str)

        try:
            acct_name = self.ui.threadaccounts[threading.currentThread()]
            return '{0:s}:{1:s}:{2:s}'.format(record.name, acct_name, log_str)
        except ValueError:
            return '{0:s}:General:{1:s}'.format(record.name, log_str)

    def setui(self, ui):
        self.ui = ui

class Synologyui(UIBase):
    """'Syslog' sets log level to INFO and outputs to syslog instead of stdout"""

    def __init__(self, config, loglevel=logging.INFO):
        super(Synologyui, self).__init__(config, loglevel)

        #add by Synolog. Write exit status.
        #0: exit normally
        #1: exit by signal
        #2: exit with error
        self._sync_exit_status = None
        import atexit
        atexit.register(self.__write_sync_exit_status)

    #add by Synolog. Write exit status.
    @property
    def sync_exit_status(self):
        return self._sync_exit_status
    @sync_exit_status.setter
    def sync_exit_status(self, value):
        if value in [0, 1]:
            self._sync_exit_status = value
        else:
            self._sync_exit_status = 2
    def __write_sync_exit_status(self):
        import os, signal
        exit_status_file_path = self.config.getmetadatadir() + "/exit_status"
        if None == self.sync_exit_status:
            return
        with open(exit_status_file_path, "w") as w_fd:
            w_fd.write(str(self.sync_exit_status) + "\n")
            w_fd.close()
        if os.path.exists("/var/run/mailplus_server/syno_mailserverd.pid"):
            with open("/var/run/mailplus_server/syno_mailserverd.pid", "r") as pidfile:
                pid = int(pidfile.read())
                os.kill(pid, signal.SIGUSR1)

    def setup_consolehandler(self):
        # create syslog handler
        ch = logging.handlers.SysLogHandler('/dev/log', facility=logging.handlers.SysLogHandler.LOG_LOCAL1)
        # create formatter and add it to the handlers
        self.formatter = SynoFormatter("%(message)s")
        self.formatter.setui(self)
        ch.setFormatter(self.formatter)
        # add the handlers to the logger
        self.logger.addHandler(ch)
        return ch

    def setup_sysloghandler(self):
        pass # Do not honor -s (log to syslog) CLI option.

    def threadException(self, thread):
        """Called when a thread has terminated with an exception.
        The argument is the ExitNotifyThread that has so terminated."""

        #add by synology, writer some error log to be shown on ui
        self.internal_error()
        super(Synologyui, self).threadException(thread)

    def terminate(self, exitstatus = 0, errortitle = None, errormsg = None):
        """ Removing the _log_con_handler before logging the call_stack of exceptions.
        It only write to the file log handler if it exists"""

        # NOTE: The whole program should exit soon after the call.
        # As a result, I think remove the _log_con_handler would be fine here.
        # There is no more programe log at that time. So remove the handler
        # would not affect to much.

        #add by Synolog. Write exit status.
        if 0 != exitstatus or not self.exc_queue.empty() or None != errortitle or None != errormsg or self.uidval_problem:
            self.sync_exit_status = 2

        self.logger.removeHandler(self._log_con_handler)
        if 0 == len(self.logger.handlers):
            #it may cause "No handlers could be found for logger" error, so add a NullHandler
            null_handler = logging.NullHandler()
            self.logger.addHandler(null_handler)

        super(Synologyui, self).terminate(exitstatus = exitstatus, errortitle = errortitle, errormsg = errormsg)

    #Modified by Synology. Rearrange logs.
    def _msg(self, msg, should_show_on_ui = False):
        """Display a message."""

        # TODO: legacy function, rip out.
        self.info(msg, should_show_on_ui)

    #Modified by Synology. Rearrange logs.
    def info(self, msg, should_show_on_ui = False):
        """Display a message."""

        self.logger.info('{0}:{1}'.format(1 if should_show_on_ui else 0, msg))

    #Modified by Synology. Rearrange logs.
    def warn(self, msg, minor=0, should_show_on_ui = False):
        self.logger.warning('{0}:{1}'.format(1 if should_show_on_ui else 0, msg))

    #Modified by Synology. Rearrange logs.
    def error(self, exc, exc_traceback=None, msg=None, should_show_on_ui = False):
        if msg:
            self.logger.error('{0}:{1}'.format(1 if should_show_on_ui else 0, msg))
        self.logger.error('0:{0}'.format(exc))

        instant_traceback = exc_traceback
        if not self.debuglist:
            # only output tracebacks in debug mode
            instant_traceback = None
        # push exc on the queue for later output
        self.exc_queue.put((msg, exc, exc_traceback))
        if instant_traceback:
            self.logger.error('0:{0}'.format(traceback.format_tb(instant_traceback)))

    #Modified by Synology. Rearrange logs.
    def logacctchange(self, account, isdone = False, duration = 0):
        """Output that we start/finished syncing an account."""
        #add by Synology. Write account change log

        if isdone == True:
            self.info("Mail migration %s completed (Time: %dmins %02d secs)."%
                (account.get_synology_name(), duration // 60, duration % 60), should_show_on_ui = True)
        else:
            self.info("Mail migration %s is about to start."% account.get_synology_name(), should_show_on_ui = True)

    #Modified by Synology. Rearrange logs.
    def syncingfolder(self, srcrepos, srcfolder, destrepos, destfolder):
        """Called when a folder sync operation is started."""

        self.info("Mail migration for folder [%s] is about to start."% srcfolder.get_decoded_name(), should_show_on_ui = True)

    #Modified by Synology. Rearrange logs.
    def finish_syncing_folder(self, srcfolder):
        """Called when a folder sync operation is done."""

        self.info("Mail migration for folder [%s] completed."% srcfolder.get_decoded_name(), should_show_on_ui = True)

    #Modified by Synology. Rearrange logs.
    def validityproblem(self, folder):
        self.uidval_problem = True
        self.warn("UID validity unmatched. Folder [%s] was skipped for migration."% \
                (folder.get_decoded_name(),), should_show_on_ui = True)
        self.warn("UID validity problem for folder [%s] (repo %s) "
                            "(saved %d; got %d); skipping it. Please see offlineimap"
                            "FAQ and manual on how to handle this."% \
               (folder.get_decoded_name(), folder.getrepository(),
                folder.get_saveduidvalidity(), folder.get_uidvalidity()))

    #Modified by Synology. Rearrange logs.
    def ignorecopyingmessage(self, uid, src, destfolder):
        """Output a log line stating which message is ignored."""
        if self.debuglist:
            self.debug('imap', "IGNORED: Copy message UID %s %s:%s -> %s"% (
                uid, src.repository, src, destfolder.repository))

    #Modified by Synology. Rearrange logs.
    def copyingmessage(self, uid, num, num_to_copy, src, destfolder):
        """Output a log line stating which message we copy."""

        if self.debuglist:
            self.debug('imap', "Copy message UID %s (%d/%d) %s:%s -> %s"% (
                uid, num, num_to_copy, src.repository, src,
                destfolder.repository))

    def savemessage(self, debugtype, uid, flags, folder):
        """Output a log line stating that we save a msg."""

        self.debug(debugtype, "Write mail '%s:%s' with flags %s"%
            (folder, str(uid), repr(flags)))

    def deletingmessages(self, uidlist, destlist):
        ds = self.folderlist(destlist)
        prefix = "[DRYRUN] " if self.dryrun else ""
        self.info("{0}Deleting {1} messages ({2}) in {3}".format(
            prefix, len(uidlist), u','.join(uidlist), ds))

    #Modified by Synology. Rearrange logs.
    def addingflags(self, uidlist, flags, dest):
        if self.debuglist:
            self.debug('imap', "Adding flag %s to %d messages on %s" % (
                ", ".join(flags), len(uidlist), dest))

    #Modified by Synology. Rearrange logs.
    def deletingflags(self, uidlist, flags, dest):
        if self.debuglist:
            self.debug('imap', "Deleting flag %s from %d messages on %s" % (
                ", ".join(flags), len(uidlist), dest))

    #Added by Synology. Rearrange logs.
    def folder_encounter_error(self, folder_name, error_num):
        """Show log if encountered error in folder"""

        self.logger.error('1:The folder [{0}] encountered {1} migration errors.'.format(folder_name, error_num))

    #Added by Synology. Rearrange logs.
    def account_encounter_error(self, account):
        """Show log if encountered error in any folder"""

        self.logger.error('1:{0} folders encountered migration errors.'.format(account.folder_error_num))

    #Added by Synology. Rearrange logs.
    def all_authentication_failed(self):
        self.logger.error('1:Authentication failed.')

    #Added by Synology. Rearrange logs.
    def ssl_error(self, reason):
        self.logger.error('1:{0}'.format(reason))

    #Added by Synology. Rearrange logs.
    def connection_refused(self, host, port):
        self.logger.error('1:Connection to server [{0}:{1}] failed.'\
                'Please make sure the server address and the port are valid.'\
                .format(host, port))

    #Added by Synology. Rearrange logs.
    def hostname_lookup_error(self, host):
        self.logger.error('1:Could not resolve hostname [{0}]. '\
                'Make sure you have configured the server address correctly.'\
                .format(host))

    #Added by Synology. Rearrange logs.
    def internal_error(self):
        self.logger.error('1:Errors found. Please export the log to view details.')

    #Added by Synology. Rearrange logs.
    def gsuite_json_keyfile_error(self):
        self.logger.error('1:Your private key file may be invalid.')
