#!/bin/sh

. /var/packages/MailPlus-Server/target/scripts/daemon/util.sh

RSPAMD_PID="/var/run/mailplus_server/rspamd.pid"
RSPAMD="${MAIL_SERVER_PKG}/target/bin/rspamd"
SZF_RSPAMD_LOCK="/tmp/syno_rspamd.lock"

checkStarted()
{
	if [ -f ${RSPAMD_PID} ]; then
		PID=`cat ${RSPAMD_PID}`
		# Add -a to let grep search through binary content
		PROCESS=`cat "/proc/${PID}/cmdline" | grep -a rspamd`
	fi

	if [ ! -z "${PROCESS}" ]; then
		return 1
	fi
	return 0
}

bin_status()
{
	checkPidFileStatus "rspamd" "${RSPAMD_PID}"
}

conf_status()
{
	checkConfKey "anti_virus_enable"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		return "${RUNKEY_ENABLE}"
	fi
	checkConfKey "spam_enable"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		return "${RUNKEY_ENABLE}"
	fi
	checkConfKey "mcp_enable"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		return "${RUNKEY_ENABLE}"
	fi
	return "${RUNKEY_DISABLE}"
}

status()
{
	conf_status
	local rspamd_enable=$?
	bin_status
	local rspamd_status=$?

	if [ "${SERVICE_NOT_RUNNING}" -eq ${rspamd_status} ] && [ "${RUNKEY_DISABLE}" -ne "${rspamd_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	elif [ "${SERVICE_RUNNING}" -eq ${rspamd_status} ] && [ "${RUNKEY_ENABLE}" -ne "${rspamd_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	fi
	return ${rspamd_status}
}

runDaemon()
{
	${RSPAMD} -u MailPlus-Server -g system

	local daemon_status=""
	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		status
		daemon_status=$?
		debug_log "daemon_status rspamd: ${daemon_status}"
		if [ ${daemon_status} -eq "${SERVICE_RUNNING}" ]; then
			#binary is ready
			return ${START_SUCCESSFUL}
		elif [ ${daemon_status} -eq "${SERVICE_DEAD_FPID}" ]; then
			#[FIXME] Maybe we can do something to recover
			return ${START_FAILED}
		else
			sleep "${SLEEP_TIME}"
		fi
	done
	return ${START_FAILED}
}

start()
{
	(
	if flock -x 8; then
		checkStarted
		if [ 1 -eq $? ] ; then
			echo "rspamd is already running"
		else
			runDaemon
		fi

		flock -u 8
		rm ${SZF_RSPAMD_LOCK}
	fi
	)8> ${SZF_RSPAMD_LOCK}
}

checkStopped()
{
	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		if ! pidof rspamd > /dev/null; then
			# rspamd is terminated
			break
		fi
		sleep "${SLEEP_TIME}"
	done

	# if rspamd still exists, force stop it
	if pidof rspamd > /dev/null; then
		killall -9 rspamd
	fi
}

stopDaemon()
{
	kill `cat ${RSPAMD_PID}` &> /dev/null
	checkStopped
}

stop()
{
	(
	if flock -x 8; then
		stopDaemon
		rm ${RSPAMD_PID} &> /dev/null

		flock -u 8
		rm ${SZF_RSPAMD_LOCK}
	fi
	)8> ${SZF_RSPAMD_LOCK}
}

restart()
{
	(
	if flock -x 8; then
		stopDaemon
		sleep 1
		runDaemon

		flock -u 8
		rm ${SZF_RSPAMD_LOCK}
	fi
	)8> ${SZF_RSPAMD_LOCK}
}

getpid()
{
	ps -C 'rspamd' -o pid= -o args= | grep 'rspamd: main' | awk '{print $1}'
}

reload()
{
	if [ -f ${RSPAMD_PID} ]; then
		kill -s HUP `cat ${RSPAMD_PID}`
	else
		local pid="$(getpid)"

		if [ -n "${pid}" ]; then
			kill -s HUP "${pid}"
		else
			err_log "rspamd is not alive"
		fi
	fi
}

composePID()
{
	local pid=$(cat ${RSPAMD_PID} | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
	echo "${pid}"
}

composeScript()
{
	echo "/var/packages/MailPlus-Server/target/scripts/daemon/rspamd.sh"
}

case "$1" in
	start)
		startOperation "mailserver_service_rspamd"
		;;
	restart)
		restartOperation "mailserver_service_rspamd"
		;;
	stop)
		stopOperation "mailserver_service_rspamd"
		;;
	reload)
		reloadOperation "mailserver_service_rspamd"
		;;
	status)
		status
		;;
esac
