#!/bin/sh

. /var/packages/MailPlus-Server/target/scripts/daemon/util.sh
. /var/packages/MailPlus-Server/target/scripts/DsmVersionUtil.sh

MAILPLUS_SERVER_PKG="/var/packages/MailPlus-Server"
MAILPLUS_SERVER_TARGET_DIR="${MAILPLUS_SERVER_PKG}/target"

BACKEND_BINARY="${MAILPLUS_SERVER_PKG}/target/usr/bin/syno-mailplus-server-go-utils"

PERL5LIB="${MAILPLUS_SERVER_TARGET_DIR}/lib/MIMEDefang:"\
"${MAILPLUS_SERVER_TARGET_DIR}/lib/perl5/vendor_perl:"\
"${MAILPLUS_SERVER_TARGET_DIR}/lib/perl5/core_perl:"\
"${MAILPLUS_SERVER_TARGET_DIR}/share/perl5/vendor_perl"

# Mimedefang
MIMEDEFANG_DAEMON="pkg-MailPlus-Server_mimedefang"

MIMEDEFANG_LOCK="/var/lock/mailplus_server/mimedefang.lock"
MIMEDEFANG_SPOOL="/var/run/mailplus_server/mimedefang"

MIMEDEFANG_PID="/var/run/mailplus_server/mimedefang.pid"
MIMEDEFANG_SOCK="${MIMEDEFANG_SPOOL}/mimedefang.sock"
MIMEDEFANG_SERVICE_NAME="mailserver_service_mimedefang"
MIMEDEFANG_SCRIPT_NAME="${MAILPLUS_SERVER_PKG}/target/scripts/daemon/mimedefang.sh"

# Multiplexor
MULTIPLEXOR_DAEMON="pkg-MailPlus-Server_mimedefang_multiplexor"
MD_MX_CTRL="${MAIL_SERVER_PKG}/target/bin/md-mx-ctrl"

MULTIPLEXOR_PID="/var/run/mailplus_server/mimedefang-multiplexor.pid"
MULTIPLEXOR_SOCK="${MIMEDEFANG_SPOOL}/multiplexor.sock"
MULTIPLEXOR_STATUS_SOCK="${MIMEDEFANG_SPOOL}/multiplexor-status.sock"
MULTIPLEXOR_NOTIFICATION_SOCK="${MIMEDEFANG_SPOOL}/multiplexor-notification.sock"
MULTIPLEXOR_SERVICE_NAME="mailserver_service_mimedefang_multiplexor"

MULTIPLEXOR_SLAVE_NUM="5" # FIXME: need tune slave number

# Mimedefang command
MIMEDEFANG_BIN="${MAILPLUS_SERVER_TARGET_DIR}/bin/mimedefang "\
"-U MailPlus-Server "\
"-z ${MIMEDEFANG_SPOOL} "\
"-p ${MIMEDEFANG_SOCK} "\
"-m ${MULTIPLEXOR_SOCK} "\
"-G "\
"-q "\
"-P ${MIMEDEFANG_PID} "\
"-X "\
"-N "\
"-S mail "

# Multiplexor command
MULTIPLEXOR_BIN="${MAILPLUS_SERVER_TARGET_DIR}/bin/mimedefang-multiplexor "\
"-U MailPlus-Server "\
"-x ${MULTIPLEXOR_SLAVE_NUM} "\
"-b 300 "\
"-Z "\
"-z ${MIMEDEFANG_SPOOL} "\
"-s ${MULTIPLEXOR_SOCK} "\
"-a ${MULTIPLEXOR_STATUS_SOCK} "\
"-p ${MULTIPLEXOR_PID} "\
"-l "\
"-q 128 "\
"-O ${MULTIPLEXOR_NOTIFICATION_SOCK} "\
"-S mail "\
"-E "\
"-G "

checkAndUpgradeQuarantine()
{
	local UPGRADE_QUARANTINE="/var/packages/MailPlus-Server/target/bin/upgrade_quarantine.pl"
	local MCP_QUARANTINE_UPGRADE_FLAG="/var/packages/MailPlus-Server/etc/upgrade_mcp_quarantine"
	local MCP_QUARANTINE_UPGRADE_PID="/var/run/mailplus_server/mcp_quarantine_upgrade.pid"
	local VIRUS_QUARANTINE_UPGRADE_FLAG="/var/packages/MailPlus-Server/etc/upgrade_virus_quarantine"
	local VIRUS_QUARANTINE_UPGRADE_PID="/var/run/mailplus_server/virus_quarantine_upgrade.pid"

	# upgrade mcp quarantine
	if [ -e ${MCP_QUARANTINE_UPGRADE_FLAG} ] && [ ! -e ${MCP_QUARANTINE_UPGRADE_PID} ]; then
		${MAILPLUS_SERVER_BACKEND_BINARY} "--isClusterInit"
		if [ 0 -eq $? ]; then
			${UPGRADE_QUARANTINE} "mcp" &
		fi
	fi

	# upgrade virus quarantine
	if [ -e ${VIRUS_QUARANTINE_UPGRADE_FLAG} ] && [ ! -e ${VIRUS_QUARANTINE_UPGRADE_PID} ]; then
		${MAILPLUS_SERVER_BACKEND_BINARY} "--isClusterInit"
		if [ 0 -eq $? ]; then
			${UPGRADE_QUARANTINE} "virus" &
		fi
	fi
}

checkStarted()
{
	if [ -f ${MIMEDEFANG_PID} ]; then
		PID=`cat ${MIMEDEFANG_PID}`
		# Add -a to let grep search through binary content
		PROCESS=`cat "/proc/${PID}/cmdline" | grep -a mimedefang`
	fi

	if [ ! -z "${PROCESS}" ]; then
		return 1
	fi
	return 0
}

bin_status()
{
	checkPidFileStatus "mimedefang" "${MIMEDEFANG_PID}"
}

conf_status()
{
	checkConfKey "mimedefang_enable"
	return $?
}

status()
{
	conf_status
	local mimedefang_enable=$?
	bin_status
	local mimedefang_status=$?

	if [ "${SERVICE_NOT_RUNNING}" -eq ${mimedefang_status} ] && [ "${RUNKEY_DISABLE}" -ne "${mimedefang_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	elif [ "${SERVICE_RUNNING}" -eq ${mimedefang_status} ] && [ "${RUNKEY_ENABLE}" -ne "${mimedefang_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	fi
	return ${mimedefang_status}
}

runDaemon()
{
	if isDSM7OrAbove; then
		synosystemctl start ${MULTIPLEXOR_DAEMON}
		synosystemctl start ${MIMEDEFANG_DAEMON}
	else
		initctl start ${MULTIPLEXOR_DAEMON}
		initctl start ${MIMEDEFANG_DAEMON}
	fi

	# FIXME: need to check multiplexor status ?

	local daemon_status=""
	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		status
		daemon_status=$?
		debug_log "daemon_status mimedefang: ${daemon_status}"
		if [ ${daemon_status} -eq "${SERVICE_RUNNING}" ]; then
			#binary is ready
			return ${START_SUCCESSFUL}
		elif [ ${daemon_status} -eq "${SERVICE_DEAD_FPID}" ]; then
			#[FIXME] Maybe we can do something to recover
			return ${START_FAILED}
		else
			sleep "${SLEEP_TIME}"
		fi
	done
	return ${START_FAILED}
}

start()
{
	(
	if flock -x 8; then
		checkAndUpgradeQuarantine

		checkStarted
		if [ 1 -eq $? ] ; then
			echo "mimedefang is already running"
		else
			runDaemon
		fi

		flock -u 8
		rm ${MIMEDEFANG_LOCK}
	fi
	)8> ${MIMEDEFANG_LOCK}
}

checkStopped()
{
	# FIXME: need to check multiplexor status ?

	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		if ! pidof mimedefang > /dev/null; then
			# mimedefang is terminated
			break
		fi
		sleep "${SLEEP_TIME}"
	done

	# if mimedefang still exists, force stop it
	if pidof mimedefang > /dev/null; then
		killall -9 mimedefang
	fi
}

stopDaemon()
{
	if isDSM7OrAbove; then
		synosystemctl stop ${MULTIPLEXOR_DAEMON}
		synosystemctl stop ${MIMEDEFANG_DAEMON}
	else
		initctl stop ${MULTIPLEXOR_DAEMON}
		initctl stop ${MIMEDEFANG_DAEMON}
	fi
}

stop()
{
	(
	if flock -x 8; then
		stopDaemon

		flock -u 8
		rm ${MIMEDEFANG_LOCK}
	fi
	)8> ${MIMEDEFANG_LOCK}
}

restart()
{
	(
	if flock -x 8; then
		stopDaemon
		sleep 1
		runDaemon

		flock -u 8
		rm ${MIMEDEFANG_LOCK}
	fi
	)8> ${MIMEDEFANG_LOCK}
}

reload()
{
	(
	if flock -x 8; then
		checkAndUpgradeQuarantine

		${MD_MX_CTRL} -s ${MULTIPLEXOR_SOCK} reread

		flock -u 8
		rm ${MIMEDEFANG_LOCK}
	fi
	)8> ${MIMEDEFANG_LOCK}
}

composePID()
{
	local pid=$(cat ${MIMEDEFANG_PID} | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
	echo "${pid}"
}

composeScript()
{
	echo "/var/packages/MailPlus-Server/target/scripts/daemon/mimedefang.sh"
}

start_mimedefang() {
	export PERL5LIB
	exec ${MIMEDEFANG_BIN}
}

prestart_mimedefang() {
	rm -f ${MIMEDEFANG_SOCK}
}

poststart_mimedefang() {
	waitTillPidReady "${MIMEDEFANG_PID}"
	local status=$?
	if [ "${SUCCESS}" -eq "${status}" ]; then
		# Register service
		pid=$(cat ${MIMEDEFANG_PID} | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
		opt="pid:${pid}|script:${MIMEDEFANG_SCRIPT_NAME}"

		${BACKEND_BINARY} mailplus_server_command --register --service_name "${MIMEDEFANG_SERVICE_NAME}" --opt "${opt}"
		return ${START_SUCCESSFUL}
	else
		return ${START_FAILED}
	fi
}

poststop_mimedefang() {
	rm -f ${MIMEDEFANG_PID}

	# Deregister service
	${BACKEND_BINARY} mailplus_server_command --deregister --service_name "${MIMEDEFANG_SERVICE_NAME}"
}

start_mimedefang_multiplexor() {
	export PERL5LIB
	exec ${MULTIPLEXOR_BIN}
}

prestart_mimedefang_multiplexor() {
	rm -f ${MULTIPLEXOR_SOCK}
	rm -f ${MULTIPLEXOR_STATUS_SOCK}
	rm -f ${MULTIPLEXOR_NOTIFICATION_SOCK}
}

poststart_mimedefang_multiplexor() {
	waitTillPidReady "${MULTIPLEXOR_PID}"
	local status=$?
	if [ "${SUCCESS}" -eq "${status}" ]; then
		# Register service
		pid=$(cat ${MULTIPLEXOR_PID} | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
		opt="pid:${pid}|script:${MIMEDEFANG_SCRIPT_NAME}"

		${BACKEND_BINARY} mailplus_server_command --register --service_name "${MULTIPLEXOR_SERVICE_NAME}" --opt "${opt}"
		return ${START_SUCCESSFUL}
	else
		return ${START_FAILED}
	fi
}

poststop_mimedefang_multiplexor() {

	rm -f ${MULTIPLEXOR_NOTIFICATION_SOCK}
	rm -f ${MULTIPLEXOR_PID}

	# Deregister service
	${BACKEND_BINARY} mailplus_server_command --deregister --service_name "${MULTIPLEXOR_SERVICE_NAME}"
}

case "$1" in
	start)
		startOperation "mailserver_service_mimedefang"
		;;
	restart)
		restartOperation "mailserver_service_mimedefang"
		;;
	stop)
		stopOperation "mailserver_service_mimedefang"
		;;
	reload)
		reloadOperation "mailserver_service_mimedefang"
		;;
	status)
		status
		;;
	start_mimedefang)
		start_mimedefang
		;;
	prestart_mimedefang)
		prestart_mimedefang
		;;
	poststart_mimedefang)
		poststart_mimedefang
		;;
	poststart_mimedefang)
		poststop_mimedefang
		;;
	start_mimedefang_multiplexor)
		start_mimedefang_multiplexor
		;;
	prestart_mimedefang_multiplexor)
		prestart_mimedefang_multiplexor
		;;
	poststart_mimedefang_multiplexor)
		poststart_mimedefang_multiplexor
		;;
	poststart_mimedefang_multiplexor)
		poststop_mimedefang_multiplexor
		;;
	*)
		echo "Usage: $0 {start|stop|restart|status|start_mimedefang|prestart_mimedefang|poststart_mimedefang|poststop_mimedefang|start_mimedefang_multiplexor|prestart_mimedefang_multiplexor|poststart_mimedefang_multiplexor|poststop_mimedefang_multiplexor}" >&2
		exit 1
		;;
esac
