#!/bin/sh

. /var/packages/MailPlus-Server/target/scripts/daemon/util.sh
. /var/packages/MailPlus-Server/target/backend_hook/hookUtils.conf

DIRECTOR_BIN="${MAIL_SERVER_PKG}/target/sbin/dovecot"
DIRECTOR_CONF="${MAIL_SERVER_PKG}/target/etc/director/dovecot.conf"
DIRECTOR_PID="/var/run/director/master.pid"
DOVEADM_BIN="${MAIL_SERVER_PKG}/target/bin/doveadm"
BACKEND_BIN="${MAIL_SERVER_PKG}/target/bin/syno_mailserver_backend"

MAILPLUS_SERVER_PORT_PATH="/usr/local/etc/services.d/MailPlus-Server.sc"
GET_SECTION_KEY_VALUE="/usr/syno/bin/get_section_key_value"

checkDirectorStatus()
{
	local target_proc_name=$1
	local target_pid_file=$2
	local target_filter=$3
	#echo " checkDirectorStatus ----- ${target_proc_name} ---- ${target_pid_file} ----- ${target_filter} ----- "
	if [ x"`/bin/pidof "${target_proc_name}"`" != x"" ] ; then
		if [ -f "${target_pid_file}" ] ; then
			local pid=`cat "${target_pid_file}"`
			if [ 0 != `ps -I | grep "${target_proc_name}" | grep "${target_filter}" | grep "${pid}" | wc -l` ] ; then
				#echo "I'm running"
				return "${SERVICE_RUNNING}"
			else
				return "${SERVICE_DEAD_FPID}"
			fi
		else
			if [ 0 != `ps -I | grep "${target_proc_name}" | grep -v "grep" | grep "${target_filter}" | wc -l` ] ; then
				#echo "I'm running"
				return "${SERVICE_RUNNING}"
			else
				return "${SERVICE_NOT_RUNNING}"
			fi
		fi
	else
		if [ -f "${target_pid_file}" ] ; then
			return "${SERVICE_DEAD_FPID}"
		else
			#echo "I'm not running"
			return "${SERVICE_NOT_RUNNING}"
		fi
	fi > /dev/null
}

get_port()
{
	local port_section=$1
	echo `${GET_SECTION_KEY_VALUE} "${MAILPLUS_SERVER_PORT_PATH}" "${port_section}" dst.ports | awk -F"/" 'NR == 1 {print $1}'`
}

check_single_port()
{
	local config_key=$1
	local port_section=$2
	local balancer_address=$3

	local port=""
	port=$(get_port "${port_section}")
	if [ -z "${port}" ] ; then
		return "${SERVICE_UNKNOWN}"
	fi

	checkConfKey "${config_key}"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		checkPortUseStatus "dovecot" "${port}" "${balancer_address}"
		local status=$?
		if [ "${SERVICE_RUNNING}" -ne ${status} ] ; then
			return ${status}
		fi
	else
		checkPortUseStatus "dovecot" "${port}" "${balancer_address}"
		local status=$?
		if [ "${PORT_NOT_LISTEN}" -ne ${status} ] ; then
			return "${PORT_LISTEN}"
		fi
	fi

	return "${SERVICE_RUNNING}"
}

bin_status()
{
	checkDirectorStatus "dovecot" "${DIRECTOR_PID}" "director"
	return $?
}

conf_status()
{
	if ! isLoadBalancer; then
		return "${RUNKEY_DISABLE}"
	fi
	checkConfKey "imap_enabled"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		return "${RUNKEY_ENABLE}"
	fi
	checkConfKey "imaps_enabled"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		return "${RUNKEY_ENABLE}"
	fi
	checkConfKey "pop3_enabled"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		return "${RUNKEY_ENABLE}"
	fi
	checkConfKey "pop3s_enabled"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		return "${RUNKEY_ENABLE}"
	fi
	return "${RUNKEY_DISABLE}"
}

port_status()
{
	local balancerAddress="$(${BACKEND_BIN} --getConfKeyVal "balancer_address")"
	if [ x"" == x"${balancerAddress}" ] ; then
		return "${SERVICE_UNKNOWN}"
	fi

	check_single_port "imap_enabled" "mail_imap" "${balancerAddress}"
	local imap_status=$?
	if [ "${SERVICE_RUNNING}" -ne ${imap_status} ]; then
		return ${imap_status}
	fi

	check_single_port "imaps_enabled" "mail_imaps" "${balancerAddress}"
	local imaps_status=$?
	if [ "${SERVICE_RUNNING}" -ne ${imaps_status} ]; then
		return ${imaps_status}
	fi

	check_single_port "pop3_enabled" "mail_pop3" "${balancerAddress}"
	local pop3_status=$?
	if [ "${SERVICE_RUNNING}" -ne ${pop3_status} ]; then
		return ${pop3_status}
	fi

	check_single_port "pop3s_enabled" "mail_pop3s" "${balancerAddress}"
	local pop3s_status=$?
	if [ "${SERVICE_RUNNING}" -ne ${pop3s_status} ]; then
		return ${pop3s_status}
	fi

	return "${SERVICE_RUNNING}"
}

status()
{
	conf_status
	local director_enable=$?
	bin_status
	local director_status=$?

	if [ "${SERVICE_NOT_RUNNING}" -eq ${director_status} ] && [ "${RUNKEY_DISABLE}" -ne "${director_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	elif [ "${SERVICE_RUNNING}" -eq ${director_status} ] && [ "${RUNKEY_ENABLE}" -ne "${director_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	fi
	if [ "${SERVICE_RUNNING}" -ne ${director_status} ]; then
		return ${director_status};
	fi

	port_status
	local director_port=$?
	return ${director_port}
}

setBackendWeight()
{
	local mailServers="$(${MAILPLUS_SERVER_BACKEND_BINARY} --getAliveServ "mailserver_service_dovecot")"
	local weightList="$(${MAILPLUS_SERVER_BACKEND_BINARY} --getConfKeyVal "mailer_weight_list")"
	local mailServerWeight=""
	local serv=""
	for serv in ${mailServers}; do
		mailServerWeight=$(getWeight "${weightList}" "${serv}")
		if [ "${mailServerWeight}" = "" ]; then
			mailServerWeight=1
		fi
		${DOVEADM_BIN} -c "${DIRECTOR_CONF}" director add "${serv}" "${mailServerWeight}"
	done
}

start() {
	local daemon_status=""
	if ! ${DIRECTOR_BIN} -c "${DIRECTOR_CONF}"; then
		err_log "Cannot start director"
		return ${START_FAILED}
	fi

	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		status
		daemon_status=$?
		debug_log "daemon_status director: ${daemon_status}"
		if [ ${daemon_status} -eq "${SERVICE_RUNNING}" ]; then
			#binary is ready
			setBackendWeight
			return ${START_SUCCESSFUL}
		elif [ ${daemon_status} -eq "${SERVICE_DEAD_FPID}" ]; then
			#[FIXME] Maybe we can do something to recover
			return ${START_FAILED}
		else
			sleep "${SLEEP_TIME}"
		fi
	done
	return ${START_FAILED}
}

checkStopped()
{
	waitTillPidRemoved "${DIRECTOR_PID}"
	local status=$?
	if [ "${FAIL}" -eq "${status}" ]; then
		# Force stop
		local pid=$(cat ${DIRECTOR_PID})
		kill -9 "${pid}" &> /dev/null
	fi
}

stop(){
	[ -e ${DIRECTOR_PID} ] || return 0
	local pid=$(cat ${DIRECTOR_PID})
	kill "$pid" &> /dev/null
	checkStopped
}

restart()
{
	stop
	sleep 1
	start
}

reload()
{
	if [ -e ${DIRECTOR_PID} ]; then
		${DOVEADM_BIN} -c "${DIRECTOR_CONF}" reload
	else
		start
	fi
}

composePort()
{
	local ports=()
	local port=""

	checkConfKey "imap_enabled"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		port=$(get_port "mail_imap")
		if [ x"${port}" != x"" ]; then
			ports[${#ports[*]}]=${port}
		else
			err_log "ports imap get fail"
		fi
	fi

	checkConfKey "imaps_enabled"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		port=$(get_port "mail_imaps")
		if [ x"${port}" != x"" ]; then
			ports[${#ports[*]}]=${port}
		else
			err_log "ports imaps get fail"
		fi
	fi

	checkConfKey "pop3_enabled"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		port=$(get_port "mail_pop3")
		if [ x"${port}" != x"" ]; then
			ports[${#ports[*]}]=${port}
		else
			err_log "ports pop3 get fail"
		fi
	fi

	checkConfKey "pop3s_enabled"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		port=$(get_port "mail_pop3s")
		if [ x"${port}" != x"" ]; then
			ports[${#ports[*]}]=${port}
		else
			err_log "ports pop3s get fail"
		fi
	fi
	echo "$(IFS=, ; echo "${ports[*]}")"
}

composeIP()
{
	local balancer_address="$(${BACKEND_BIN} --getConfKeyVal "balancer_address")"
	if [ x"" == x"${balancer_address}" ] ; then
		err_log "balancer_address not exist"
		return
	fi
	echo "${balancer_address}"
}

composeProtocol()
{
	echo "tcp"
}

composePID()
{
	local pid=$(cat ${DIRECTOR_PID} | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
	echo "${pid}"
}

composeScript()
{
	echo "/var/packages/MailPlus-Server/target/scripts/daemon/director.sh"
}

case "$1" in
	start)
		startOperation "mailserver_service_director"
		;;
	restart)
		restartOperation "mailserver_service_director"
		;;
	stop)
		stopOperation "mailserver_service_director"
		;;
	reload)
		reloadOperation "mailserver_service_director"
		;;
	status)
		status
		;;
	*)
		echo "Usage: $0 {start|stop|reload|restart|status}" >&2
		exit 1
		;;
esac
