#!/bin/bash

# Example attachment decoder script. The attachment comes from stdin, and
# the script is expected to output UTF-8 data to stdout. (If the output isn't
# UTF-8, everything except valid UTF-8 sequences are dropped from it.)

# The attachment decoding is enabled by setting:
#
# plugin {
#   fts_decoder = decode2text
# }
# service decode2text {
#   executable = script /var/packages/MailPlus-Server/target/libexec/dovecot/syno_decode2text.sh
#   user = admin
#   unix_listener decode2text {
#     mode = 0666
#   }
# }

content_type="$1"
filename="$2"

# The second parameter is the format's filename extension, which is used when
# found from a filename of application/octet-stream. You can also add more
# extensions by giving more parameters.
formats='application/vnd.ms-excel xls xlm xla xlc xlt xlw
application/ms-excel xls
application/x-msexcel xls
application/pdf pdf
application/x-pdf pdf
application/vnd.ms-powerpoint ppt pps pot
application/mspowerpoint ppt
application/msword doc dot
application/vnd.oasis.opendocument.chart odc
application/vnd.oasis.opendocument.chart-template otc
application/vnd.oasis.opendocument.formula odf
application/vnd.oasis.opendocument.formula-template otf
application/vnd.oasis.opendocument.graphics odg
application/vnd.oasis.opendocument.graphics-template otg
application/vnd.oasis.opendocument.image odi
application/vnd.oasis.opendocument.image-template oti
application/vnd.oasis.opendocument.presentation odp
application/vnd.oasis.opendocument.presentation-template otp
application/vnd.oasis.opendocument.spreadsheet ods
application/vnd.oasis.opendocument.spreadsheet-template ots
application/vnd.oasis.opendocument.text odt
application/vnd.oasis.opendocument.text-master otm
application/vnd.oasis.opendocument.text-template ott
application/vnd.oasis.opendocument.text-web oth
application/vnd.ms-word.document.macroEnabled.12 docm
application/vnd.openxmlformats-officedocument.wordprocessingml.document docx
application/vnd.ms-word.template.macroEnabled.12 dotm
application/vnd.openxmlformats-officedocument.wordprocessingml.template dotx
application/vnd.ms-powerpoint.slideshow.macroEnabled.12 ppsm
application/vnd.openxmlformats-officedocument.presentationml.slideshow ppsx
application/vnd.ms-powerpoint.presentation.macroEnabled.12 pptm
application/vnd.openxmlformats-officedocument.presentationml.presentation pptx
application/vnd.ms-excel.sheet.binary.macroEnabled.12 xlsb
application/vnd.ms-excel.sheet.macroEnabled.12 xlsm
application/vnd.openxmlformats-officedocument.spreadsheetml.sheet xlsx
application/vnd.ms-xpsdocument xps
application/x-rar-compressed rar
application/x-7z-compressed 7z
application/x-7z-compressed s7z
text/vcard vcf
text/calendar ics
image/gif gif
image/png png
image/x-png png
image/jpeg jpg
image/jpg jpg
image/pjpeg jpg
image/tiff tif
message/rfc822 eml
text/x-mail eml
'

if [ "$content_type" = "" ]; then
	echo "$formats"
	exit 0
fi

fmt=$(echo "$formats" | grep -w "^$content_type" | cut -d ' ' -f 2)
if [ x"$fmt" = x"" ] && [ x"$filename" = x"" ]; then
	echo "Not supported (Content-Type: '$content_type', filename: '$filename')" >&2
	exit 1
fi

# most decoders can't handle stdin directly, so write the attachment
# to a temp file
default_trap()
{
	trap 'rm -f "$path" "$new_path"' 0 1 2 3 14 15
}

path=$(mktemp)
new_path=
default_trap
cat > "$path"

get_file_extension() {
    local file_basename
	file_basename=$(basename "$1")
    local file_ext=
    local tmp_ext=
    while [[ $file_basename = ?*.* &&
        ( ${file_basename##*.} = [A-Za-z]* || ${file_basename##*.} = 7z ) ]]
    do
        tmp_ext=.${file_basename##*.}$file_ext
        if (( ${#tmp_ext} > 10 )); then
            break
        fi
        file_ext=$tmp_ext
        file_basename=${file_basename%.*}
    done
    echo "$file_ext"
}

wait_timeout() {
	trap 'kill -9 -"$childpid"; rm -f "$path" "$new_path"' 0 1 2 3 14 15
	wait "$childpid"
}

LANG=en_US.UTF-8
export LANG
case "$filename" in
	*.*)
		extension=$(get_file_extension "${filename}")
		new_path="${path}${extension}"
		if ! mv "$path" "$new_path" 2>/dev/null; then
			echo "failed to rename tmp file" >&2
			exit 1
		fi
		set -m
		(/usr/syno/bin/synocontentextract -d -i "$new_path" -m | /usr/bin/jq -Mr ".SYNOMDAuthors + \"\n\" + .SYNOMDTitle + \"\n\" + .SYNOMDHeadline + \"\n\" + .SYNOMDTextContent + \"\n\" + .SYNOMDKeywords // empty";echo "$filename") 2>/dev/null&
		childpid=$!
		set +m
		wait_timeout 2>/dev/null
		default_trap
		;;
	*)
		if [ x"$fmt" != x"" ]; then
			new_path="$path.$fmt"
			if ! mv "$path" "$new_path" 2>/dev/null; then
				echo "failed to rename tmp file without extension" >&2
				exit 1
			fi
			set -m
			(/usr/syno/bin/synocontentextract -d -i "$new_path" -m | /usr/bin/jq -Mr ".SYNOMDAuthors + \"\n\" + .SYNOMDTitle + \"\n\" + .SYNOMDHeadline + \"\n\" + .SYNOMDTextContent + \"\n\" + .SYNOMDKeywords // empty";echo "$filename") 2>/dev/null&
			childpid=$!
			set +m
			wait_timeout 2>/dev/null
			default_trap
		else
			echo "Buggy decoder script: $fmt not handled" >&2
			exit 1
		fi
		;;
esac

exit 0
