from _Define import *
import os
MAILPLUS_SERVER_BACKEND_BINARY = "/var/packages/MailPlus-Server/target/bin/syno_mailserver_backend"

class Singleton(type):
    _instances = {}
    def __call__(cls, *args, **kwargs):
        if cls not in cls._instances:
            cls._instances[cls] = super(Singleton, cls).__call__(*args, **kwargs)
        return cls._instances[cls]

class SyslogClass(object):
    __metaclass__ = Singleton #python 2 only
    def __init__(self):
        import logging.handlers, socket
        self.logger = logging.getLogger('MailPlus-Server')
        hdlr = logging.handlers.SysLogHandler(address='/var/run/syslog', facility='mail', socktype=socket.SOCK_STREAM)
        formatter = logging.Formatter('%(name)s: %(message)s')
        hdlr.formatter = formatter
        self.logger.addHandler(hdlr)
    def log(self, log_type, log_str):
        self.logger.log(log_type, log_str)

class BackendAPI(object):
    def registerServ(self, serviceName, opt = ""):
        cmd = '%s mailplus_server_command --register --service_name "%s" --opt "%s"' % (BACKEND_BINARY, serviceName, opt)
        return executeCommand(cmd, False)
    def deRegisterServ(self, serviceName):
        cmd = '%s mailplus_server_command --deregister --service_name "%s"' % (BACKEND_BINARY, serviceName)
        return executeCommand(cmd, False)
    def getTree(self, tree):
        import json
        cmd = '%s --getTree "%s"' % (MAILPLUS_SERVER_BACKEND_BINARY, tree)
        result, _ = executeCommand(cmd, True)
        output = {}
        try:
            if result:
                output = json.loads(result)
        except Exception as e:
            SYSLOG(LOG_ERR, "Failed to ls %s" % e)
        return output
    def mailconfGet(self, key):
        cmd = '%s --getConfKeyVal "%s"' % (MAILPLUS_SERVER_BACKEND_BINARY, key)
        output, _ = executeCommand(cmd, True)
        return output
    def mailconfSet(self, key, val):
        cmd = '%s --setConfKeyVal "%s" "%s"' % (MAILPLUS_SERVER_BACKEND_BINARY, key, val)
        return executeCommand(cmd, False)
    def mailconfDirectSet(self, key, val):
        cmd = '%s --setDirectConfKeyVal "%s" "%s"' % (MAILPLUS_SERVER_BACKEND_BINARY, key, val)
        return executeCommand(cmd, False)
    def removeTask(self, key):
        cmd = '%s --removeTask "%s"' % (MAILPLUS_SERVER_BACKEND_BINARY, key)
        return executeCommand(cmd, False)
    def setFile(self, key, filePath):
        cmd = '%s --setFile "%s" "%s"' % (MAILPLUS_SERVER_BACKEND_BINARY, key, filePath)
        return executeCommand(cmd, False)
    def syncFile(self, filePath, *args):
        if args:
            cmd = '%s --syncFile "%s" %s' % (MAILPLUS_SERVER_BACKEND_BINARY, filePath, " ".join(args))
        else:
            cmd = '%s --syncFile "%s"' % (MAILPLUS_SERVER_BACKEND_BINARY, filePath)
        return executeCommand(cmd, False)
    def uploadFile(self, key):
        cmd = '%s --uploadFile "%s"' % (MAILPLUS_SERVER_BACKEND_BINARY, key)
        return executeCommand(cmd, False)
    def fileContentDump(self, uploadKey):
        cmd = '%s --fileContentDump "%s"' % (MAILPLUS_SERVER_BACKEND_BINARY, uploadKey)
        return executeCommand(cmd, False)
    def mailDirGet(self, username):
        cmd = '%s --getMailDir "%s"' % (MAILPLUS_SERVER_BACKEND_BINARY, username)
        output, _ = executeCommand(cmd, True)
        return output
    def isLoadBalancer(self):
        cmd = '%s --isBalancerMaster' % (MAILPLUS_SERVER_BACKEND_BINARY)
        (output, result) = executeCommand(cmd, False)
        return result
    def getMasterBalancer(self):
        cmd = '%s --getMasterBalancer' % (MAILPLUS_SERVER_BACKEND_BINARY)
        output, _ = executeCommand(cmd, True)
        return output
    def getPeersNum(self):
        cmd = '%s --getPeersNum' % (MAILPLUS_SERVER_BACKEND_BINARY)
        output, _ = executeCommand(cmd, True)
        return int(output)
    def getHostID(self):
        cmd = '%s --getHostID' % (MAILPLUS_SERVER_BACKEND_BINARY)
        output, _ = executeCommand(cmd, True)
        return output
    def getHostIP(self):
        cmd = '%s --getHostIP' % (MAILPLUS_SERVER_BACKEND_BINARY)
        output, _ = executeCommand(cmd, True)
        return output
    def getHostIF(self):
        cmd = '%s --getHostIF' % (MAILPLUS_SERVER_BACKEND_BINARY)
        output, _ = executeCommand(cmd, True)
        return output
    def getAliveServ(self, serv):
        cmd = '%s --getAliveServ "%s"' % (MAILPLUS_SERVER_BACKEND_BINARY, serv)
        output, _ = executeCommand(cmd, True)
        return output
    def idToIP(self, nodeID):
        cmd = '%s --idToIP "%s"' % (MAILPLUS_SERVER_BACKEND_BINARY, nodeID)
        output, _ = executeCommand(cmd, True)
        return output
    def setServSyncTag(self, serv, servID, tag, isDone):
        if isDone:
            cmd = '%s --setServSyncTagDone %s %s %s' % (MAILPLUS_SERVER_BACKEND_BINARY, serv, servID, tag)
        else:
            cmd = '%s --setServSyncTag %s %s %s' % (MAILPLUS_SERVER_BACKEND_BINARY, serv, servID, tag)
        return executeCommand(cmd, False)
    def getActiveUserList(self):
        cmd = '%s --getActiveUserList' % (MAILPLUS_SERVER_BACKEND_BINARY)
        output, _ = executeCommand(cmd, True)
        return output

class PrivilegePromote(object):
    def __init__(self):
        self._euid = os.geteuid()
        self._egid = os.getegid()
    def __enter__(self):
        os.seteuid(0)
        os.setegid(0)
    def __exit__(self, type, value, traceback):
        os.setegid(self._egid)
        os.seteuid(self._euid)

def executeCommand(args, needOutput = True):
    from subprocess import Popen, STDOUT, PIPE
    try:
        process = Popen(args.encode('utf8'), shell=True, stdin=PIPE, stdout=PIPE, stderr=STDOUT)
        if needOutput:
            output = process.stdout.read().strip()
            process.communicate()
            ret = process.returncode == 0
            return output, ret
        else:
            process.communicate()
            return "", process.returncode == 0
    except Exception as e:
        SYSLOG(LOG_ERR, "Exception: %s" % e)
def SYSLOG(log_type, log_str):
    import inspect
    logger = SyslogClass()

    if LOG_DEBUG < log_type or ENABLE_DEBUG :
        #FIXME only allow at most warn severity level for now, to prevent log center send notification
        if LOG_WARNING < log_type:
			log_type = LOG_WARNING
        callerframerecord = inspect.stack()[1]
        frame = callerframerecord[0]
        info = inspect.getframeinfo(frame)
        logger.log(log_type, info.filename+':'+ str(info.lineno) + ':' + info.function+'(): '+ log_str)

def tracelog(strTrace):
    log_lines = strTrace.split('\n')
    for line in log_lines:
        if len(line):
            SYSLOG(LOG_ERR, "%s" % line)
def getHostID():
    eAPI = BackendAPI()
    return eAPI.getHostID()
def execWebAPI(api, method, version, **kwargs):
    """
    Send web API to localhost by /usr/syno/bin/synowebapi and return response (in Json) as Python dict.
    kwargs are parameters to web API. For example, following creates a share via web API.

    resp = execWebAPI("SYNO.Core.Share", "create", 1,
        name="public",
        shareinfo={
            "name": "public",
            "vol_path": "/volume1",
            "desc": "Public Shared Folder"
        }
    )
    """
    import json
    import subprocess

    cmd = ["/usr/syno/bin/synowebapi", "--exec"]
    cmd.append("api=" + api)
    cmd.append("method=" + method)
    cmd.append("version=" + str(version))
    for key, value in kwargs.items():
        cmd.append(key + "=" + json.dumps(value))

    try:
        with open("/dev/null", "w") as null_f:
            raw_resp = subprocess.check_output(cmd, stderr=null_f)
        return json.loads(raw_resp)
    except ValueError as e:
        SYSLOG(syslog.LOG_ERR, "Failed to parse web API response (" + str(e) + ")")
    except Exception as e:
        SYSLOG(syslog.LOG_ERR, "Failed to executeCommand web API (" + str(e) + ")")
    return None

def dumpKeyValAsFile(path, keyValDict):
    import fcntl
    try:
        with open(path, 'w') as genFile:
            fcntl.flock(genFile, fcntl.LOCK_EX | fcntl.LOCK_NB)
            if not keyValDict:
                return
            for key, value in keyValDict.items():
                if key == "":
                    continue
                genFile.write(str(key) + " " + str(value) + "\n")
    except Exception as e:
        SYSLOG(LOG_ERR, "gen user diction restriciton file error (" + str(e) + ")")

def clusterAlive():
    '''
        Call cluster script to get status
    '''
    from subprocess import Popen

    with open("/dev/null", "w") as null_f:
        p = Popen([CLUSTER_SCRIPT, "status"], stdout=null_f, stderr=null_f, stdin=null_f)
    p.communicate()
    return p.returncode == 0
def parseFileKey(cAPI, key):
    import json
    '''
    Format is json, reference to lib/mailserver_cluster.cpp setFile()
    '''
    val = cAPI.mailconfGet(key)
    try:
        d = json.loads(val)
        if not d.has_key('dstFilePath'):
            SYSLOG(LOG_ERR, "val doesn't has key: dstFilePath")
            return "", "", ""

    except Exception as e:
        SYSLOG(LOG_ERR, "Failed to transfer json from value %s, error: %s" % (val, e))
        return "", "", ""

    return d["dstFilePath"], d["fileContentKey"], d["senderIP"]
def isChange(compareKey, envKey, prefixMode):
    changeKeys = ""
    with open(ENV_OUTPUT_FILE, "r") as fp:
        for line in fp:
            s = line.split("=")
            key = s[0]
            if key == envKey:
                ## Remove "
                value = s[1]
                changeKeys = value[1:-1]
                break

    for _ in changeKeys.split(" "):
        if prefixMode and _.startswith(compareKey):
            return True
        elif _ == compareKey:
            return True
    return False
def isKeyChange(key):
    return isChange(key, ENV_TRIGGER_EVENTS_KEY, False)
def isTreeChange(key):
    return isChange(key, ENV_TRIGGER_TREE_KEY, False)
def isPrefixKeyChange(prefixKey):
    return isChange(prefixKey, ENV_TRIGGER_EVENTS_KEY, True)

class HAProxyConfigGenerator(object):
    __HAPROXY_GLOBAL_SETTING="""
global
	tune.ssl.default-dh-param 2048
"""
    __HAPROXY_TEMPLATE="""
frontend ft_{__NAME__}
	bind 0.0.0.0:{__HAPROXY_PORT__} {__OPT__}
	mode tcp
	no option http-server-close
	timeout client 1m
	log global
	default_backend bk_{__NAME__}
backend bk_{__NAME__}
	mode tcp
	no option http-server-close
	log global
	timeout server 1m
	timeout connect 5s
{__SERVER__}
"""
    __SERVER_TEMPLATE = "	server postfix-host{INDEX} {IP}:{POSTFIX_PORT} {SSLOPT} send-proxy weight {WEIGHT}"
    __POSTFIX_TLS_TEMPLATE = "/usr/syno/etc/security-profile/tls-profile/config/postfix.conf"

    __ConfigPath = "/var/packages/MailPlus-Server/target/etc/synoMailPlusServerHaproxy.cfg"
    __ConfigPathBak = "/var/packages/MailPlus-Server/target/etc/synoMailPlusServerHaproxy.cfg.bak"
    __PostfixSmtpPort = "10025"
    __PostfixSmtpsPort = "10465"
    __PostfixSmtpTlsPort = "10587"

    def __init__(self):
        self.__hasBackup = False
        self.__configPathTmp = ""
        self.__eAPI = BackendAPI()

    def __del__(self):
        try:
            if (self.__hasBackup):
                os.remove(self.__ConfigPathBak)
            if (self.__configPathTmp != ""):
                os.remove(self.__configPathTmp)
        except:
            pass

    def __genGlobalSection(self):
        import re

        section = "global\n"
        cipherList = {}
        optionList = {
            'low': 'no-tls-tickets',
            'medium': 'no-sslv3 no-tls-tickets',
            'high': 'no-sslv3 no-tlsv10 no-tlsv11 no-tls-tickets'
        }

        if not os.path.isfile(self.__POSTFIX_TLS_TEMPLATE):
            return self.__HAPROXY_GLOBAL_SETTING

        with open(self.__POSTFIX_TLS_TEMPLATE, "r") as f:
            for line in f.readlines():
                m = re.search(r"tls_(.*)_cipherlist *= *(.*)", line)
                if m:
                    cipherList[m.group(1)] = m.group(2)
                    continue

                m = re.search(r"smtpd_tls_ciphers *= *(.*)", line)
                if m:
                    section += "	ssl-default-bind-ciphers " + cipherList[m.group(1)] + "\n"
                    section += "	ssl-default-bind-options " + optionList[m.group(1)] + "\n"
                    section += "	ssl-default-server-ciphers " + cipherList[m.group(1)] + "\n"
                    section += "	ssl-default-server-options " + optionList[m.group(1)] + "\n"
                    continue

                m = re.search(r"smtpd_tls_dh1024_param_file *=.*dh(.*)\.pem", line)
                if m:
                    section += "	tune.ssl.default-dh-param " + m.group(1) + "\n"
                    continue

        return section

    def __parseWeight(self, weightList):
        weightMap = {}
        for _ in weightList.split(' '):
            if not _:
                continue
            nodeID = _.split(":")[0]
            weight = _.split(":")[1]
            weightMap[self.__eAPI.idToIP(nodeID)] = weight
        return weightMap

    def __genServiceSections(self, name, weightMap, postfixPort, haproxyPort, server, opt, sslOpt):
        args={}
        args["__NAME__"] = name
        args["__HAPROXY_PORT__"] = haproxyPort
        args["__OPT__"] = opt

        serverList = []
        for ip in server.split(' '):
            servArg = {}
            if not ip:
                continue

            if weightMap.has_key(ip):
                weight = weightMap[ip]
            else:
                weight = 1

            servArg["INDEX"] = str(len(serverList))
            if ip == self.__eAPI.getHostIP():
                servArg["IP"] = "127.0.0.1"
            else:
                servArg["IP"] = ip
            servArg["POSTFIX_PORT"] = postfixPort
            servArg["SSLOPT"] = sslOpt
            servArg["WEIGHT"] = weight
            serverList.append(self.__SERVER_TEMPLATE.format(**servArg))

        args["__SERVER__"] = "\n".join(serverList)

        return self.__HAPROXY_TEMPLATE.format(**args)

    def genConfig(self):
        import tempfile
        import shutil

        mailServers = self.__eAPI.getAliveServ("mailserver_service_postfix")
        weightList = self.__eAPI.mailconfGet("mailer_weight_list")
        smtpSSLEnable = self.__eAPI.mailconfGet("smtp_ssl_enabled")
        smtpTLSEnable = self.__eAPI.mailconfGet("smtp_tls_enabled")

        weightMap = self.__parseWeight(weightList)

        (fd, self.__configPathTmp) = tempfile.mkstemp('', os.path.basename(self.__ConfigPath), os.path.dirname(self.__ConfigPath))
        with os.fdopen(fd, "w") as wp:
            wp.write(self.__genGlobalSection())

            smtpPort = self.__eAPI.mailconfGet("smtp_port")
            if not smtpPort:
                smtpPort = "25"
            wp.write(self.__genServiceSections("postfix", weightMap, self.__PostfixSmtpPort, smtpPort, mailServers, "", ""))

            if "yes" == smtpSSLEnable:
                smtpSSLPort = self.__eAPI.mailconfGet("smtps_port")
                if not smtpSSLPort:
                    smtpSSLPort = "465"
                wp.write(self.__genServiceSections("ssl_postfix", weightMap, self.__PostfixSmtpsPort, smtpSSLPort, mailServers,
                                        "ssl crt /usr/local/etc/certificate/MailPlus-Server/postfix/privkey_fullchain.pem", "ssl verify none"))

            if "yes" == smtpTLSEnable:
                smtpTLSPort = self.__eAPI.mailconfGet("smtp_tls_port")
                if not smtpTLSPort:
                    smtpTLSPort = "587"
                wp.write(self.__genServiceSections("tls_postfix", weightMap, self.__PostfixSmtpTlsPort, smtpTLSPort, mailServers, "", ""))

        if os.path.isfile(self.__ConfigPath):
            shutil.move(self.__ConfigPath, self.__ConfigPathBak)
            self.__hasBackup = True
        shutil.move(self.__configPathTmp, self.__ConfigPath)

        return True

    def rollbackConfig(self):
        import shutil

        if not self.__hasBackup:
            return False
        try:
            shutil.move(self.__ConfigPathBak, self.__ConfigPath)
        except:
            return False

        return True
