# Local status cache virtual folder: SQLite backend
# Copyright (C) 2009-2016 Stewart Smith and contributors.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import os

from .LocalStatusSQLite import LocalStatusSQLiteFolder
from offlineimap import OfflineImapError
import threading

def remove_file_dir(path):
    import shutil
    try:
        if os.path.exists(path):
            if os.path.isdir(path):
                shutil.rmtree(path)
            else:
                os.remove(path)
    except OSError as e:
        raise OfflineImapError('Failed to delete the target [{0}]'.format(path) , OfflineImapError.ERROR.FOLDER)

class LocalStatusSynoSQLiteFolder(LocalStatusSQLiteFolder):
    """LocalStatus backend implemented with an SQLite database
    When trying to sync a folder it would trigger 'openfiles' method.
    And we then touch a file to indicating the folder is being processed
    When the sync is finished, it would trigger 'closefiles'. And we then
    delete the file."""

    def __init__(self, name, repository):
        super(LocalStatusSynoSQLiteFolder, self).__init__(name, repository)

        #Added by Synology. Rearrange logs. Record any message processing error
        self.message_error_num = 0
        self.count_message_error_lock = threading.Lock()

        #add by Synology. Add a file to indicate it is currently processing this folder
        self.processing_root = repository.processing_root
        if not os.path.exists(self.processing_root):
            os.makedirs(self.processing_root, 0700)
        dirname = os.path.dirname(self.processing_root)
        if not os.path.isdir(self.processing_root):
            raise UserWarning("processing root path '%s' is not a directory."%
                dirname)

        self.curr_processing_filename = os.path.join(self.processing_root, self.getfolderbasename())

        remove_file_dir(self.curr_processing_filename)

    def openfiles(self):
        # Make sure sqlite is in multithreading SERIALIZE mode.
        super(LocalStatusSynoSQLiteFolder, self).openfiles()

        #add by Synology. Add a file to indicate it is currently processing this folder
        with open(self.curr_processing_filename, 'a'):
            #touch the file
            pass

    def closefiles(self):
        super(LocalStatusSynoSQLiteFolder, self).closefiles()

        #add by Synology. Add a file to indicate it is currently processing this folder
        #remove the file once finish processing the folder
        remove_file_dir(self.curr_processing_filename)

    #Added by Synology. Rearrange logs. Record any message processing error
    def add_message_error_num_by_one(self):
        with self.count_message_error_lock:
            self.message_error_num = self.message_error_num + 1
