#!/bin/sh

. /var/packages/MailPlus-Server/target/scripts/daemon/util.sh

RSPAMD_REDIS_PID="/var/run/mailplus_server/rspamd_redis.pid"
RSPAMD_REDIS_PORT="8505"
RSPAMD_REDIS_CONF="${MAIL_SERVER_PKG}/etc/rspamd_redis/redis.conf"
RSPAMD_REDIS_SERVER="${MAIL_SERVER_PKG}/target/usr/bin/redis-server"
RSPAMD_REDIS_CLIENT="${MAIL_SERVER_PKG}/target/usr/bin/redis-cli"
SZF_RSPAMD_REDIS_LOCK="/tmp/syno_rspamd_redis.lock"

BACKEND_CONF="${MAIL_SERVER_PKG}/etc/backend.conf"

checkStarted()
{
	if [ -f ${RSPAMD_REDIS_PID} ]; then
		PID=`cat ${RSPAMD_REDIS_PID}`
		PROCESS=`cat "/proc/${PID}/cmdline" | grep redis`
	fi

	if [ ! -z "${PROCESS}" ]; then
		return 1
	fi
	return 0
}

bin_status()
{
	checkPidFileStatus "redis-server *:${RSPAMD_REDIS_PORT}" "${RSPAMD_REDIS_PID}"
}

conf_status()
{
	checkConfKey "anti_virus_enable"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		return "${RUNKEY_ENABLE}"
	fi
	checkConfKey "spam_enable"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		return "${RUNKEY_ENABLE}"
	fi
	checkConfKey "mcp_enable"
	if [ "${RUNKEY_ENABLE}" -eq $? ]; then
		return "${RUNKEY_ENABLE}"
	fi
	return "${RUNKEY_DISABLE}"
}

status()
{
	conf_status
	local rspamd_redis_enable=$?
	bin_status
	local rspamd_redis_status=$?

	if [ "${SERVICE_NOT_RUNNING}" -eq ${rspamd_redis_status} ] && [ "${RUNKEY_DISABLE}" -ne "${rspamd_redis_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	elif [ "${SERVICE_RUNNING}" -eq ${rspamd_redis_status} ] && [ "${RUNKEY_ENABLE}" -ne "${rspamd_redis_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	fi
	return ${rspamd_redis_status}
}

runDaemon()
{
	${RSPAMD_REDIS_SERVER} ${RSPAMD_REDIS_CONF}

	local daemon_status=""
	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		status
		daemon_status=$?
		debug_log "daemon_status rspamd_redis: ${daemon_status}"
		if [ ${daemon_status} -eq "${SERVICE_RUNNING}" ]; then
			#binary is ready
			return ${START_SUCCESSFUL}
		elif [ ${daemon_status} -eq "${SERVICE_DEAD_FPID}" ]; then
			#[FIXME] Maybe we can do something to recover
			return ${START_FAILED}
		else
			sleep "${SLEEP_TIME}"
		fi
	done
	return ${START_FAILED}
}

start()
{
	(
	if flock -x 8; then
		checkStarted
		if [ 1 -eq $? ] ; then
			echo "rspamd_redis is already running"
		else
			runDaemon
		fi

		flock -u 8
		rm ${SZF_RSPAMD_REDIS_LOCK}
	fi
	)8> ${SZF_RSPAMD_REDIS_LOCK}
}

# Differentiate rspamd redis server from the main redis server
getPid()
{
	ps -C 'redis-server' -o pid= -o args= | grep "${RSPAMD_REDIS_PORT}" | awk '{print $1}'
}

checkStopped()
{
	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		if [ -z "$(getPid)" ]; then
			# rspamd_redis is terminated
			break
		fi
		sleep "${SLEEP_TIME}"
	done

	# if rspamd_redis still exists, force stop it
	if [ ! -z "$(getPid)" ]; then
		kill -9 "$(getPid)" &> /dev/null
	fi
}

stopDaemon()
{
	# Delete all rspamd cache
	local password="$(jq -r '.password' "${BACKEND_CONF}")"
	"${RSPAMD_REDIS_CLIENT}" -a "${password}" -p "${RSPAMD_REDIS_PORT}" keys 'rspamd/cache/*' |
		sed 's/^/del /' |
		"${RSPAMD_REDIS_CLIENT}" -a "${password}" -p "${RSPAMD_REDIS_PORT}" > /dev/null

	kill `cat ${RSPAMD_REDIS_PID}` &> /dev/null
	checkStopped
}

stop()
{
	(
	if flock -x 8; then
		stopDaemon
		rm ${RSPAMD_REDIS_PID} &> /dev/null

		flock -u 8
		rm ${SZF_RSPAMD_REDIS_LOCK}
	fi
	)8> ${SZF_RSPAMD_REDIS_LOCK}
}

restart()
{
	(
	if flock -x 8; then
		stopDaemon
		sleep 1
		runDaemon

		flock -u 8
		rm ${SZF_RSPAMD_REDIS_LOCK}
	fi
	)8> ${SZF_RSPAMD_REDIS_LOCK}
}

reload()
{
	if [ -f ${RSPAMD_REDIS_PID} ]; then
		kill -s HUP `cat ${RSPAMD_REDIS_PID}`
	else
		kill -s HUP "$(getpid)"
	fi
}

composePID()
{
	local pid=$(cat ${RSPAMD_REDIS_PID} | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
	echo "${pid}"
}

composeScript()
{
	echo "/var/packages/MailPlus-Server/target/scripts/daemon/rspamd_redis.sh"
}

case "$1" in
	start)
		startOperation "mailserver_service_rspamd_redis"
		;;
	restart)
		restartOperation "mailserver_service_rspamd_redis"
		;;
	stop)
		stopOperation "mailserver_service_rspamd_redis"
		;;
	reload)
		reloadOperation "mailserver_service_rspamd_redis"
		;;
	status)
		status
		;;
esac
