#!/usr/bin/python

import os
import re
import pwd
import grp
import sys
import glob
import json
import fcntl
import subprocess
import shutil
from functools import wraps

sys.path.append("/var/packages/MailPlus-Server/target/backend_hook/")
from _Define import *
from _Utils import *

BASE_DIR       = '/var/packages/MailPlus-Server/target/etc/reportlearn'
HAM_CMD_DIR    = BASE_DIR + '/ham'
SPAM_CMD_DIR   = BASE_DIR + '/spam'
UID_MPS        = pwd.getpwnam('MailPlus-Server').pw_uid
GID_MPS        = grp.getgrnam('MailPlus-Server').gr_gid
GID_System     = grp.getgrnam('system').gr_gid
RSPAMC         = '/var/packages/MailPlus-Server/target/bin/rspamc'
LOCK_PATH      = '/var/lock/mailplus_server/report_learn_lock'
HISTORY_MASTER = BASE_DIR + '/history_master'
HISTORY_SLAVE  = BASE_DIR + '/history_slave'

SZK_REPORT_LEARN_HISTORY_FILE_PREIFX  = 'report_learn_history_file-'
SZK_REPORT_LEARN_CLEAR_HISTORY = 'report_learn_clear_history'

# modified from execWebAPI
def execDownloadWebAPI(api, method, version, outfile, **kwargs):
    """
    Send web API to localhost by /usr/syno/bin/synowebapi and return response (in Json) as Python dict.
    kwargs are parameters to web API. For example, following creates a share via web API.

    resp = execWebAPI("SYNO.Core.Share", "create", 1,
        name="public",
        shareinfo={
            "name": "public",
            "vol_path": "/volume1",
            "desc": "Public Shared Folder"
        }
    )
    """

    cmd = ["/usr/syno/bin/synowebapi", "--exec"]
    cmd.append("api=" + api)
    cmd.append("method=" + method)
    cmd.append("version=" + str(version))
    cmd.append("outfile=" + outfile)
    for key, value in kwargs.items():
        cmd.append(key + "=" + json.dumps(value))

    try:
        with open("/dev/null", "w") as null_f:
            raw_resp = subprocess.check_output(cmd, stderr=null_f)
        return json.loads(raw_resp)
    except ValueError as e:
        SYSLOG(LOG_ERR, "Failed to parse web API response (" + str(e) + ")")
    except Exception as e:
        SYSLOG(LOG_ERR, "Failed to executeCommand web API (" + str(e) + ")")
    return None

def removeExtraHeader(mailPath):
    mail = ''
    with open(mailPath, 'r') as f:
        mail = f.read()

    arr = mail.split("\n\n")

    # convert some headers to empty lines
    arr[0] = re.sub(r"^X-MailScanner.*?:.*(\n\t.*)*$", r"", arr[0], flags = re.MULTILINE)
    arr[0] = re.sub(r"^X-Spam-Status:.*(\n\t.*)*$", r"", arr[0], flags = re.MULTILINE)

    # remove empty line
    arr[0] = re.sub(r"[\n]{2,}", r"\n", arr[0]).rstrip()

    mail = "\n\n".join(arr)
    with open(mailPath, 'w') as f:
        f.write(mail)

def fetchAllMailByCmdFiles(webapi, cmdFiles, mailDir):
    for cmdFile in cmdFiles:
        print "process '%s'" % cmdFile
        with open(cmdFile, 'r') as f:
            for mailUid in f.read().splitlines():
                mailPath = mailDir + '/' + mailUid
                resp = execDownloadWebAPI(webapi, 'get_original', 1, mailPath, mail_uid=int(mailUid))
                if resp['success'] == False:
                    SYSLOG(LOG_ERR, "Failed to get mail (%s, %s)" % (webapi, mailUid))
                    continue
                removeExtraHeader(mailPath)
                os.chmod(mailPath, 0660)
                os.chown(mailPath, UID_MPS, GID_MPS)

def appendCmdFilesToHistory(cmdFiles, historyFile):
    with open(historyFile, "a") as f:
        for cmdFile in cmdFiles:
            f.write(cmdFile + "\n")
    os.chmod(historyFile, 0644)
    os.chown(historyFile, UID_MPS, GID_System)

def getHistoryFile():
    cAPI = BackendAPI()
    historyFile = ''
    if cAPI.isLoadBalancer():
        historyFile = HISTORY_MASTER
    else:
        historyFile = HISTORY_SLAVE
    return historyFile

def prepareDir(dirPath):
    if os.path.isdir(dirPath):
        shutil.rmtree(dirPath)
    elif os.path.exists(dirPath):
        os.unlink(dirPath)
    os.mkdir(dirPath)

def learn(mailType, mailDirPath):
    cmd = "%s -c bayes learn_%s %s" % (RSPAMC, mailType, mailDirPath)
    cmd = "su postfix -s /bin/sh -c '%s'" % cmd
    executeCommand(cmd, False)

def acquireLock():
    f = open(LOCK_PATH, 'w')
    fcntl.flock(f, fcntl.LOCK_EX)
    return f

def releaseLock(f):
    f.close()

def isAA():
    cAPI = BackendAPI()
    return cAPI.getPeersNum() == 2

def LockFunc(func):
    @wraps(func)
    def funcWrap(*args, **kwargs):
        lockF = None
        try:
            lockF = acquireLock()
            func(*args, **kwargs)
        except Exception as e:
            SYSLOG(LOG_ERR, "Exception: %s" % e)
        finally:
            if lockF:
                releaseLock(lockF)
    return funcWrap

@LockFunc
def learnMailHandler():
    hamMailDir   = BASE_DIR + '/ham_mail'
    hamCmdFiles  = glob.glob(HAM_CMD_DIR + '/*.txt')
    spamMailDir  = BASE_DIR + '/spam_mail'
    spamCmdFiles = glob.glob(SPAM_CMD_DIR + '/*.txt')
    historyFile  = getHistoryFile()

    prepareDir(hamMailDir)
    prepareDir(spamMailDir)

    # read cmd history and get unprocessed cmd files
    processedCmdFiles = []
    if os.path.isfile(historyFile):
        with open(historyFile, 'r') as f:
            processedCmdFiles = f.read().splitlines()
    unProcessedHamCmdFiles = list(set(hamCmdFiles) - set(processedCmdFiles))
    unProcessedSpamCmdFiles = list(set(spamCmdFiles) - set(processedCmdFiles))

    fetchAllMailByCmdFiles('SYNO.MailPlusServer.Spam.ReportedHam', unProcessedHamCmdFiles, hamMailDir)
    fetchAllMailByCmdFiles('SYNO.MailPlusServer.Spam.ReportedSpam', unProcessedSpamCmdFiles, spamMailDir)

    if len(unProcessedHamCmdFiles) > 0:
        learn('ham', hamMailDir)
    if len(unProcessedSpamCmdFiles) > 0:
        learn('spam', spamMailDir)

    appendCmdFilesToHistory(unProcessedHamCmdFiles + unProcessedSpamCmdFiles, historyFile)

    cAPI = BackendAPI()
    cAPI.setFile(SZK_REPORT_LEARN_HISTORY_FILE_PREIFX + os.path.basename(historyFile), historyFile)

    shutil.rmtree(hamMailDir)
    shutil.rmtree(spamMailDir)

@LockFunc
def deleteMailHandler():
    masterHistoryCmdFiles = []
    slaveHistoryCmdFiles  = []
    deleteCmdFiles = []

    if os.path.isfile(HISTORY_MASTER):
        with open(HISTORY_MASTER, 'r') as f:
            masterHistoryCmdFiles = f.read().splitlines()

    if isAA():
        if os.path.isfile(HISTORY_SLAVE):
            with open(HISTORY_SLAVE, 'r') as f:
                slaveHistoryCmdFiles = f.read().splitlines()
        deleteCmdFiles = list(set(masterHistoryCmdFiles) & set(slaveHistoryCmdFiles))
    else:
        deleteCmdFiles = masterHistoryCmdFiles

    deleteCmdFiles.sort()

    for cmdFile in deleteCmdFiles:
        mailUidList = []
        if os.path.isfile(cmdFile):
            with open(cmdFile, 'r') as f:
                mailUidList = f.read().splitlines()
        mailUidList = [ int(x) for x in mailUidList ]

        webapi = ''
        if cmdFile.startswith(HAM_CMD_DIR):
            webapi = 'SYNO.MailPlusServer.Spam.ReportedHam'
        else:
            webapi = 'SYNO.MailPlusServer.Spam.ReportedSpam'

        resp = execWebAPI(webapi, 'delete', 1, mail_uid_list=mailUidList)
        if resp['success'] == False:
            SYSLOG(LOG_ERR, "Failed to delete mail (%s)" % cmdFile)
            continue

    hamLastCmdFile = ''
    spamLastCmdFile = ''
    for cmdFile in deleteCmdFiles:
        if cmdFile.startswith(HAM_CMD_DIR):
            if cmdFile > hamLastCmdFile:
                hamLastCmdFile = cmdFile
        else:
            if cmdFile > spamLastCmdFile:
                spamLastCmdFile = cmdFile
    # set key to trigger deleting cmd files
    cAPI = BackendAPI()
    cAPI.mailconfSet(SZK_REPORT_LEARN_CLEAR_HISTORY, "%s,%s" % (hamLastCmdFile, spamLastCmdFile))

@LockFunc
def clearHistoryHandler():
    cAPI = BackendAPI()
    value = cAPI.mailconfGet(SZK_REPORT_LEARN_CLEAR_HISTORY)
    (hamLastCmdFile, spamLastCmdFile) = value.split(',')

    for cmdFile in glob.glob(HAM_CMD_DIR + '/*.txt'):
        if cmdFile <= hamLastCmdFile:
            os.unlink(cmdFile)

    for cmdFile in glob.glob(SPAM_CMD_DIR + '/*.txt'):
        if cmdFile <= spamLastCmdFile:
            os.unlink(cmdFile)

    historyFile = getHistoryFile()
    unProcessedCmdFiles = []
    with open(historyFile, 'r') as f:
        for line in f.readlines():
            cmdFile = line.strip()
            if cmdFile.startswith(HAM_CMD_DIR):
                if cmdFile > hamLastCmdFile:
                    unProcessedCmdFiles.append(cmdFile)
            else:
                if cmdFile > spamLastCmdFile:
                    unProcessedCmdFiles.append(cmdFile)
    with open(historyFile, 'w') as f:
        for cmdFile in unProcessedCmdFiles:
            f.write("%s\n" % cmdFile)

    cAPI.setFile(SZK_REPORT_LEARN_HISTORY_FILE_PREIFX + os.path.basename(historyFile), historyFile)

def webapiLearnAllHandler():
    resp = execWebAPI('SYNO.MailPlusServer.Spam.ReportedHam', 'learn_all', 1)
    if resp['success'] == False:
        SYSLOG(LOG_ERR, "Failed to learn all ham")
    resp = execWebAPI('SYNO.MailPlusServer.Spam.ReportedSpam', 'learn_all', 1)
    if resp['success'] == False:
        SYSLOG(LOG_ERR, "Failed to learn all spam")

def usage():
    print "%s: (learn_mail|delete_mail|clear_history|webapi_learn_all)" % sys.argv[0]

if __name__ == "__main__":
    if len(sys.argv) != 2:
        usage()
        sys.exit()

    if sys.argv[1] == 'learn_mail':
        learnMailHandler()
    elif sys.argv[1] == 'delete_mail':
        cAPI = BackendAPI()
        if cAPI.isLoadBalancer():
            deleteMailHandler()
        else:
            print "only load balancer can delete mails"
    elif sys.argv[1] == 'clear_history':
        clearHistoryHandler()
    elif sys.argv[1] == 'webapi_learn_all':
        webapiLearnAllHandler()
    else:
        usage()
        sys.exit()
