#!/usr/bin/perl

use strict;
use warnings;

use lib '/var/packages/MailPlus-Server/target/share/perl5/vendor_perl';
use lib '/var/packages/MailPlus-Server/target/lib/perl5/vendor_perl';
use lib '/var/packages/MailPlus-Server/target/lib/MIMEDefang';

use Encode;
use File::Temp qw(tempfile tempdir);
use JSON;
use MIME::Parser;
use POSIX qw(floor);
use Sys::Syslog;
use Sys::Syslog qw(:standard :macros);

use MailPlusServer::Util;

my $SignalGot          = 0;
my $MailType           = '';
my $TempDirTemplate    = '/var/packages/MailPlus-Server/target/var/mime_parser_XXXXXX';

my $MCPUpgradingFlag   = '/var/packages/MailPlus-Server/etc/upgrade_mcp_quarantine';
my $MCPPidFile         = '/var/run/mailplus_server/mcp_quarantine_upgrade.pid';

my $VirusUpgradingFlag = '/var/packages/MailPlus-Server/etc/upgrade_virus_quarantine';
my $VirusPidFile       = '/var/run/mailplus_server/virus_quarantine_upgrade.pid';

package VirusQuarantine {
	sub new {
		my $class = shift;
		my $self = {};

		$self->{api} = 'SYNO.MailPlusServer.Security.VirusQuarantine';
		$self->{db}  = '/var/packages/MailPlus-Server/etc/virus_quarantine.db';

		bless($self, $class);
	}

	sub get_total {
		my ($self) = @_;
		my $resp = main::exec_webapi($self->{api}, 'list', 1, {'offset' => 0, 'limit' => 1});

		if (defined($resp)) {
			return $resp->{data}->{total};
		}
		return undef;
	}

	sub list {
		my ($self, $offset, $limit) = @_;
		my $resp = main::exec_webapi($self->{api}, 'list', 1, {
			'sort_by'        => 'time',
			'sort_direction' => 'DESC',
			'offset'         => $offset,
			'limit'          => $limit
		});

		if (defined($resp)) {
			return $resp->{data}->{mail_list};
		}
		return undef;
	}

	sub dump {
		my ($self, $mail, $filename) = @_;
		my $resp = main::exec_webapi($self->{api}, 'get_original_mail', 1, {
			'message_id' => "'\"$mail->{message_id}\"'",
			'date_num'   => "'\"$mail->{date_num}\"'"
		}, $filename);

		unlink $filename if not defined($resp);

		return defined($resp);
	}

	sub delete {
		my ($self, $mail) = @_;
		my $resp = main::exec_webapi($self->{api}, 'delete_mail', 1, {
			'mail_list' => "'[{\"message_id\":\"$mail->{message_id}\",\"date_num\":\"$mail->{date_num}\"}]'"
		});

		return defined($resp);
	}

	sub get_id {
		my ($self, $mail) = @_;
		return "message_id = '$mail->{message_id}', date_num = '$mail->{date_num}'";
	}

	sub add_extra_headers {
		my ($self, $mail, $entity) = @_;

		$entity->head->replace($MailPlusServer::Util::HeaderSender, main::convert_addresses($mail->{sender}));
		$entity->head->replace($MailPlusServer::Util::HeaderRecipients, main::convert_addresses($mail->{recipient}));
		$entity->head->replace($MailPlusServer::Util::HeaderVirusReport, $mail->{virus});
	}

	sub get_db {
		my ($self) = @_;
		return $self->{db};
	}
}

package MCPQuarantine {
	sub new {
		my $class = shift;
		my $self = {};

		$self->{api} = 'SYNO.MailPlusServer.Security.MCP';
		$self->{db}  = '/var/packages/MailPlus-Server/etc/mcp_quarantine.db';

		bless($self, $class);
	}

	sub get_total {
		my ($self) = @_;
		my $resp = main::exec_webapi($self->{api}, 'list_quarantine_mail', 1, {'offset' => 0, 'limit' => 1});

		if (defined($resp)) {
			return $resp->{data}->{total_num};
		}
		return undef;
	}

	sub list {
		my ($self, $offset, $limit) = @_;
		my $resp = main::exec_webapi($self->{api}, 'list_quarantine_mail', 1, {
			'sort_by'        => 'receivedTime',
			'sort_direction' => 'DESC',
			'offset'         => $offset,
			'limit'          => $limit
		});

		if (defined($resp)) {
			return $resp->{data}->{mcp_quarantine_list};
		}
		return undef;
	}

	sub dump {
		my ($self, $mail, $filename) = @_;
		my $resp = main::exec_webapi($self->{api}, 'get_original_mail', 1, {
			'message_id' => "'\"$mail->{message_id}\"'",
			'dateNum'   => "'\"$mail->{dateNum}\"'"
		}, $filename);

		unlink $filename if not defined($resp);

		return defined($resp);
	}

	sub delete {
		my ($self, $mail) = @_;
		my $resp = main::exec_webapi($self->{api}, 'delete_quarantine_mail', 1, {
			'delete_list' => "'[{\"message_id\":\"$mail->{message_id}\",\"dateNum\":\"$mail->{dateNum}\"}]'"
		});

		return defined($resp);
	}

	sub get_id {
		my ($self, $mail) = @_;
		return "message_id = '$mail->{message_id}', dateNum = '$mail->{dateNum}'";
	}

	sub add_extra_headers {
		my ($self, $mail, $entity) = @_;

		$entity->head->replace($MailPlusServer::Util::HeaderSender, main::convert_addresses($mail->{sender}));
		$entity->head->replace($MailPlusServer::Util::HeaderRecipients, main::convert_addresses($mail->{receivers}));

		if ($mail->{report} =~ /score=(\d*), required[ =](\d*), (.*)\)/) {
			$entity->head->replace($MailPlusServer::Util::HeaderMCPScore, $1);
			$entity->head->replace($MailPlusServer::Util::HeaderMCPRequired, $2);
			$entity->head->replace($MailPlusServer::Util::HeaderMCPReport, Encode::encode('MIME-Header', $3));
		}
	}

	sub get_db {
		my ($self) = @_;
		return $self->{db};
	}
}

# the following is main package

sub exec_webapi {
	my ($api, $method, $version, $args_href, $outfile) = @_;
	my $cmd = '';
	my $output = '';
	my $resp = {};
	my @arr = ('/usr/syno/bin/synowebapi', '--exec');

	if (!defined($api) || !defined($method) || !defined($version)) {
		errlog('invalid argument for exec_webapi()');
		return undef;
	}

	push @arr, "api=$api";
	push @arr, "method=$method";
	push @arr, "version=$version";
	push @arr, "outfile=$outfile" if defined($outfile);
	if (defined $args_href) {
		foreach my $key (keys %{$args_href}) {
			push @arr, "$key=" . $args_href->{$key};
		}
	}

	$cmd = join(' ', @arr);
	$output = `$cmd 2>/dev/null`;
	if ($? != 0) {
		errlog("Error on the command: [%s]", $cmd);
		return undef;
	}

	$resp = JSON::decode_json($output);

	if (!$resp->{success}) {
		errlog("Failure on the webapi: [%s]", $cmd);
		return undef;
	}

	return $resp;
}

sub errlog {
	my $format = shift;
	syslog('err', "upgrade_quarantine.pl: ($MailType) $format", @_);
}

sub sig_handler {
	$SignalGot = 1;
}

sub gen_pid_file {
	my $path = shift;

	open FH, ">$path";
	print FH $$;
	close FH;
}

sub convert_addresses {
	my $str = shift;
	my @arr = split(',', $str);
	foreach my $addr (@arr) {
		$addr = '<' . $addr . '>';
	}
	return join(',', @arr);
}

sub upgrade_quarantine {
	my ($type) = @_;
	my ($account, $template, $total, $offset, $mail_list, $q_handler);
	my $limit = 50;
	my $need_redo = 0;

	if ($type eq 'virus') {
		$account   = $MailPlusServer::Util::VirusAccount;
		$template  = '/var/spool/mail/@quarantine/virus_XXXXXX';
		$q_handler = VirusQuarantine->new();
	} elsif ($type eq 'mcp') {
		$account   = $MailPlusServer::Util::MCPAccount;
		$template  = '/var/spool/mail/@quarantine/mcp_XXXXXX';
		$q_handler = MCPQuarantine->new();
	}

	if (!defined($total = $q_handler->get_total())) {
		errlog('cannot get total number');
		return -1;
	}
	return 0 if $total == 0;

	my $parser = new MIME::Parser;
	my $tempdir = tempdir($TempDirTemplate, CLEANUP => 1);
	$parser->output_dir($tempdir);

	$offset = floor(($total - 1) / $limit) * $limit;
	for (; $offset >= 0; $offset -= $limit) {
		if ($SignalGot) {
			errlog("got signal, interrupted");
			return -1;
		}

		errlog("progress (%d/%d)", $total - $offset, $total);

		if (!defined($mail_list = $q_handler->list($offset, $limit))) {
			errlog("failed to list mail (offset = $offset, limit = $limit)");
			return -1;
		}

		foreach my $mail (reverse @{$mail_list}) {
			last if $SignalGot;

			my ($entity, $raw_mail);
			my (undef, $filename) = tempfile($template, OPEN => 0);
			my $done = 0;

			if (!$q_handler->dump($mail, $filename)) {
				errlog("failed to dump original mail (%s)", $q_handler->get_id($mail));
				next;
			}

			eval { $entity = $parser->parse_open($filename) };
			if ($@) {
				errlog("failed to parse file %s (%s)", $filename, $@);
				unlink $filename;
				next;
			}
			unlink $filename;

			$q_handler->add_extra_headers($mail, $entity);

			$raw_mail = MailPlusServer::Util::entity_print($entity);
			for (my $i = 0; $i < 10; $i++) {
				$done = MailPlusServer::Util::lda_send_mail('', $account, $raw_mail);
				last if ($done || $SignalGot);

				# sometimes lda would be failed because dovecot is restarting, wait until dovecot is ready
				sleep 5;
			}
			if (!$done) {
				errlog("abort to call lda for the mail (%s)", $q_handler->get_id($mail));
				$need_redo = 1;
				next;
			}

			if (!$q_handler->delete($mail)) {
				errlog("failed to delete mail (%s)", $q_handler->get_id($mail));
				next;
			}
		}
	}

	if ($need_redo) {
		errlog("need to do upgrade again later");
		return -1;
	}

	unlink $q_handler->get_db();
	return 0;
}

sub main {
	my ($type) = @ARGV;

	if (!defined($type) || ($type ne 'virus' && $type ne 'mcp')) {
		print "Usage: upgrade_quarantine.pl (virus|mcp)\n";
		exit(1);
	}

	$MailType = $type;

	$SIG{TERM} = \&sig_handler;
	$SIG{INT}  = \&sig_handler;

	my $pidfile = '';
	my $flag = '';
	if ($type eq 'virus') {
		$pidfile = $VirusPidFile;
		$flag = $VirusUpgradingFlag;
	} elsif ($type eq 'mcp') {
		$pidfile = $MCPPidFile;
		$flag = $MCPUpgradingFlag;
	}
	gen_pid_file($pidfile);

	openlog('MailPlus-Server', 'pid,ndelay', 'mail');

	my $ret = 0;
	eval { $ret = upgrade_quarantine($type) };
	if ($@) {
		errlog("got exception: $@");
	} elsif (0 > $ret) {
		errlog("something wrong when upgrading quarantine");
	} else {
		errlog("finished upgrading quarantine");
		unlink $flag;
	}

	closelog();

	unlink $pidfile;
}

main();
