#!/bin/sh

. /var/packages/MailPlus-Server/target/scripts/daemon/util.sh

DKIM_SIGN_PID="/var/run/mailplus_server/opendkim_sign.pid"
DKIM_SIGN_CONF="${MAIL_SERVER_PKG}/target/etc/opendkim_sign.conf"
DKIM_SIGN_BIN="${MAIL_SERVER_PKG}/target/sbin/opendkim -x ${DKIM_SIGN_CONF}"
DKIM_SIGN_LOCK="/var/lock/mailplus_server/opendkim_sign.lock"
DKIM_SIGN_SOCK="/var/spool/@MailPlus-Server/opendkim_sign.sock"

checkStarted()
{
	if [ -f ${DKIM_SIGN_PID} ]; then
		PID=`cat ${DKIM_SIGN_PID}`
		PROCESS=`ps --pid ${PID} -o pid,args | grep opendkim | grep sign`
	fi

	if [ ! -z "${PROCESS}" ]; then
		return 1
	fi
	return 0
}

bin_status()
{
	if [ -f "${DKIM_SIGN_PID}" ]; then
		PID=`cat ${DKIM_SIGN_PID}`
		PROCESS=`cat "/proc/${PID}/cmdline" | grep "opendkim" | grep "sign"`

		if [ ! -z "${PROCESS}" ]; then
			return ${SERVICE_RUNNING}
		fi
		return ${SERVICE_DEAD_FPID}
	else
		if [ 0 != `ps -C opendkim -o pid,args | grep "${DKIM_SIGN_CONF}" | wc -l` ] ; then
			return ${SERVICE_RUNNING}
		fi
		return ${SERVICE_NOT_RUNNING}
	fi > /dev/null
}

conf_status()
{
	checkConfKey "enable_dkim_sign"
	return $?
}

status()
{
	conf_status
	local opendkim_enable=$?
	bin_status
	local opendkim_status=$?

	if [ "${SERVICE_NOT_RUNNING}" -eq ${opendkim_status} ] && [ "${RUNKEY_DISABLE}" -ne "${opendkim_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	elif [ "${SERVICE_RUNNING}" -eq ${opendkim_status} ] && [ "${RUNKEY_ENABLE}" -ne "${opendkim_enable}" ]; then
		return "${SERVICE_UNKNOWN}"
	fi
	return ${opendkim_status}
}

runDaemon()
{
	rm -f "${DKIM_SIGN_SOCK}"
	su MailPlus-Server -s /bin/sh -c "${DKIM_SIGN_BIN}"

	local daemon_status=""
	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		status
		daemon_status=$?
		debug_log "daemon_status opendkim (sign): ${daemon_status}"
		if [ ${daemon_status} -eq "${SERVICE_RUNNING}" ]; then
			#binary is ready
			return 0
		elif [ ${daemon_status} -eq "${SERVICE_DEAD_FPID}" ]; then
			#[FIXME] Maybe we can do something to recover
			return 1
		else
			sleep "${SLEEP_TIME}"
		fi
	done
	return 1
}

start()
{
	(
	if flock -x 8; then
		checkStarted
		if [ 1 -eq $? ] ; then
			echo "opendkim (sign) is already running"
			stopDaemon
			sleep 1
			runDaemon
		else
			runDaemon
		fi

		flock -u 8
		rm ${DKIM_SIGN_LOCK}
	fi
	)8> ${DKIM_SIGN_LOCK}
}

checkStopped()
{
	local stopped="no"
	for i in `seq 0 1 "${MAX_TRY_TIMES}"`
	do
		if [ 0 == `ps -C opendkim -o pid,args | grep "${DKIM_SIGN_CONF}" | wc -l` ] ; then
			# opendkim (sign) is terminated
			stopped="yes"
			break
		fi
		sleep "${SLEEP_TIME}"
	done

	if [ "yes" != "${stopped}" ]; then
		echo "Force kill opendkim (sign)"
		ps -C opendkim -o pid,args | grep "${DKIM_SIGN_CONF}" | awk '{kill -9 $1}'
	fi
}

stopDaemon()
{
	kill `cat ${DKIM_SIGN_PID}` &> /dev/null
	checkStopped
}

stop()
{
	(
	if flock -x 8; then
		stopDaemon

		flock -u 8
		rm ${DKIM_SIGN_LOCK}
	fi
	)8> ${DKIM_SIGN_LOCK}
}

restart()
{
	(
	if flock -x 8; then
		stopDaemon
		sleep 1
		runDaemon

		flock -u 8
		rm ${DKIM_SIGN_LOCK}
	fi
	)8> ${DKIM_SIGN_LOCK}
}

reload()
{
	ps -C opendkim -o pid,args | grep "${DKIM_SIGN_CONF}" | awk '{kill -SIGUSR1 $1}'
}

composePID()
{
	local pid=$(cat ${DKIM_SIGN_PID} | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')
	echo "${pid}"
}

composeScript()
{
	echo "/var/packages/MailPlus-Server/target/scripts/daemon/opendkim_sign.sh"
}


case "$1" in
	start)
		startOperation "mailserver_service_opendkim_sign"
		;;
	restart)
		restartOperation "mailserver_service_opendkim_sign"
		;;
	stop)
		stopOperation "mailserver_service_opendkim_sign"
		;;
	reload)
		reloadOperation "mailserver_service_opendkim_sign"
		;;
	status)
		status
		;;
	*)
		echo "Usage: $0 {start|stop|restart|status}" >&2
		exit 1
		;;
esac
