#!/usr/bin/python
import sys
import os

sys.path.append("/var/packages/MailPlus-Server/target/backend_hook/")
from _Utils import *

# path define
MAILPLUS_SERVER_TARGET = "/var/packages/MailPlus-Server/target"

SDK_TOOL = MAILPLUS_SERVER_TARGET + "/bin/syno_personal_policy"
DOVECOT = MAILPLUS_SERVER_TARGET + "/scripts/daemon/dovecot.sh"
DOVECOT_POLICY_PATH = MAILPLUS_SERVER_TARGET + "/etc/dovecot/conf.d"

POSTFIX = MAILPLUS_SERVER_TARGET + "/scripts/daemon/postfix.sh"
POSTFIX_POLICY_PATH = MAILPLUS_SERVER_TARGET + "/etc"
POSTMAP = MAILPLUS_SERVER_TARGET + "/sbin/postmap"

# redis key
USER_POLICY_TREE        = "user_policy_setting"
# redis keys set user policy configuration
IMAP_DISABLE            = "imap_enable_list-false"
IMAP_LOCAL              = "imap_local_enable_list-true"
POP3_DISABLE            = "pop3_enable_list-false"
POP3_LOCAL              = "pop3_local_enable_list-true"

AUTO_FORWARD_FALSE      = "smtp_auto_forward_disable_list-false"
AUTO_FORWARD_TRUE       = "smtp_auto_forward_disable_list-true"
SMTP_LOCAL              = "smtp_local_enable_list-true"

ATTACHMENT_SIZE_PREFIX  = "attachment_size_list"
DAILY_QUOTA_PREFIX      = "daily_quota_list"
DAILY_FLOW_PREFIX       = "daily_flow_limit_list"
# redis key to reset postfix main.cf
DISABLE_AUTOFORWARD     = "user_policy_disable_auto_forward"
ENABLE_ATTACHMENT       = "cleanup_user_attachment_restrict"
ENABLE_FLOW             = "smtpd_user_send_flow_restrict"
ENABLE_LOCAL_ONLY       = "smtpd_send_local_only"
ENABLE_QUOTA            = "smtpd_user_send_quota_restrict"

MAIN_DOMAIN             = "smtp_main_domain"
ACCOUNT_TYPE            = "account_type"
ACCOUNT_DOMAIN          = "acc_domain_name"
ACCOUNT_WIN_SHORT       = "win_domain_short_name"

MULTI_DOMAIN                = "virtual_multiple_domain"
MULTI_DOMAIN_ADDITION       = "additional_domain-"
MULTI_DOMAIN_SEPARATOR      = ", "

def callBinaryService(service, action):
    cmd = "%s %s" % (service, action)
    result, _ = executeCommand(cmd, False)

class InfoUser(object):
    def __init__(self):
        self.redisCall = BackendAPI()

    def turnUidStringToFullName(self, uidString):
        from subprocess import Popen, STDOUT, PIPE
        p = Popen([SDK_TOOL, 'getFullNameList'], stdout=PIPE, stdin=PIPE)
        userFullName = p.communicate(input=uidString)[0]
        if p.returncode:
            SYSLOG(LOG_ERR, 'error with getting user full name list')
        if userFullName:
            return userFullName
        return ''

    def getAccountType(self):
        return self.redisCall.mailconfGet(ACCOUNT_TYPE)

    def getAccountDomain(self):
        return self.redisCall.mailconfGet(ACCOUNT_DOMAIN)

    def getAccountWinShort(self):
        return self.redisCall.mailconfGet(ACCOUNT_WIN_SHORT)

    def composeAdDomain(self, accType):
        if accType == 'ldap':
            return '@' + self.getAccountDomain()
        elif accType == 'win':
            # Caution about double back slash
            return self.getAccountWinShort() + '\\'
        elif accType == 'local':
            return ''
        else:
            SYSLOG(LOG_ERR, 'Unknown account type, please check redis account type key')
            exit(1)

    def replaceLocal(self, inputAccount, domainName, appendFunc):
        if not callable(appendFunc):
            return inputAccount + "@" + domainName
        return appendFunc(inputAccount, domainName)

    def replaceWinDomain(self, inputAccount, domainName, appendFunc):
        if not callable(appendFunc):
            return inputAccount + "@" + domainName
        winDomain = self.composeAdDomain('win')
        if inputAccount.startswith(winDomain):
            return appendFunc(inputAccount[len(winDomain):], domainName)
        return appendFunc(inputAccount, domainName)

    def replaceLdap(self, inputAccount, domainName, appendFunc):
        if not callable(appendFunc):
            return inputAccount + "@" + domainName
        ldapDomain = self.composeAdDomain('ldap')
        if inputAccount.endswith(ldapDomain):
            return appendFunc(inputAccount[:len(inputAccount) - len(ldapDomain)], domainName)
        return appendFunc(inputAccount, domainName)

class InfoDomain(object):
    def __init__(self):
        self.redisCall = BackendAPI()

    def getPrimaryDomain(self):
        return self.redisCall.mailconfGet(MAIN_DOMAIN)

    def getAdditionalDomain(self, domainId):
        redisKey = MULTI_DOMAIN_ADDITION + str(domainId)
        return filter(None, self.redisCall.mailconfGet(redisKey).split(", "))

    def getAllDomain(self):
        idList = []
        domainList = []
        idDomain = self.redisCall.mailconfGet(MULTI_DOMAIN)
        idDomainList = filter(None, idDomain.split(", "))
        for each in idDomainList:
            delimiterPos = each.find("/")
            idList.append(each[:delimiterPos])
            domainList.append(each[delimiterPos + 1:])
        for each in idList:
            domainList.extend(self.getAdditionalDomain(each))

        return filter(None, domainList)

    def appendRecipientDomain(self, localPart, domainName):
        # This is for local recipient domain when using multiple domain
        return localPart + "@" + domainName.decode('utf8').encode('idna')

class ControlUtil(object):
    def __init__(self):
        self.redisCall = BackendAPI()
        self.domainCall = InfoDomain()
        self.userCall = InfoUser()

    def _setFilePath(self, service, fileSubPath):
        if service == "postfix":
            return os.path.join(POSTFIX_POLICY_PATH, fileSubPath)
        elif service == "dovecot":
            return os.path.join(DOVECOT_POLICY_PATH, fileSubPath)
        else:
            SYSLOG(LOG_ERR, "Unknown service type, please add the relative rule")
            exit(1)

    def _countBits(self, mask):
        transform = int(mask)
        bits = 8
        count = 0
        while bits > 0:
            bits = bits - 1
            if transform & 1:
                count = count + 1
            transform = transform >> 1
        return count

    def getLocalSection(self):
        import re
        bindIF = self.redisCall.getHostIF()
        hostIP = self.redisCall.getHostIP()
        ipConfig, _ = executeCommand("synonet --show", True)
        hostMask = re.search(bindIF + r".*?^Mask: (\d+)\.(\d+)\.(\d+)\.(\d+)", ipConfig, re.DOTALL | re.MULTILINE)
        section = 0
        if hostMask:
            section = self._countBits(hostMask.group(1)) + self._countBits(hostMask.group(2)) + self._countBits(hostMask.group(3)) + self._countBits(hostMask.group(4))
        return hostIP + "/" + str(section)

    def getAllRedisKeyWithPrefix(self, tree, prefix):
        # Get a huge dict with all redis set
        allSetInTree = self.redisCall.getTree(tree)
        allKeysWithPrefixList = []

        if not allSetInTree or not prefix:
            return []
        for eachKey in allSetInTree.keys():
            if eachKey.startswith(prefix):
                allKeysWithPrefixList.append(eachKey)
        return filter(None, allKeysWithPrefixList)

    def getPrefixKeyInTree(self, tree, prefix):
        prefixKeyVal = {}
        prefixKeyList = self.getAllRedisKeyWithPrefix(tree, prefix)
        if not prefixKeyList:
            return {}
        for key in prefixKeyList:
            value = self.redisCall.mailconfGet(key)
            if not value:
                SYSLOG(LOG_ERR, 'Cannot get redis key: ' + key)
                continue
            fullNameList = self.userCall.turnUidStringToFullName(value)
            if not fullNameList:
                SYSLOG(LOG_ERR, 'Cannot get users full name')
                continue
            prefixKeyVal[key] = filter(None, fullNameList.split(","))
        return prefixKeyVal

    def turnUid2FullNameFromRedis(self, key):
        uidString = self.redisCall.mailconfGet(key)
        if not uidString:
            return []
        userList = self.userCall.turnUidStringToFullName(uidString)
        return filter(None, userList.split(","))

    def assignKeyListWithValueToDict(self, keyList, constValue):
        if keyList:
            return {dictionKey:constValue for dictionKey in keyList}
        else:
            return {}

    def appendPrimaryDomainToUserFullNameList(self, inputList):
        accType = self.userCall.getAccountType()
        retList = []

        if not accType or not inputList:
            return []

        if not accType == 'win' and not accType == 'ldap' and not accType == 'local':
            SYSLOG(LOG_ERR, 'Unknown account type: ' + accType + ', please check rediskey')
            return []

        appendDomain = self.domainCall.appendRecipientDomain
        replaceCallback = {
                'win':self.userCall.replaceWinDomain,
                'ldap':self.userCall.replaceLdap,
                'local':self.userCall.replaceLocal}

        domainList = self.domainCall.getAllDomain()
        if not domainList:
            SYSLOG(LOG_ERR, "Get domain Info fail")
            return retList
        # TODO: Since only one callback function (appendDomain), maybe we should remove it
        for fullName in inputList:
            for domainName in domainList:
                retList.append(replaceCallback[accType](fullName, domainName, appendDomain))

        return retList

    def extractSufixNum(self, prefixDict, magnification):
        import re
        retDict = {}

        if not prefixDict:
            return {}
        for policy, userList in prefixDict.items():
            result = re.search(r"-(-*\d+)", policy)
            if not result or not userList:
                continue
            limitNum = long(result.group(1)) * magnification
            retDict.update(self.assignKeyListWithValueToDict(userList, str(limitNum)))
        return retDict

    def handleDoubleEscape(self, fullNameList):
        import re
        fixList = []
        regex = re.compile("\\\\")
        if not fullNameList:
            return []
        for fullName in fullNameList:
            fixList.append(regex.sub("\\\\\\\\", fullName))
        return fixList

    def dumpListAsFile(self, path, listVal):
        import fcntl
        try:
            with open(path, 'w') as genFile:
                fcntl.flock(genFile, fcntl.LOCK_EX | fcntl.LOCK_NB)
                if not listVal:
                    return
                for value in listVal:
                    if value == "":
                        continue
                    genFile.write(str(value) + "\n")
        except Exception as e:
            SYSLOG(LOG_ERR, "gen user list restriction file error (" + str(e) + ")")

    def checkDovecotSwitch(self):
        imap_disable = self.redisCall.mailconfGet(USER_POLICY_TREE + "/" + IMAP_DISABLE)
        pop3_disable = self.redisCall.mailconfGet(USER_POLICY_TREE + "/" + POP3_DISABLE)
        imap_local = self.redisCall.mailconfGet(USER_POLICY_TREE + "/" + IMAP_LOCAL)
        pop3_local = self.redisCall.mailconfGet(USER_POLICY_TREE + "/" + POP3_LOCAL)

        if (imap_disable and imap_disable != "") or (pop3_disable and pop3_disable != ""):
            self.redisCall.mailconfSet("imap_pop3/enable_dovecot_auth_deny", "yes")
        else:
            self.redisCall.mailconfSet("imap_pop3/enable_dovecot_auth_deny", "no")

        if (imap_local and imap_local != "") or (pop3_local and pop3_local != ""):
            self.redisCall.mailconfSet("imap_pop3/enable_dovecot_auth_local", "yes")
        else:
            self.redisCall.mailconfSet("imap_pop3/enable_dovecot_auth_local", "no")

class PostfixPolicy(object):
    def __init__(self):
        self.infoUtil = ControlUtil()
        self.policyStatus = self.setDefaultStatus()
        self.policyChangeKey = self.setKeyChange()

    def regenAllFile(self):
        self.genAutoForwardConf()
        self.genSendLocalOnlyConf()
        self.genSenderQuotaConf()
        self.genFlowLimitConf()
        self.genAttachmentLimitConf()

    def setDefaultStatus(self):
        return {DISABLE_AUTOFORWARD: 'no',
                ENABLE_LOCAL_ONLY:'no',
                ENABLE_QUOTA:'no',
                ENABLE_FLOW:'no',
                ENABLE_ATTACHMENT:'no'}

    def setKeyChange(self):
        return {DISABLE_AUTOFORWARD:False,
                ENABLE_LOCAL_ONLY:False,
                ENABLE_QUOTA:False,
                ENABLE_FLOW:False,
                ENABLE_ATTACHMENT:False}

    def setPolicyStatusToRedis(self):
        for key, changed in self.policyChangeKey.items():
            if not changed:
                continue
            self.infoUtil.redisCall.mailconfSet(key, self.policyStatus[key])

    def setUserPolicy(self):
        if (isKeyChange(AUTO_FORWARD_FALSE) or isKeyChange(AUTO_FORWARD_TRUE) or isKeyChange(MULTI_DOMAIN) or isPrefixKeyChange(MULTI_DOMAIN_ADDITION)):
            self.genAutoForwardConf()
            self.policyChangeKey[DISABLE_AUTOFORWARD] = True

        if (isKeyChange(SMTP_LOCAL) or
            isKeyChange(MULTI_DOMAIN) or isPrefixKeyChange(MULTI_DOMAIN_ADDITION)):
            self.genSendLocalOnlyConf()
            self.policyChangeKey[ENABLE_LOCAL_ONLY] = True

        if (isPrefixKeyChange(DAILY_QUOTA_PREFIX)):
            self.genSenderQuotaConf()
            self.policyChangeKey[ENABLE_QUOTA] = True

        if (isPrefixKeyChange(DAILY_FLOW_PREFIX)):
            self.genFlowLimitConf()
            self.policyChangeKey[ENABLE_FLOW] = True

        if (isPrefixKeyChange(ATTACHMENT_SIZE_PREFIX)):
            self.genAttachmentLimitConf()
            self.policyChangeKey[ENABLE_ATTACHMENT] = True

    def genAutoForwardConf(self):
        filePath = self.infoUtil._setFilePath('postfix', 'user_forward')
        shortListFalse = self.infoUtil.turnUid2FullNameFromRedis(AUTO_FORWARD_FALSE)
        shortListTrue = self.infoUtil.turnUid2FullNameFromRedis(AUTO_FORWARD_TRUE)
        userKeyVal = {}
        if shortListFalse:
            shortListFalse = self.infoUtil.appendPrimaryDomainToUserFullNameList(shortListFalse)
            userKeyVal.update(self.infoUtil.assignKeyListWithValueToDict(shortListFalse, 'enable'))
            self.policyStatus[DISABLE_AUTOFORWARD] = 'yes'
        if shortListTrue:
            shortListTrue = self.infoUtil.appendPrimaryDomainToUserFullNameList(shortListTrue)
            userKeyVal.update(self.infoUtil.assignKeyListWithValueToDict(shortListTrue, 'disable'))
            self.policyStatus[DISABLE_AUTOFORWARD] = 'yes'
        # add no setting back
        dumpKeyValAsFile(filePath, userKeyVal)
        callBinaryService(POSTMAP, filePath)

    def genSendLocalOnlyConf(self):
        # gen user restriction file
        userFilePath = self.infoUtil._setFilePath('postfix', 'user_local_only')
        shortList = self.infoUtil.turnUid2FullNameFromRedis(SMTP_LOCAL)
        userKeyVal = {}
        if shortList:
            userKeyVal = self.infoUtil.assignKeyListWithValueToDict(shortList, 'local_only')
            self.policyStatus[ENABLE_LOCAL_ONLY] = 'yes'
        else:
            self.policyStatus[ENABLE_LOCAL_ONLY] = 'no'
        dumpKeyValAsFile(userFilePath, userKeyVal)
        callBinaryService(POSTMAP, userFilePath)

        # gen permission domain file
        domainFilePath = self.infoUtil._setFilePath('postfix', 'permit_domain')
        allDomainList = self.infoUtil.domainCall.getAllDomain()
        allDomain = self.infoUtil.assignKeyListWithValueToDict(allDomainList, 'OK')
        dumpKeyValAsFile(domainFilePath, allDomain)
        callBinaryService(POSTMAP, domainFilePath)

    def genSenderQuotaConf(self):
        filePath = self.infoUtil._setFilePath('postfix', 'sender_quota_map')
        shortDict = self.infoUtil.getPrefixKeyInTree(USER_POLICY_TREE, DAILY_QUOTA_PREFIX)
        extendDict = {}
        if shortDict:
            extendDict = self.infoUtil.extractSufixNum(shortDict, 1)
            self.policyStatus[ENABLE_QUOTA] = 'yes'
        else:
            self.policyStatus[ENABLE_QUOTA] = 'no'
        dumpKeyValAsFile(filePath, extendDict)
        callBinaryService(POSTMAP, filePath)

    def genFlowLimitConf(self):
        filePath = self.infoUtil._setFilePath('postfix', 'flow_limit')
        shortDict = self.infoUtil.getPrefixKeyInTree(USER_POLICY_TREE, DAILY_FLOW_PREFIX)
        extendDict = {}
        if shortDict:
            # MB turns into Byte
            extendDict = self.infoUtil.extractSufixNum(shortDict, 1048576)
            self.policyStatus[ENABLE_FLOW] = 'yes'
        else:
            self.policyStatus[ENABLE_FLOW] = 'no'
        dumpKeyValAsFile(filePath, extendDict)
        callBinaryService(POSTMAP, filePath)

    def genAttachmentLimitConf(self):
        filePath = self.infoUtil._setFilePath('postfix', 'attachment_limit')
        shortDict = self.infoUtil.getPrefixKeyInTree(USER_POLICY_TREE, ATTACHMENT_SIZE_PREFIX)
        extendDict = {}
        if shortDict:
            # MB turns into Byte and multiply increase of base64 encode -> *4/3
            extendDict = self.infoUtil.extractSufixNum(shortDict, 1398101)
            self.policyStatus[ENABLE_ATTACHMENT] = 'yes'
        else:
            self.policyStatus[ENABLE_ATTACHMENT] = 'no'
        dumpKeyValAsFile(filePath, extendDict)
        callBinaryService(POSTMAP, filePath)

class DovecotPolicy(object):
    def __init__(self):
        self.change = False
        self.infoUtil = ControlUtil()

    def regenAllFile(self):
        self.genLocalPassDbConf()
        self.genLimitedUserConf(IMAP_DISABLE, 'disable_imap')
        self.genLimitedUserConf(POP3_DISABLE, 'disable_pop3')
        self.genLimitedUserConf(IMAP_LOCAL, 'local_imap')
        self.genLimitedUserConf(POP3_LOCAL, 'local_pop3')

    def setUserPolicy(self):
        if (isKeyChange(IMAP_DISABLE) or isKeyChange(POP3_DISABLE)):
            self.genLimitedUserConf(IMAP_DISABLE, 'disable_imap')
            self.genLimitedUserConf(POP3_DISABLE, 'disable_pop3')
            self.change = True
        if (isKeyChange(IMAP_LOCAL) or isKeyChange(POP3_LOCAL)):
            self.genLimitedUserConf(IMAP_LOCAL, 'local_imap')
            self.genLimitedUserConf(POP3_LOCAL, 'local_pop3')
            self.genLocalPassDbConf()
            self.change = True
        self.infoUtil.checkDovecotSwitch()
        return self.change

    def genLocalPassDbConf(self):
        import fcntl
        filePath = self.infoUtil._setFilePath("dovecot", "auth-local.conf.ext")
        try:
            with open(filePath, 'w') as confFile:
                fcntl.flock(confFile, fcntl.LOCK_EX | fcntl.LOCK_NB)
                confFile.write("passdb {\n\
                        driver = passwd-file\n\
                        args = %s \n\
                        default_fields = nopassword\n\
                        result_success = continue-ok\n\
                        result_failure = continue-fail\n\
                }\n" % (DOVECOT_POLICY_PATH + "/local_%s"))

                confFile.write("passdb {\n\
                        driver = static\n\
                        default_fields = nopassword allow_nets=127.0.0.1,%s\n\
                        skip = unauthenticated\n\
                        result_success = continue-fail\n\
                        result_failure = return-fail\n\
                }" % (self.infoUtil.getLocalSection()))

        except Exception as e:
            SYSLOG(LOG_ERR, "gen dovecot auth-local fail (" + str(e) + ")")

    def genLimitedUserConf(self, key, fileName):
        filePath = self.infoUtil._setFilePath('dovecot', fileName)
        shortList = self.infoUtil.turnUid2FullNameFromRedis(key)
        shortList = self.infoUtil.handleDoubleEscape(shortList)
        self.infoUtil.dumpListAsFile(filePath, shortList)

def usage():
    print('If there is no parameter, it runs when redis key change')
    print('Or use [regenPostfixPolicy]')

if __name__ == '__main__':
    postfixPart = PostfixPolicy()
    dovecotPart = DovecotPolicy()
    if len(sys.argv) < 2:
        postfixPart.setUserPolicy()
        if (dovecotPart.setUserPolicy()):
            os.system("/var/packages/MailPlus-Server/target/scripts/daemon/dovecot.sh restart")

        postfixPart.setPolicyStatusToRedis()
    elif len(sys.argv) == 2 and sys.argv[1] == 'regenPostfixPolicy':
        postfixPart.regenAllFile()
    elif len(sys.argv) == 2 and sys.argv[1] == 'regenDovecotPolicy':
        dovecotPart.regenAllFile()
        os.system("/var/packages/MailPlus-Server/target/scripts/daemon/dovecot.sh restart")
    else:
        usage()
