#!/usr/bin/python
import sys
sys.path.append("/var/packages/MailPlus-Server/target/backend_hook/")
from _Utils import *

TREE                    = "virtual_domain"
VIRTUAL_DOMAIN          = "virtual_multiple_domain"
ADDITIONAL_DOMAIN       = "additional_domain-"
ENABLE_ATTACHMENT_LIMIT = "enable_attachment-"
ENABLE_AUTO_FORWARD     = "enable_auto_forward-"
ENABLE_DAILY_QUOTA      = "enable_daily_quota-"
ENABLE_OUTBOUND_FLOW    = "enable_outbound_flow-"
ATTACHMENT_LIMIT        = "attachment_limit-"
DAILY_QUOTA_LIMIT       = "daily_quota_limit-"
MAIL_SIZE_LIMIT         = "mail_size_limit-"
OUTBOUND_FLOW_LIMIT     = "outbound_flow_limit-"

# Key for memcached
DOMAIN_FLOW_RESTRICT    = "smtpd_domain_send_flow_restrict"
DOMAIN_QUOTA_RESTRICT   = "smtpd_domain_send_quota_restrict"

MAILPLUS_SERVER_TARGET  = "/var/packages/MailPlus-Server/target"
POSTMAP                 = MAILPLUS_SERVER_TARGET + "/sbin/postmap"
DIR_PATH                = MAILPLUS_SERVER_TARGET + "/etc/multidomain"
FILE_PATH               = DIR_PATH + "/domain_setting_maps"

class DomainUsageSetting(object):
    def __init__(self):
        self.redisCall = BackendAPI()
        self.defaultSettings = self.CreateDefaultValueDict()
        self.domainIdDict = {}
        self.allOutputSettingDict = {}

    def AppendAllDomain(self, domainId, entry, setValue):
        if not setValue:
            setValue = str(self.defaultSettings[entry])
        for domainName in self.domainIdDict[domainId]:
            self.allOutputSettingDict[entry + domainName.decode('utf8').encode('idna').lower()] = setValue

    def CheckRedisChange(self):
        if isKeyChange(VIRTUAL_DOMAIN) or isPrefixKeyChange(ADDITIONAL_DOMAIN):
            return True
        for prefixKey in self.CreateUsageLimitKeyList():
            if isPrefixKeyChange(prefixKey):
                return True
        return False

    def CreateUsageLimitKeyList(self):
        return [ENABLE_ATTACHMENT_LIMIT,
                ENABLE_AUTO_FORWARD,
                ENABLE_DAILY_QUOTA,
                ENABLE_OUTBOUND_FLOW,
                ATTACHMENT_LIMIT,
                DAILY_QUOTA_LIMIT,
                MAIL_SIZE_LIMIT,
                OUTBOUND_FLOW_LIMIT]

    def CreateDefaultValueDict(self):
        return {ENABLE_ATTACHMENT_LIMIT : 'no',
                ENABLE_AUTO_FORWARD     : 'yes',
                ENABLE_DAILY_QUOTA      : 'no',
                ENABLE_OUTBOUND_FLOW    : 'no',
                ATTACHMENT_LIMIT        : 10,
                DAILY_QUOTA_LIMIT       : 500,
                MAIL_SIZE_LIMIT         : 10,
                OUTBOUND_FLOW_LIMIT     : 2000}

    def GenAllDomainSettingDict(self):
        usageLimitList = self.CreateUsageLimitKeyList()
        for domainId in self.domainIdDict.keys():
            for redisKey in usageLimitList:
                redisValue = self.redisCall.mailconfGet(redisKey + domainId)
                trueValue = self.TurnMBToExactValue(redisKey, redisValue)
                self.AppendAllDomain(domainId, redisKey, trueValue)

    def GetMultipleDomainList(self):
        return filter(None, self.redisCall.mailconfGet(VIRTUAL_DOMAIN).split(", "))

    def LoadDomainIdDict(self):
        for idAndDomain in self.GetMultipleDomainList():
            [domainId, domainName] = idAndDomain.split('/')
            self.domainIdDict[domainId] = [domainName]

    def MakeSureDirPath(self):
        if os.path.exists(MAILPLUS_SERVER_TARGET) and not os.path.exists(DIR_PATH):
            os.makedirs(DIR_PATH)

    # This function is for memcached
    def RewriteDomainRedisStatus(self):
        dailyFlags = False
        outboundFlags = False
        for key, value in self.allOutputSettingDict.items():
            if dailyFlags and outboundFlags:
                break
            if not dailyFlags and key.startswith(ENABLE_DAILY_QUOTA) and value == 'yes':
                dailyFlags = True
            if not outboundFlags and key.startswith(ENABLE_OUTBOUND_FLOW) and value == 'yes':
                outboundFlags = True
        self.redisCall.mailconfSet(DOMAIN_QUOTA_RESTRICT, 'yes' if dailyFlags else 'no')
        self.redisCall.mailconfSet(DOMAIN_FLOW_RESTRICT, 'yes' if outboundFlags else 'no')

    def TurnMBToExactValue(self, redisKey, redisValue):
        if redisKey.startswith(ATTACHMENT_LIMIT):
            if not redisValue:
                redisValue = str(self.defaultSettings[ATTACHMENT_LIMIT])
        # Fix: Base64 encode is our client encoded method, but it might have other encoded method, so *4/3 is not a good choice.
            return str((int(redisValue) << 20) * 4 / 3) # MB turns byte and multiply base64 encode size -> *4/3
        if redisKey.startswith(OUTBOUND_FLOW_LIMIT) or redisKey.startswith(MAIL_SIZE_LIMIT):
            if not redisValue:
                redisValue = str(self.defaultSettings[OUTBOUND_FLOW_LIMIT])
            return str(int(redisValue) << 20) # MB turns byte
        if redisKey.startswith(OUTBOUND_FLOW_LIMIT) or redisKey.startswith(MAIL_SIZE_LIMIT):
            if not redisValue:
                redisValue = str(self.defaultSettings[MAIL_SIZE_LIMIT])
            return str(int(redisValue) << 20) # MB turns byte
        return redisValue

def GenDomainUsageLimitSettings():
    usageLimitSettings = DomainUsageSetting()

    if not usageLimitSettings.CheckRedisChange():
        return
    usageLimitSettings.MakeSureDirPath()
    usageLimitSettings.LoadDomainIdDict()
    usageLimitSettings.GenAllDomainSettingDict()
    usageLimitSettings.RewriteDomainRedisStatus()
    dumpKeyValAsFile(FILE_PATH, usageLimitSettings.allOutputSettingDict)
    executeCommand(POSTMAP + " " + FILE_PATH)

def SetInitStat():
    usageLimitSettings = DomainUsageSetting()
    usageLimitSettings.MakeSureDirPath()
    with open(FILE_PATH, "a+") as checkFile:
        pass
    executeCommand(POSTMAP + " " + FILE_PATH)

if __name__ == '__main__':
    try:
        if len(sys.argv) > 1 and sys.argv[1] == 'init':
            SetInitStat()
        else:
            GenDomainUsageLimitSettings()
    except Exception as e:
        SYSLOG(LOG_ERR, "Gen domain usage limit settings fail, error: %s" % (e))
